<?php
/**
 * Order Tracking Admin View
 */

if (!defined('ABSPATH')) {
    exit;
}

use Cartt\Services\OrderTrackingService;
use Cartt\Services\CurrencyService;

$carriers = OrderTrackingService::getCarriers();
$statuses = OrderTrackingService::getStatuses();

// Get order ID from query string
$orderId = isset($_GET['order_id']) ? intval($_GET['order_id']) : 0;
$tracking = $orderId ? OrderTrackingService::getOrderTracking($orderId) : [];

// Get order details
$order = null;
if ($orderId) {
    global $wpdb;
    $order = $wpdb->get_row($wpdb->prepare(
        "SELECT o.*, c.first_name, c.last_name, c.email
         FROM {$wpdb->prefix}cartt_orders o
         LEFT JOIN {$wpdb->prefix}cartt_customers c ON o.customer_id = c.id
         WHERE o.id = %d",
        $orderId
    ));
}
?>

<div class="cartt-tracking-content">
    <h2>
        Order Tracking
        <?php if ($order): ?>
        - Order #<?php echo esc_html($order->order_number); ?>
        <?php endif; ?>
    </h2>

    <?php if (!$orderId): ?>
    <!-- Intro -->
    <div class="cartt-zones-intro" style="background:#f6f7f7;padding:16px 20px;border-radius:8px;margin-bottom:20px;">
        <p style="margin:0;color:#50575e;">
            Add tracking numbers to orders so customers can track their shipments. 
            When you add tracking, customers receive an email with the tracking link.
            <span class="cartt-tooltip">
                <span class="cartt-tooltip-icon">?</span>
                <span class="cartt-tooltip-text">For automatic tracking updates from carriers (UPS, FedEx, USPS), enable the Live Rates addon which includes real-time tracking status.</span>
            </span>
        </p>
    </div>
    
    <!-- Order Search -->
    <div class="cartt-card">
        <h2>Find Order</h2>
        <form method="get" action="">
            <input type="hidden" name="page" value="cartt-tracking">
            <p>
                <label for="order_id"><strong>Order ID or Number:</strong></label><br>
                <input type="text" name="order_id" id="order_id" class="regular-text" placeholder="Enter order ID">
                <button type="submit" class="button button-primary">Find Order</button>
            </p>
        </form>
        
        <h3 style="margin-top: 30px;">Recent Orders Needing Tracking</h3>
        <?php
        global $wpdb;
        $pendingOrders = $wpdb->get_results(
            "SELECT o.id, o.order_number, o.created_at, o.total, c.first_name, c.last_name
             FROM {$wpdb->prefix}cartt_orders o
             LEFT JOIN {$wpdb->prefix}cartt_customers c ON o.customer_id = c.id
             LEFT JOIN {$wpdb->prefix}cartt_order_tracking t ON o.id = t.order_id
             WHERE o.status IN ('processing', 'shipped') AND t.id IS NULL
             ORDER BY o.created_at DESC
             LIMIT 20"
        );
        ?>
        <?php if (empty($pendingOrders)): ?>
        <p style="color: #666;">All orders have tracking information.</p>
        <?php else: ?>
        <table class="wp-list-table widefat fixed striped">
            <thead>
                <tr>
                    <th>Order</th>
                    <th>Customer</th>
                    <th>Date</th>
                    <th>Total</th>
                    <th>Action</th>
                </tr>
            </thead>
            <tbody>
                <?php foreach ($pendingOrders as $o): ?>
                <tr>
                    <td><strong>#<?php echo esc_html($o->order_number); ?></strong></td>
                    <td><?php echo esc_html($o->first_name . ' ' . $o->last_name); ?></td>
                    <td><?php echo date('M j, Y', strtotime($o->created_at)); ?></td>
                    <td><?php echo esc_html(CurrencyService::formatPrice($o->total)); ?></td>
                    <td>
                        <a href="<?php echo admin_url('admin.php?page=cartt-tracking&order_id=' . $o->id); ?>" class="button button-small">Add Tracking</a>
                    </td>
                </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
        <?php endif; ?>
    </div>

    <?php else: ?>
    <!-- Order Tracking Details -->
    <p><a href="<?php echo admin_url('admin.php?page=cartt-tracking'); ?>">&larr; Back to Tracking</a></p>

    <div style="display: grid; grid-template-columns: 2fr 1fr; gap: 20px;">
        <div>
            <!-- Add Tracking Form -->
            <div class="cartt-card">
                <h2>Add Tracking Information</h2>
                <form id="add-tracking-form">
                    <input type="hidden" name="order_id" value="<?php echo esc_attr($orderId); ?>">
                    <table class="form-table">
                        <tr>
                            <th><label for="carrier">Carrier *</label></th>
                            <td>
                                <select name="carrier" id="carrier" required style="width: 100%;">
                                    <option value="">Select carrier...</option>
                                    <?php foreach ($carriers as $key => $carrier): ?>
                                    <option value="<?php echo esc_attr($key); ?>"><?php echo esc_html($carrier['name']); ?></option>
                                    <?php endforeach; ?>
                                </select>
                            </td>
                        </tr>
                        <tr>
                            <th><label for="tracking_number">Tracking Number *</label></th>
                            <td>
                                <input type="text" name="tracking_number" id="tracking_number" class="regular-text" required>
                            </td>
                        </tr>
                        <tr>
                            <th><label for="tracking_url">Tracking URL</label></th>
                            <td>
                                <input type="url" name="tracking_url" id="tracking_url" class="large-text" placeholder="Auto-generated if left blank">
                            </td>
                        </tr>
                        <tr>
                            <th><label for="estimated_delivery">Est. Delivery</label></th>
                            <td>
                                <input type="date" name="estimated_delivery" id="estimated_delivery">
                            </td>
                        </tr>
                        <tr>
                            <th>Notification</th>
                            <td>
                                <label>
                                    <input type="checkbox" name="notify_customer" value="1" checked>
                                    Send tracking email to customer
                                </label>
                            </td>
                        </tr>
                    </table>
                    <p class="submit">
                        <button type="submit" class="button button-primary">Add Tracking</button>
                    </p>
                </form>
            </div>

            <!-- Existing Tracking -->
            <?php if (!empty($tracking)): ?>
            <div class="cartt-card" style="margin-top: 20px;">
                <h2>Tracking History</h2>
                <?php foreach ($tracking as $t): ?>
                <div class="tracking-item" style="padding: 15px; background: #f9f9f9; border-radius: 4px; margin-bottom: 10px;">
                    <div style="display: flex; justify-content: space-between; align-items: start;">
                        <div>
                            <strong><?php echo esc_html($t->carrier_name); ?></strong>
                            <span class="tracking-status tracking-status-<?php echo esc_attr($t->status); ?>" style="margin-left: 10px;">
                                <?php echo esc_html($t->status_label); ?>
                            </span>
                            <br>
                            <code style="font-size: 14px;"><?php echo esc_html($t->tracking_number); ?></code>
                            <?php if ($t->tracking_url): ?>
                            <a href="<?php echo esc_url($t->tracking_url); ?>" target="_blank" style="margin-left: 10px;">Track →</a>
                            <?php endif; ?>
                        </div>
                        <div style="text-align: right; font-size: 12px; color: #666;">
                            Added: <?php echo date('M j, Y g:i A', strtotime($t->created_at)); ?>
                            <?php if ($t->estimated_delivery): ?>
                            <br>Est: <?php echo date('M j, Y', strtotime($t->estimated_delivery)); ?>
                            <?php endif; ?>
                        </div>
                    </div>
                    <?php if ($t->status_detail): ?>
                    <div style="margin-top: 10px; font-size: 13px; color: #666;">
                        <?php echo esc_html($t->status_detail); ?>
                    </div>
                    <?php endif; ?>
                    <div style="margin-top: 10px;">
                        <select class="update-status-select" data-id="<?php echo $t->id; ?>" style="font-size: 12px;">
                            <?php foreach ($statuses as $key => $label): ?>
                            <option value="<?php echo esc_attr($key); ?>" <?php selected($t->status, $key); ?>><?php echo esc_html($label); ?></option>
                            <?php endforeach; ?>
                        </select>
                        <button type="button" class="button button-small delete-tracking-btn" data-id="<?php echo $t->id; ?>">Delete</button>
                    </div>
                </div>
                <?php endforeach; ?>
            </div>
            <?php endif; ?>
        </div>

        <!-- Order Summary Sidebar -->
        <div>
            <div class="cartt-card">
                <h2>Order Summary</h2>
                <table style="width: 100%;">
                    <tr>
                        <td><strong>Customer:</strong></td>
                        <td><?php echo esc_html($order->first_name . ' ' . $order->last_name); ?></td>
                    </tr>
                    <tr>
                        <td><strong>Email:</strong></td>
                        <td><?php echo esc_html($order->email); ?></td>
                    </tr>
                    <tr>
                        <td><strong>Status:</strong></td>
                        <td><?php echo esc_html(ucfirst($order->status)); ?></td>
                    </tr>
                    <tr>
                        <td><strong>Total:</strong></td>
                        <td><?php echo esc_html(CurrencyService::formatPrice($order->total)); ?></td>
                    </tr>
                    <tr>
                        <td><strong>Date:</strong></td>
                        <td><?php echo date('M j, Y', strtotime($order->created_at)); ?></td>
                    </tr>
                </table>
                
                <?php
                $shipping = json_decode($order->shipping_address, true);
                if ($shipping):
                ?>
                <h3 style="margin-top: 20px;">Shipping Address</h3>
                <p style="margin: 0;">
                    <?php echo esc_html($shipping['first_name'] . ' ' . $shipping['last_name']); ?><br>
                    <?php echo esc_html($shipping['address_1']); ?><br>
                    <?php if (!empty($shipping['address_2'])): ?><?php echo esc_html($shipping['address_2']); ?><br><?php endif; ?>
                    <?php echo esc_html($shipping['city'] . ', ' . $shipping['state'] . ' ' . $shipping['postcode']); ?><br>
                    <?php echo esc_html($shipping['country']); ?>
                </p>
                <?php endif; ?>
                
                <p style="margin-top: 20px;">
                    <a href="<?php echo admin_url('admin.php?page=cartt-orders&action=view&id=' . $orderId); ?>" class="button">View Full Order</a>
                </p>
            </div>
        </div>
    </div>
    <?php endif; ?>
</div>

<style>
.cartt-card { background: #fff; border: 1px solid #ccd0d4; border-radius: 4px; padding: 20px; }
.cartt-card h2 { margin-top: 0; padding-bottom: 10px; border-bottom: 1px solid #eee; }
.tracking-status { display: inline-block; padding: 2px 8px; border-radius: 3px; font-size: 11px; font-weight: 500; }
.tracking-status-pending { background: #fff3cd; color: #856404; }
.tracking-status-in_transit { background: #cce5ff; color: #004085; }
.tracking-status-out_for_delivery { background: #d4edda; color: #155724; }
.tracking-status-delivered { background: #28a745; color: #fff; }
.tracking-status-failed { background: #f8d7da; color: #721c24; }
</style>

<script>
jQuery(function($) {
    const nonce = '<?php echo wp_create_nonce('cartt_tracking'); ?>';

    // Add tracking
    $('#add-tracking-form').on('submit', function(e) {
        e.preventDefault();
        
        const data = {
            action: 'cartt_add_tracking',
            nonce: nonce,
            order_id: $('input[name="order_id"]').val(),
            carrier: $('#carrier').val(),
            tracking_number: $('#tracking_number').val(),
            tracking_url: $('#tracking_url').val(),
            estimated_delivery: $('#estimated_delivery').val(),
            notify_customer: $('input[name="notify_customer"]').is(':checked') ? 1 : 0
        };

        $.post(ajaxurl, data, function(response) {
            if (response.success) {
                location.reload();
            } else {
                alert('Error: ' + response.data.message);
            }
        });
    });

    // Update status
    $('.update-status-select').on('change', function() {
        const id = $(this).data('id');
        const status = $(this).val();

        $.post(ajaxurl, {
            action: 'cartt_update_tracking_status',
            nonce: nonce,
            id: id,
            status: status
        }, function(response) {
            if (response.success) {
                location.reload();
            } else {
                alert('Error: ' + response.data.message);
            }
        });
    });

    // Delete tracking
    $('.delete-tracking-btn').on('click', function() {
        if (!confirm('Delete this tracking information?')) return;
        
        const id = $(this).data('id');

        $.post(ajaxurl, {
            action: 'cartt_delete_tracking',
            nonce: nonce,
            id: id
        }, function(response) {
            if (response.success) {
                location.reload();
            } else {
                alert('Error: ' + response.data.message);
            }
        });
    });

    // Auto-generate tracking URL
    $('#carrier').on('change', function() {
        const carrier = $(this).val();
        if (carrier && !$('#tracking_url').val()) {
            // URL will be auto-generated on server side
        }
    });
});
</script>
