<?php
/**
 * Social Proof Admin View
 * 
 * @package Cartt
 * @since 1.4.0
 */

defined('ABSPATH') || exit;

$settings = get_option('cartt_social_proof_settings', [
    'enabled' => true,
    'purchase_notifications' => [
        'enabled' => true,
        'position' => 'bottom-left',
        'display_duration' => 5000,
        'delay_between' => 8000,
        'show_product_image' => true,
        'show_location' => true,
        'anonymize_names' => true,
        'max_age_hours' => 24,
        'message_template' => '{name} from {location} purchased {product}'
    ],
    'visitor_count' => [
        'enabled' => true,
        'show_on_product' => true,
        'show_on_category' => false,
        'time_window' => 15,
        'minimum_threshold' => 2,
        'message' => '{count} people are viewing this right now'
    ],
    'stock_urgency' => [
        'enabled' => true,
        'low_stock_threshold' => 10,
        'critical_stock_threshold' => 3,
        'show_exact_count' => false,
        'low_stock_message' => 'Only a few left in stock',
        'critical_stock_message' => 'Almost gone! Only {count} left'
    ],
    'cart_activity' => [
        'enabled' => true,
        'time_window' => 60,
        'minimum_threshold' => 1,
        'message' => '{count} people have this in their cart'
    ],
    'recent_sales' => [
        'enabled' => true,
        'time_window' => 24,
        'show_on_product' => true,
        'message' => '{count} sold in the last 24 hours'
    ],
    'fomo_countdown' => [
        'enabled' => false,
        'type' => 'sale_end',
        'message' => 'Sale ends in {countdown}'
    ]
]);

// Get stats (with fallbacks for missing tables)
global $wpdb;
$sessions_table = $wpdb->prefix . 'cartt_visitor_sessions';
$cart_table = $wpdb->prefix . 'cartt_cart_activity';
$orders_table = $wpdb->prefix . 'cartt_orders';

$active_sessions = 0;
$active_carts = 0;
$today_sales = 0;
$hour_sales = 0;

$wpdb->suppress_errors(true);
$active_sessions = (int) $wpdb->get_var($wpdb->prepare(
    "SELECT COUNT(*) FROM {$sessions_table} WHERE last_activity > %s",
    gmdate('Y-m-d H:i:s', strtotime('-5 minutes'))
));
$active_carts = (int) $wpdb->get_var($wpdb->prepare(
    "SELECT COUNT(*) FROM {$cart_table} WHERE created_at > %s",
    gmdate('Y-m-d H:i:s', strtotime('-60 minutes'))
));
$today_sales = (int) $wpdb->get_var($wpdb->prepare(
    "SELECT COUNT(*) FROM {$orders_table} WHERE DATE(created_at) = %s AND status IN ('processing', 'completed')",
    gmdate('Y-m-d')
));
$hour_sales = (int) $wpdb->get_var($wpdb->prepare(
    "SELECT COUNT(*) FROM {$orders_table} WHERE created_at > %s AND status IN ('processing', 'completed')",
    gmdate('Y-m-d H:i:s', strtotime('-1 hour'))
));
$wpdb->suppress_errors(false);

$current_tab = isset($_GET['subtab']) ? sanitize_key($_GET['subtab']) : 'overview';
$isEmbedded = isset($GLOBALS['cartt_embedded']) && $GLOBALS['cartt_embedded'];
?>

<?php if (!$isEmbedded): ?>
<div class="wrap cartt-admin">
    <h1 class="wp-heading-inline">Social Proof</h1>
    <hr class="wp-header-end">
<?php endif; ?>

    <div class="cartt-stats-grid" style="display: grid; grid-template-columns: repeat(4, 1fr); gap: 15px; margin-bottom: 20px;">
        <div class="cartt-stat-card" style="background: #fff; padding: 20px; border-radius: 8px; box-shadow: 0 1px 3px rgba(0,0,0,0.1);">
            <div style="color: #666; font-size: 13px; margin-bottom: 5px;">Active Visitors</div>
            <div style="font-size: 28px; font-weight: 600; color: #00a32a;"><?php echo $active_sessions; ?></div>
            <div style="color: #666; font-size: 12px;">Last 5 minutes</div>
        </div>
        <div class="cartt-stat-card" style="background: #fff; padding: 20px; border-radius: 8px; box-shadow: 0 1px 3px rgba(0,0,0,0.1);">
            <div style="color: #666; font-size: 13px; margin-bottom: 5px;">Active Carts</div>
            <div style="font-size: 28px; font-weight: 600; color: #50575e;"><?php echo $active_carts; ?></div>
            <div style="color: #666; font-size: 12px;">Last 60 minutes</div>
        </div>
        <div class="cartt-stat-card" style="background: #fff; padding: 20px; border-radius: 8px; box-shadow: 0 1px 3px rgba(0,0,0,0.1);">
            <div style="color: #666; font-size: 13px; margin-bottom: 5px;">Sales Today</div>
            <div style="font-size: 28px; font-weight: 600; color: #1d2327;"><?php echo $today_sales; ?></div>
            <div style="color: #666; font-size: 12px;">Completed orders</div>
        </div>
        <div class="cartt-stat-card" style="background: #fff; padding: 20px; border-radius: 8px; box-shadow: 0 1px 3px rgba(0,0,0,0.1);">
            <div style="color: #666; font-size: 13px; margin-bottom: 5px;">Last Hour</div>
            <div style="font-size: 28px; font-weight: 600; color: #dba617;"><?php echo $hour_sales; ?></div>
            <div style="color: #666; font-size: 12px;">Orders placed</div>
        </div>
    </div>

    <?php 
    $baseUrl = $isEmbedded ? '?page=cartt-ai-conversion&tab=social-proof&subtab=' : '?page=cartt-social-proof&subtab=';
    ?>
    <nav class="nav-tab-wrapper">
        <a href="<?php echo $baseUrl; ?>overview" class="nav-tab <?php echo $current_tab === 'overview' ? 'nav-tab-active' : ''; ?>">Overview</a>
        <a href="<?php echo $baseUrl; ?>notifications" class="nav-tab <?php echo $current_tab === 'notifications' ? 'nav-tab-active' : ''; ?>">Purchase Notifications</a>
        <a href="<?php echo $baseUrl; ?>visitors" class="nav-tab <?php echo $current_tab === 'visitors' ? 'nav-tab-active' : ''; ?>">Visitor Count</a>
        <a href="<?php echo $baseUrl; ?>urgency" class="nav-tab <?php echo $current_tab === 'urgency' ? 'nav-tab-active' : ''; ?>">Stock Urgency</a>
        <a href="<?php echo $baseUrl; ?>fomo" class="nav-tab <?php echo $current_tab === 'fomo' ? 'nav-tab-active' : ''; ?>">FOMO Elements</a>
        <a href="<?php echo $baseUrl; ?>preview" class="nav-tab <?php echo $current_tab === 'preview' ? 'nav-tab-active' : ''; ?>">Preview</a>
    </nav>

    <div style="background: #fff; padding: 20px; border: 1px solid #c3c4c7; border-top: none;">
        <?php if ($current_tab === 'overview'): ?>
            <h3>Social Proof Features</h3>
            <p style="color: #666; margin-bottom: 25px;">Enable psychological triggers that boost conversions by showing real-time activity and creating urgency.</p>

            <div class="cartt-features-grid" style="display: grid; grid-template-columns: repeat(3, 1fr); gap: 20px;">
                <div class="feature-card" style="border: 1px solid #ddd; border-radius: 8px; padding: 20px;">
                    <div style="display: flex; align-items: center; margin-bottom: 15px;">
                        <span class="dashicons dashicons-bell" style="font-size: 24px; color: #50575e; margin-right: 10px;"></span>
                        <h4 style="margin: 0;">Purchase Notifications</h4>
                    </div>
                    <p style="color: #666; font-size: 13px; margin-bottom: 15px;">Show popup notifications when other customers make purchases.</p>
                    <div style="display: flex; justify-content: space-between; align-items: center;">
                        <span class="status-badge <?php echo $settings['purchase_notifications']['enabled'] ? 'enabled' : 'disabled'; ?>" 
                              style="padding: 4px 10px; border-radius: 12px; font-size: 12px; background: <?php echo $settings['purchase_notifications']['enabled'] ? '#d4edda' : '#f8d7da'; ?>; color: <?php echo $settings['purchase_notifications']['enabled'] ? '#155724' : '#721c24'; ?>;">
                            <?php echo $settings['purchase_notifications']['enabled'] ? 'Enabled' : 'Disabled'; ?>
                        </span>
                        <a href="<?php echo $baseUrl; ?>notifications" class="button button-small">Configure</a>
                    </div>
                </div>

                <div class="feature-card" style="border: 1px solid #ddd; border-radius: 8px; padding: 20px;">
                    <div style="display: flex; align-items: center; margin-bottom: 15px;">
                        <span class="dashicons dashicons-groups" style="font-size: 24px; color: #1d2327; margin-right: 10px;"></span>
                        <h4 style="margin: 0;">Live Visitor Count</h4>
                    </div>
                    <p style="color: #666; font-size: 13px; margin-bottom: 15px;">Display how many people are currently viewing a product.</p>
                    <div style="display: flex; justify-content: space-between; align-items: center;">
                        <span class="status-badge" 
                              style="padding: 4px 10px; border-radius: 12px; font-size: 12px; background: <?php echo $settings['visitor_count']['enabled'] ? '#d4edda' : '#f8d7da'; ?>; color: <?php echo $settings['visitor_count']['enabled'] ? '#155724' : '#721c24'; ?>;">
                            <?php echo $settings['visitor_count']['enabled'] ? 'Enabled' : 'Disabled'; ?>
                        </span>
                        <a href="<?php echo $baseUrl; ?>visitors" class="button button-small">Configure</a>
                    </div>
                </div>

                <div class="feature-card" style="border: 1px solid #ddd; border-radius: 8px; padding: 20px;">
                    <div style="display: flex; align-items: center; margin-bottom: 15px;">
                        <span class="dashicons dashicons-warning" style="font-size: 24px; color: #1d2327; margin-right: 10px;"></span>
                        <h4 style="margin: 0;">Stock Urgency</h4>
                    </div>
                    <p style="color: #666; font-size: 13px; margin-bottom: 15px;">Show low stock warnings to create purchase urgency.</p>
                    <div style="display: flex; justify-content: space-between; align-items: center;">
                        <span class="status-badge" 
                              style="padding: 4px 10px; border-radius: 12px; font-size: 12px; background: <?php echo $settings['stock_urgency']['enabled'] ? '#d4edda' : '#f8d7da'; ?>; color: <?php echo $settings['stock_urgency']['enabled'] ? '#155724' : '#721c24'; ?>;">
                            <?php echo $settings['stock_urgency']['enabled'] ? 'Enabled' : 'Disabled'; ?>
                        </span>
                        <a href="<?php echo $baseUrl; ?>urgency" class="button button-small">Configure</a>
                    </div>
                </div>

                <div class="feature-card" style="border: 1px solid #ddd; border-radius: 8px; padding: 20px;">
                    <div style="display: flex; align-items: center; margin-bottom: 15px;">
                        <span class="dashicons dashicons-cart" style="font-size: 24px; color: #1d2327; margin-right: 10px;"></span>
                        <h4 style="margin: 0;">Cart Activity</h4>
                    </div>
                    <p style="color: #666; font-size: 13px; margin-bottom: 15px;">Show how many people have this item in their cart.</p>
                    <div style="display: flex; justify-content: space-between; align-items: center;">
                        <span class="status-badge" 
                              style="padding: 4px 10px; border-radius: 12px; font-size: 12px; background: <?php echo $settings['cart_activity']['enabled'] ? '#d4edda' : '#f8d7da'; ?>; color: <?php echo $settings['cart_activity']['enabled'] ? '#155724' : '#721c24'; ?>;">
                            <?php echo $settings['cart_activity']['enabled'] ? 'Enabled' : 'Disabled'; ?>
                        </span>
                        <a href="<?php echo $baseUrl; ?>fomo" class="button button-small">Configure</a>
                    </div>
                </div>

                <div class="feature-card" style="border: 1px solid #ddd; border-radius: 8px; padding: 20px;">
                    <div style="display: flex; align-items: center; margin-bottom: 15px;">
                        <span class="dashicons dashicons-chart-bar" style="font-size: 24px; color: #1d2327; margin-right: 10px;"></span>
                        <h4 style="margin: 0;">Recent Sales Count</h4>
                    </div>
                    <p style="color: #666; font-size: 13px; margin-bottom: 15px;">Display number of recent purchases for each product.</p>
                    <div style="display: flex; justify-content: space-between; align-items: center;">
                        <span class="status-badge" 
                              style="padding: 4px 10px; border-radius: 12px; font-size: 12px; background: <?php echo $settings['recent_sales']['enabled'] ? '#d4edda' : '#f8d7da'; ?>; color: <?php echo $settings['recent_sales']['enabled'] ? '#155724' : '#721c24'; ?>;">
                            <?php echo $settings['recent_sales']['enabled'] ? 'Enabled' : 'Disabled'; ?>
                        </span>
                        <a href="<?php echo $baseUrl; ?>fomo" class="button button-small">Configure</a>
                    </div>
                </div>

                <div class="feature-card" style="border: 1px solid #ddd; border-radius: 8px; padding: 20px;">
                    <div style="display: flex; align-items: center; margin-bottom: 15px;">
                        <span class="dashicons dashicons-clock" style="font-size: 24px; color: #1d2327; margin-right: 10px;"></span>
                        <h4 style="margin: 0;">FOMO Countdown</h4>
                    </div>
                    <p style="color: #666; font-size: 13px; margin-bottom: 15px;">Add countdown timers for sales or limited offers.</p>
                    <div style="display: flex; justify-content: space-between; align-items: center;">
                        <span class="status-badge" 
                              style="padding: 4px 10px; border-radius: 12px; font-size: 12px; background: <?php echo $settings['fomo_countdown']['enabled'] ? '#d4edda' : '#f8d7da'; ?>; color: <?php echo $settings['fomo_countdown']['enabled'] ? '#155724' : '#721c24'; ?>;">
                            <?php echo $settings['fomo_countdown']['enabled'] ? 'Enabled' : 'Disabled'; ?>
                        </span>
                        <a href="<?php echo $baseUrl; ?>fomo" class="button button-small">Configure</a>
                    </div>
                </div>
            </div>

            <div style="margin-top: 30px; padding: 20px; background: #f6f7f7; border-radius: 8px;">
                <h4 style="margin-top: 0;">Quick Enable/Disable</h4>
                <form method="post" action="" id="quick-toggle-form">
                    <?php wp_nonce_field('cartt_social_proof', 'cartt_nonce'); ?>
                    <label style="display: flex; align-items: center; cursor: pointer;">
                        <input type="checkbox" name="social_proof_enabled" value="1" <?php checked($settings['enabled']); ?> style="margin-right: 10px;">
                        <span>Enable all social proof features globally</span>
                    </label>
                    <p style="color: #666; font-size: 12px; margin: 10px 0 0 28px;">When disabled, no social proof elements will appear on your store.</p>
                </form>
            </div>

        <?php elseif ($current_tab === 'notifications'): ?>
            <form method="post" action="" id="notifications-form" style="max-width: 700px;">
                <?php wp_nonce_field('cartt_social_proof', 'cartt_nonce'); ?>
                
                <h3>Purchase Notifications</h3>
                <p style="color: #666; margin-bottom: 25px;">Show real-time popup notifications when customers make purchases, building trust and creating FOMO.</p>

                <table class="form-table">
                    <tr>
                        <th scope="row">Enable Notifications</th>
                        <td>
                            <label>
                                <input type="checkbox" name="notifications[enabled]" value="1" <?php checked($settings['purchase_notifications']['enabled']); ?>>
                                Show purchase notifications to visitors
                            </label>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">Position</th>
                        <td>
                            <select name="notifications[position]">
                                <option value="bottom-left" <?php selected($settings['purchase_notifications']['position'], 'bottom-left'); ?>>Bottom Left</option>
                                <option value="bottom-right" <?php selected($settings['purchase_notifications']['position'], 'bottom-right'); ?>>Bottom Right</option>
                                <option value="top-left" <?php selected($settings['purchase_notifications']['position'], 'top-left'); ?>>Top Left</option>
                                <option value="top-right" <?php selected($settings['purchase_notifications']['position'], 'top-right'); ?>>Top Right</option>
                            </select>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">Display Duration</th>
                        <td>
                            <input type="number" name="notifications[display_duration]" value="<?php echo esc_attr($settings['purchase_notifications']['display_duration']); ?>" min="2000" max="15000" step="500" style="width: 100px;"> ms
                            <p class="description">How long each notification stays visible (2000-15000ms)</p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">Delay Between</th>
                        <td>
                            <input type="number" name="notifications[delay_between]" value="<?php echo esc_attr($settings['purchase_notifications']['delay_between']); ?>" min="3000" max="60000" step="1000" style="width: 100px;"> ms
                            <p class="description">Time between notifications (3000-60000ms)</p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">Show Product Image</th>
                        <td>
                            <label>
                                <input type="checkbox" name="notifications[show_product_image]" value="1" <?php checked($settings['purchase_notifications']['show_product_image']); ?>>
                                Include product thumbnail in notification
                            </label>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">Show Location</th>
                        <td>
                            <label>
                                <input type="checkbox" name="notifications[show_location]" value="1" <?php checked($settings['purchase_notifications']['show_location']); ?>>
                                Show buyer's city/state (e.g., "from New York, NY")
                            </label>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">Anonymize Names</th>
                        <td>
                            <label>
                                <input type="checkbox" name="notifications[anonymize_names]" value="1" <?php checked($settings['purchase_notifications']['anonymize_names']); ?>>
                                Use first name and last initial only (e.g., "John D.")
                            </label>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">Order Age Limit</th>
                        <td>
                            <input type="number" name="notifications[max_age_hours]" value="<?php echo esc_attr($settings['purchase_notifications']['max_age_hours']); ?>" min="1" max="168" style="width: 80px;"> hours
                            <p class="description">Only show purchases from the last X hours</p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">Message Template</th>
                        <td>
                            <input type="text" name="notifications[message_template]" value="<?php echo esc_attr($settings['purchase_notifications']['message_template']); ?>" class="large-text">
                            <p class="description">Available placeholders: {name}, {location}, {product}, {time_ago}</p>
                        </td>
                    </tr>
                </table>

                <p class="submit">
                    <button type="submit" class="button button-primary">Save Notification Settings</button>
                </p>
            </form>

        <?php elseif ($current_tab === 'visitors'): ?>
            <form method="post" action="" id="visitors-form" style="max-width: 700px;">
                <?php wp_nonce_field('cartt_social_proof', 'cartt_nonce'); ?>
                
                <h3>Live Visitor Count</h3>
                <p style="color: #666; margin-bottom: 25px;">Display real-time visitor counts on product pages to show popularity and create social proof.</p>

                <table class="form-table">
                    <tr>
                        <th scope="row">Enable Visitor Count</th>
                        <td>
                            <label>
                                <input type="checkbox" name="visitors[enabled]" value="1" <?php checked($settings['visitor_count']['enabled']); ?>>
                                Show live visitor count on products
                            </label>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">Display Locations</th>
                        <td>
                            <label style="display: block; margin-bottom: 8px;">
                                <input type="checkbox" name="visitors[show_on_product]" value="1" <?php checked($settings['visitor_count']['show_on_product']); ?>>
                                Product pages
                            </label>
                            <label style="display: block;">
                                <input type="checkbox" name="visitors[show_on_category]" value="1" <?php checked($settings['visitor_count']['show_on_category']); ?>>
                                Category pages (per product)
                            </label>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">Time Window</th>
                        <td>
                            <input type="number" name="visitors[time_window]" value="<?php echo esc_attr($settings['visitor_count']['time_window']); ?>" min="5" max="60" style="width: 80px;"> minutes
                            <p class="description">Consider a visitor "active" if they viewed within this time</p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">Minimum Threshold</th>
                        <td>
                            <input type="number" name="visitors[minimum_threshold]" value="<?php echo esc_attr($settings['visitor_count']['minimum_threshold']); ?>" min="1" max="10" style="width: 80px;"> visitors
                            <p class="description">Only show count when at least this many visitors are active</p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">Message</th>
                        <td>
                            <input type="text" name="visitors[message]" value="<?php echo esc_attr($settings['visitor_count']['message']); ?>" class="large-text">
                            <p class="description">Use {count} for the visitor number</p>
                        </td>
                    </tr>
                </table>

                <p class="submit">
                    <button type="submit" class="button button-primary">Save Visitor Settings</button>
                </p>
            </form>

        <?php elseif ($current_tab === 'urgency'): ?>
            <form method="post" action="" id="urgency-form" style="max-width: 700px;">
                <?php wp_nonce_field('cartt_social_proof', 'cartt_nonce'); ?>
                
                <h3>Stock Urgency</h3>
                <p style="color: #666; margin-bottom: 25px;">Display low stock warnings to encourage immediate purchases and reduce cart abandonment.</p>

                <table class="form-table">
                    <tr>
                        <th scope="row">Enable Stock Urgency</th>
                        <td>
                            <label>
                                <input type="checkbox" name="urgency[enabled]" value="1" <?php checked($settings['stock_urgency']['enabled']); ?>>
                                Show low stock warnings on products
                            </label>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">Low Stock Threshold</th>
                        <td>
                            <input type="number" name="urgency[low_stock_threshold]" value="<?php echo esc_attr($settings['stock_urgency']['low_stock_threshold']); ?>" min="1" max="100" style="width: 80px;"> items
                            <p class="description">Show "low stock" warning when inventory falls below this</p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">Critical Stock Threshold</th>
                        <td>
                            <input type="number" name="urgency[critical_stock_threshold]" value="<?php echo esc_attr($settings['stock_urgency']['critical_stock_threshold']); ?>" min="1" max="20" style="width: 80px;"> items
                            <p class="description">Show urgent "almost gone" warning below this level</p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">Show Exact Count</th>
                        <td>
                            <label>
                                <input type="checkbox" name="urgency[show_exact_count]" value="1" <?php checked($settings['stock_urgency']['show_exact_count']); ?>>
                                Display the exact remaining quantity
                            </label>
                            <p class="description">If disabled, uses vague language like "only a few left"</p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">Low Stock Message</th>
                        <td>
                            <input type="text" name="urgency[low_stock_message]" value="<?php echo esc_attr($settings['stock_urgency']['low_stock_message']); ?>" class="large-text">
                            <p class="description">Shown when stock is low but not critical</p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">Critical Stock Message</th>
                        <td>
                            <input type="text" name="urgency[critical_stock_message]" value="<?php echo esc_attr($settings['stock_urgency']['critical_stock_message']); ?>" class="large-text">
                            <p class="description">Use {count} for the remaining quantity</p>
                        </td>
                    </tr>
                </table>

                <div style="margin-top: 20px; padding: 15px; background: #fff8e5; border-left: 4px solid #dba617; border-radius: 4px;">
                    <strong>Preview:</strong>
                    <div style="margin-top: 10px; padding: 10px; background: #fff; border-radius: 4px;">
                        <span style="color: #dba617; font-weight: 500;">Low stock:</span> <?php echo esc_html($settings['stock_urgency']['low_stock_message']); ?>
                    </div>
                    <div style="margin-top: 8px; padding: 10px; background: #fff; border-radius: 4px;">
                        <span style="color: #d63638; font-weight: 500;">Critical:</span> <?php echo esc_html(str_replace('{count}', '3', $settings['stock_urgency']['critical_stock_message'])); ?>
                    </div>
                </div>

                <p class="submit">
                    <button type="submit" class="button button-primary">Save Urgency Settings</button>
                </p>
            </form>

        <?php elseif ($current_tab === 'fomo'): ?>
            <form method="post" action="" id="fomo-form" style="max-width: 700px;">
                <?php wp_nonce_field('cartt_social_proof', 'cartt_nonce'); ?>
                
                <h3>FOMO Elements</h3>
                <p style="color: #666; margin-bottom: 25px;">Additional elements that create fear of missing out and encourage immediate action.</p>

                <div style="border: 1px solid #ddd; border-radius: 8px; padding: 20px; margin-bottom: 20px;">
                    <h4 style="margin-top: 0;">Cart Activity</h4>
                    <table class="form-table" style="margin: 0;">
                        <tr>
                            <th scope="row" style="width: 180px;">Enable</th>
                            <td>
                                <label>
                                    <input type="checkbox" name="cart_activity[enabled]" value="1" <?php checked($settings['cart_activity']['enabled']); ?>>
                                    Show how many people have this in cart
                                </label>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row">Time Window</th>
                            <td>
                                <input type="number" name="cart_activity[time_window]" value="<?php echo esc_attr($settings['cart_activity']['time_window']); ?>" min="15" max="240" style="width: 80px;"> minutes
                            </td>
                        </tr>
                        <tr>
                            <th scope="row">Minimum</th>
                            <td>
                                <input type="number" name="cart_activity[minimum_threshold]" value="<?php echo esc_attr($settings['cart_activity']['minimum_threshold']); ?>" min="1" max="10" style="width: 80px;"> in cart to show
                            </td>
                        </tr>
                        <tr>
                            <th scope="row">Message</th>
                            <td>
                                <input type="text" name="cart_activity[message]" value="<?php echo esc_attr($settings['cart_activity']['message']); ?>" class="large-text">
                            </td>
                        </tr>
                    </table>
                </div>

                <div style="border: 1px solid #ddd; border-radius: 8px; padding: 20px; margin-bottom: 20px;">
                    <h4 style="margin-top: 0;">Recent Sales</h4>
                    <table class="form-table" style="margin: 0;">
                        <tr>
                            <th scope="row" style="width: 180px;">Enable</th>
                            <td>
                                <label>
                                    <input type="checkbox" name="recent_sales[enabled]" value="1" <?php checked($settings['recent_sales']['enabled']); ?>>
                                    Show recent sales count on products
                                </label>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row">Time Window</th>
                            <td>
                                <input type="number" name="recent_sales[time_window]" value="<?php echo esc_attr($settings['recent_sales']['time_window']); ?>" min="1" max="168" style="width: 80px;"> hours
                            </td>
                        </tr>
                        <tr>
                            <th scope="row">Message</th>
                            <td>
                                <input type="text" name="recent_sales[message]" value="<?php echo esc_attr($settings['recent_sales']['message']); ?>" class="large-text">
                            </td>
                        </tr>
                    </table>
                </div>

                <div style="border: 1px solid #ddd; border-radius: 8px; padding: 20px;">
                    <h4 style="margin-top: 0;">Countdown Timer</h4>
                    <table class="form-table" style="margin: 0;">
                        <tr>
                            <th scope="row" style="width: 180px;">Enable</th>
                            <td>
                                <label>
                                    <input type="checkbox" name="fomo_countdown[enabled]" value="1" <?php checked($settings['fomo_countdown']['enabled']); ?>>
                                    Show countdown timers
                                </label>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row">Timer Type</th>
                            <td>
                                <select name="fomo_countdown[type]">
                                    <option value="sale_end" <?php selected($settings['fomo_countdown']['type'], 'sale_end'); ?>>Sale End Date</option>
                                    <option value="daily" <?php selected($settings['fomo_countdown']['type'], 'daily'); ?>>Daily Reset (midnight)</option>
                                    <option value="session" <?php selected($settings['fomo_countdown']['type'], 'session'); ?>>Session-based (fixed duration)</option>
                                </select>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row">Message</th>
                            <td>
                                <input type="text" name="fomo_countdown[message]" value="<?php echo esc_attr($settings['fomo_countdown']['message']); ?>" class="large-text">
                                <p class="description">Use {countdown} for the timer display</p>
                            </td>
                        </tr>
                    </table>
                </div>

                <p class="submit">
                    <button type="submit" class="button button-primary">Save FOMO Settings</button>
                </p>
            </form>

        <?php elseif ($current_tab === 'preview'): ?>
            <h3>Live Preview</h3>
            <p style="color: #666; margin-bottom: 25px;">These are examples of how social proof elements will appear on your store. Configure each element in its respective settings tab.</p>

            <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 30px;">
                <div>
                    <h4>Purchase Notification</h4>
                    <div id="notification-preview" style="position: relative; width: 320px; background: #fff; border-radius: 8px; box-shadow: 0 4px 20px rgba(0,0,0,0.15); padding: 12px; display: flex; align-items: center; gap: 12px;">
                        <div style="width: 50px; height: 50px; background: #f0f0f0; border-radius: 6px; display: flex; align-items: center; justify-content: center;">
                            <span class="dashicons dashicons-products" style="color: #999;"></span>
                        </div>
                        <div style="flex: 1;">
                            <div style="font-size: 13px; color: #333; font-weight: 500;">John D. from New York, NY</div>
                            <div style="font-size: 12px; color: #666; margin-top: 2px;">purchased <strong>Premium Widget</strong></div>
                            <div style="font-size: 11px; color: #999; margin-top: 3px;">2 minutes ago</div>
                        </div>
                        <button type="button" style="position: absolute; top: 8px; right: 8px; background: none; border: none; cursor: pointer; color: #999;">&times;</button>
                    </div>
                </div>

                <div>
                    <h4>Visitor Count</h4>
                    <div style="display: flex; align-items: center; gap: 8px; padding: 10px 15px; background: #f6f7f7; border-radius: 6px; width: fit-content;">
                        <span class="dashicons dashicons-visibility" style="color: #50575e;"></span>
                        <span style="font-size: 13px; color: #333;"><strong>12</strong> people are viewing this right now</span>
                    </div>
                </div>

                <div>
                    <h4>Stock Urgency - Low</h4>
                    <div style="display: flex; align-items: center; gap: 8px; padding: 10px 15px; background: #fff8e5; border-left: 3px solid #dba617; width: fit-content;">
                        <span class="dashicons dashicons-warning" style="color: #1d2327;"></span>
                        <span style="font-size: 13px; color: #92660e;">Only a few left in stock</span>
                    </div>
                </div>

                <div>
                    <h4>Stock Urgency - Critical</h4>
                    <div style="display: flex; align-items: center; gap: 8px; padding: 10px 15px; background: #fcf0f0; border-left: 3px solid #d63638; width: fit-content;">
                        <span class="dashicons dashicons-warning" style="color: #1d2327;"></span>
                        <span style="font-size: 13px; color: #8a1c1c;">Almost gone! Only 2 left</span>
                    </div>
                </div>

                <div>
                    <h4>Cart Activity</h4>
                    <div style="display: flex; align-items: center; gap: 8px; padding: 10px 15px; background: #f6f7f7; border-radius: 6px; width: fit-content;">
                        <span class="dashicons dashicons-cart" style="color: #50575e;"></span>
                        <span style="font-size: 13px; color: #333;"><strong>5</strong> people have this in their cart</span>
                    </div>
                </div>

                <div>
                    <h4>Recent Sales</h4>
                    <div style="display: flex; align-items: center; gap: 8px; padding: 10px 15px; background: #edfaef; border-radius: 6px; width: fit-content;">
                        <span class="dashicons dashicons-chart-line" style="color: #1d2327;"></span>
                        <span style="font-size: 13px; color: #333;"><strong>24</strong> sold in the last 24 hours</span>
                    </div>
                </div>

                <div style="grid-column: span 2;">
                    <h4>Countdown Timer</h4>
                    <div style="display: inline-flex; align-items: center; gap: 10px; padding: 10px 15px; background: #1d2327; border-radius: 6px; color: #fff;">
                        <span class="dashicons dashicons-clock" style="font-size: 18px;"></span>
                        <div>
                            <span style="font-size: 11px; text-transform: uppercase; opacity: 0.8;">Sale ends in</span>
                            <span style="font-size: 16px; font-weight: 600; font-family: monospace; margin-left: 8px;">02:15:43</span>
                        </div>
                    </div>
                </div>
            </div>

        <?php endif; ?>
    </div>
<?php if (!$isEmbedded): ?>
</div>
<?php endif; ?>

<style>
.cartt-admin .form-table th {
    padding: 15px 10px 15px 0;
    width: 200px;
}
.cartt-admin .form-table td {
    padding: 10px 0;
}
</style>

<script>
jQuery(document).ready(function($) {
    // Quick toggle form auto-save
    $('#quick-toggle-form input[type="checkbox"]').on('change', function() {
        var data = {
            action: 'cartt_save_social_proof_settings',
            nonce: '<?php echo wp_create_nonce('cartt_social_proof'); ?>',
            settings: { enabled: $(this).is(':checked') ? 1 : 0 }
        };
        
        $.post(ajaxurl, data, function(response) {
            if (response.success) {
                location.reload();
            }
        });
    });

    // Form submissions
    $('#notifications-form, #visitors-form, #urgency-form, #fomo-form').on('submit', function(e) {
        e.preventDefault();
        
        var $form = $(this);
        var $btn = $form.find('button[type="submit"]');
        var originalText = $btn.text();
        
        $btn.prop('disabled', true).text('Saving...');
        
        $.post(ajaxurl, {
            action: 'cartt_save_social_proof_settings',
            nonce: $form.find('input[name="cartt_nonce"]').val(),
            settings: $form.serialize()
        }, function(response) {
            $btn.prop('disabled', false).text(originalText);
            
            if (response.success) {
                $('<div class="notice notice-success is-dismissible"><p>Settings saved successfully.</p></div>')
                    .insertAfter('.wp-header-end')
                    .delay(3000)
                    .fadeOut();
            } else {
                alert('Error saving settings: ' + (response.data || 'Unknown error'));
            }
        });
    });
});
</script>
