<?php
/**
 * Shipping Admin Page
 */

if (!defined('ABSPATH')) {
    exit;
}

$tab = isset($_GET['tab']) ? sanitize_text_field($_GET['tab']) : 'methods';
$addons = \Cartt\Core\Plugin::getActiveAddons();
$pendingRMA = \Cartt\Services\RMAService::getPendingCount();
?>

<div class="wrap cartt-admin">
    <nav class="nav-tab-wrapper" style="margin-bottom: 20px;">
        <a href="?page=cartt-shipping&tab=methods" class="nav-tab <?php echo $tab === 'methods' ? 'nav-tab-active' : ''; ?>">
            Zones & Methods
        </a>
        <a href="?page=cartt-shipping&tab=returns" class="nav-tab <?php echo $tab === 'returns' ? 'nav-tab-active' : ''; ?>">
            Returns
            <?php if ($pendingRMA > 0): ?>
            <span class="awaiting-mod"><?php echo $pendingRMA; ?></span>
            <?php endif; ?>
        </a>
        <a href="?page=cartt-shipping&tab=live-rates" class="nav-tab <?php echo $tab === 'live-rates' ? 'nav-tab-active' : ''; ?>">
            Live Rates & Tracking
            <?php if (!in_array('live_shipping', $addons)): ?>
            <span class="cartt-pro-badge">PRO</span>
            <?php endif; ?>
        </a>
    </nav>

<?php
// Show content based on tab
if ($tab === 'returns') {
    include CARTT_PLUGIN_DIR . 'admin/views/rma.php';
    echo '</div>';
    return;
}
if ($tab === 'live-rates') {
    if (in_array('live_shipping', $addons)) {
        include CARTT_PLUGIN_DIR . 'admin/views/live-rates.php';
    } else {
        include CARTT_PLUGIN_DIR . 'admin/views/partials/addon-promo.php';
    }
    echo '</div>';
    return;
}

// Default: Zones & Methods tab
$shippingService = new \Cartt\Services\ShippingService();
$countries = \Cartt\Services\ShippingService::getCountries();

// Handle actions
$message = '';
$action = $_GET['action'] ?? 'list';
$zoneId = isset($_GET['zone']) ? (int)$_GET['zone'] : 0;

// Continue with original shipping logic (zones & methods)

// Save zone
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['cartt_shipping_nonce'])) {
    if (wp_verify_nonce($_POST['cartt_shipping_nonce'], 'cartt_save_shipping')) {
        
        if (isset($_POST['save_zone'])) {
            $zoneData = [
                'name' => $_POST['zone_name'] ?? '',
                'countries' => !empty($_POST['zone_countries']) ? implode(',', $_POST['zone_countries']) : null,
                'postcodes' => $_POST['zone_postcodes'] ?? null,
            ];
            
            if (!empty($_POST['zone_id'])) {
                $shippingService->updateZone((int)$_POST['zone_id'], $zoneData);
                $message = __('Zone updated.', 'cartt');
            } else {
                $zoneId = $shippingService->createZone($zoneData);
                $message = __('Zone created.', 'cartt');
            }
        }
        
        if (isset($_POST['save_method'])) {
            $methodData = [
                'zone_id' => (int)$_POST['method_zone_id'],
                'name' => $_POST['method_name'] ?? '',
                'method_type' => $_POST['method_type'] ?? 'flat_rate',
                'cost' => $_POST['method_cost'] ?? 0,
                'min_amount' => $_POST['method_min_amount'] ?? null,
                'enabled' => isset($_POST['method_enabled']),
            ];
            
            if (!empty($_POST['method_id'])) {
                $shippingService->updateMethod((int)$_POST['method_id'], $methodData);
            } else {
                $shippingService->createMethod($methodData);
            }
            $message = __('Shipping method saved.', 'cartt');
            $zoneId = $methodData['zone_id'];
        }
    }
}

// Delete zone
if (isset($_GET['delete_zone']) && isset($_GET['_wpnonce'])) {
    if (wp_verify_nonce($_GET['_wpnonce'], 'delete_zone_' . $_GET['delete_zone'])) {
        $shippingService->deleteZone((int)$_GET['delete_zone']);
        $message = __('Zone deleted.', 'cartt');
    }
}

// Delete method
if (isset($_GET['delete_method']) && isset($_GET['_wpnonce'])) {
    if (wp_verify_nonce($_GET['_wpnonce'], 'delete_method_' . $_GET['delete_method'])) {
        $method = $shippingService->getMethod((int)$_GET['delete_method']);
        $shippingService->deleteMethod((int)$_GET['delete_method']);
        $message = __('Method deleted.', 'cartt');
        $zoneId = $method ? $method->zone_id : 0;
    }
}

$zones = $shippingService->getZones();
$editZone = $zoneId ? $shippingService->getZone($zoneId) : null;
$zoneMethods = $editZone ? $shippingService->getMethods($zoneId) : [];
?>

<div class="wrap cartt-wrap">
    <div class="cartt-header">
        <h1><?php esc_html_e('Shipping', 'cartt'); ?></h1>
        <?php if (!$editZone): ?>
        <button type="button" class="cartt-btn cartt-btn-primary" id="add-zone-btn">
            <?php esc_html_e('Add Zone', 'cartt'); ?>
        </button>
        <?php endif; ?>
    </div>

    <?php if ($message): ?>
        <div class="notice notice-success is-dismissible"><p><?php echo esc_html($message); ?></p></div>
    <?php endif; ?>

    <?php if ($editZone): ?>
    <!-- Edit Zone -->
    <div class="cartt-shipping-edit">
        <a href="<?php echo esc_url(admin_url('admin.php?page=cartt-shipping')); ?>" class="cartt-back-link">
            ← <?php esc_html_e('Back to zones', 'cartt'); ?>
        </a>
        
        <form method="post" class="cartt-zone-form">
            <?php wp_nonce_field('cartt_save_shipping', 'cartt_shipping_nonce'); ?>
            <input type="hidden" name="zone_id" value="<?php echo esc_attr($editZone->id); ?>">
            
            <div class="cartt-form-section">
                <h2><?php esc_html_e('Zone Settings', 'cartt'); ?></h2>
                
                <div class="cartt-form-row">
                    <label><?php esc_html_e('Zone Name', 'cartt'); ?></label>
                    <input type="text" name="zone_name" value="<?php echo esc_attr($editZone->name); ?>" required>
                </div>
                
                <div class="cartt-form-row">
                    <label><?php esc_html_e('Countries', 'cartt'); ?></label>
                    <select name="zone_countries[]" multiple class="cartt-multi-select">
                        <?php 
                        $selectedCountries = $editZone->countries ? explode(',', $editZone->countries) : [];
                        foreach ($countries as $code => $name): 
                        ?>
                        <option value="<?php echo esc_attr($code); ?>" <?php echo in_array($code, $selectedCountries) ? 'selected' : ''; ?>>
                            <?php echo esc_html($name); ?>
                        </option>
                        <?php endforeach; ?>
                    </select>
                    <p class="cartt-form-hint"><?php esc_html_e('Hold Ctrl/Cmd to select multiple.', 'cartt'); ?></p>
                </div>
                
                <div class="cartt-form-row">
                    <label><?php esc_html_e('Postcodes', 'cartt'); ?></label>
                    <textarea name="zone_postcodes" rows="3" placeholder="<?php esc_attr_e('One per line. Wildcards: 90* or ranges: 90000...90999', 'cartt'); ?>"><?php echo esc_textarea($editZone->postcodes ?? ''); ?></textarea>
                </div>
                
                <button type="submit" name="save_zone" class="button button-primary"><?php esc_html_e('Save Zone', 'cartt'); ?></button>
            </div>
        </form>
        
        <!-- Shipping Methods -->
        <div class="cartt-form-section">
            <div class="cartt-section-header">
                <h2><?php esc_html_e('Shipping Methods', 'cartt'); ?></h2>
                <button type="button" class="button" id="add-method-btn"><?php esc_html_e('Add Method', 'cartt'); ?></button>
            </div>
            
            <?php if (empty($zoneMethods)): ?>
            <p class="cartt-empty-hint"><?php esc_html_e('No shipping methods. Add one to enable shipping for this zone.', 'cartt'); ?></p>
            <?php else: ?>
            <table class="cartt-table cartt-methods-table">
                <thead>
                    <tr>
                        <th><?php esc_html_e('Method', 'cartt'); ?></th>
                        <th><?php esc_html_e('Type', 'cartt'); ?></th>
                        <th><?php esc_html_e('Cost', 'cartt'); ?></th>
                        <th><?php esc_html_e('Status', 'cartt'); ?></th>
                        <th></th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($zoneMethods as $method): 
                        $typeLabels = [
                            'flat_rate' => __('Flat Rate', 'cartt'),
                            'free_shipping' => __('Free Shipping', 'cartt'),
                            'local_pickup' => __('Local Pickup', 'cartt'),
                        ];
                    ?>
                    <tr>
                        <td><strong><?php echo esc_html($method->name); ?></strong></td>
                        <td><?php echo esc_html($typeLabels[$method->method_type] ?? $method->method_type); ?></td>
                        <td>
                            <?php if ($method->method_type === 'free_shipping'): ?>
                                <?php if ($method->min_amount): ?>
                                    <?php printf(__('Free over $%s', 'cartt'), number_format($method->min_amount, 2)); ?>
                                <?php else: ?>
                                    <?php esc_html_e('Free', 'cartt'); ?>
                                <?php endif; ?>
                            <?php else: ?>
                                $<?php echo number_format($method->cost, 2); ?>
                            <?php endif; ?>
                        </td>
                        <td>
                            <span class="cartt-status <?php echo $method->enabled ? 'cartt-status-active' : 'cartt-status-inactive'; ?>">
                                <?php echo $method->enabled ? esc_html__('Active', 'cartt') : esc_html__('Disabled', 'cartt'); ?>
                            </span>
                        </td>
                        <td class="cartt-actions">
                            <button type="button" class="cartt-action-link edit-method-btn" 
                                    data-id="<?php echo esc_attr($method->id); ?>"
                                    data-name="<?php echo esc_attr($method->name); ?>"
                                    data-type="<?php echo esc_attr($method->method_type); ?>"
                                    data-cost="<?php echo esc_attr($method->cost); ?>"
                                    data-min="<?php echo esc_attr($method->min_amount); ?>"
                                    data-enabled="<?php echo esc_attr($method->enabled); ?>">
                                <?php esc_html_e('Edit', 'cartt'); ?>
                            </button>
                            <a href="<?php echo esc_url(wp_nonce_url(admin_url('admin.php?page=cartt-shipping&zone=' . $zoneId . '&delete_method=' . $method->id), 'delete_method_' . $method->id)); ?>" 
                               class="cartt-action-link cartt-action-delete"
                               onclick="return confirm('<?php esc_attr_e('Delete this method?', 'cartt'); ?>');">
                                <?php esc_html_e('Delete', 'cartt'); ?>
                            </a>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
            <?php endif; ?>
        </div>
    </div>
    
    <!-- Method Modal -->
    <div id="method-modal" class="cartt-modal" style="display:none;">
        <div class="cartt-modal-content">
            <h3 id="method-modal-title"><?php esc_html_e('Add Shipping Method', 'cartt'); ?></h3>
            <form method="post">
                <?php wp_nonce_field('cartt_save_shipping', 'cartt_shipping_nonce'); ?>
                <input type="hidden" name="method_id" id="method-id" value="">
                <input type="hidden" name="method_zone_id" value="<?php echo esc_attr($zoneId); ?>">
                
                <div class="cartt-form-row">
                    <label><?php esc_html_e('Name', 'cartt'); ?></label>
                    <input type="text" name="method_name" id="method-name" required placeholder="<?php esc_attr_e('Standard Shipping', 'cartt'); ?>">
                </div>
                
                <div class="cartt-form-row">
                    <label><?php esc_html_e('Type', 'cartt'); ?></label>
                    <select name="method_type" id="method-type">
                        <option value="flat_rate"><?php esc_html_e('Flat Rate', 'cartt'); ?></option>
                        <option value="free_shipping"><?php esc_html_e('Free Shipping', 'cartt'); ?></option>
                        <option value="local_pickup"><?php esc_html_e('Local Pickup', 'cartt'); ?></option>
                    </select>
                </div>
                
                <div class="cartt-form-row" id="cost-row">
                    <label><?php esc_html_e('Cost', 'cartt'); ?></label>
                    <input type="number" name="method_cost" id="method-cost" step="0.01" min="0" value="0">
                </div>
                
                <div class="cartt-form-row" id="min-amount-row" style="display:none;">
                    <label><?php esc_html_e('Minimum Order Amount', 'cartt'); ?></label>
                    <input type="number" name="method_min_amount" id="method-min-amount" step="0.01" min="0" placeholder="<?php esc_attr_e('No minimum', 'cartt'); ?>">
                    <p class="cartt-form-hint"><?php esc_html_e('Order must be at least this amount for free shipping.', 'cartt'); ?></p>
                </div>
                
                <div class="cartt-form-row">
                    <label class="cartt-checkbox-label">
                        <input type="checkbox" name="method_enabled" id="method-enabled" checked>
                        <?php esc_html_e('Enabled', 'cartt'); ?>
                    </label>
                </div>
                
                <div class="cartt-modal-actions">
                    <button type="button" class="button" id="cancel-method"><?php esc_html_e('Cancel', 'cartt'); ?></button>
                    <button type="submit" name="save_method" class="button button-primary"><?php esc_html_e('Save Method', 'cartt'); ?></button>
                </div>
            </form>
        </div>
    </div>

    <?php else: ?>
    <!-- Zones List -->
    <div class="cartt-zones-intro" style="background:#f6f7f7;padding:16px 20px;border-radius:8px;margin-bottom:20px;">
        <p style="margin:0;color:#50575e;">
            <strong>Shipping zones</strong> let you define shipping rates for different regions. 
            Create a zone, add the countries/postcodes it covers, then add shipping methods (flat rate, free shipping, local pickup) with your prices.
            <span class="cartt-tooltip">
                <span class="cartt-tooltip-icon">?</span>
                <span class="cartt-tooltip-text">Example: Create a "USA" zone, add United States, then add "Standard $5.99" and "Express $12.99" methods. Orders from the US will see these options at checkout.</span>
            </span>
        </p>
    </div>
    
    <div class="cartt-zones-list">
        <?php if (empty($zones)): ?>
        <div class="cartt-empty-state">
            <p><?php esc_html_e('No shipping zones configured.', 'cartt'); ?></p>
            <p class="cartt-empty-hint"><?php esc_html_e('Create zones to define shipping rates for different regions.', 'cartt'); ?></p>
        </div>
        <?php else: ?>
        <table class="cartt-table">
            <thead>
                <tr>
                    <th><?php esc_html_e('Zone', 'cartt'); ?></th>
                    <th><?php esc_html_e('Regions', 'cartt'); ?></th>
                    <th><?php esc_html_e('Methods', 'cartt'); ?></th>
                    <th></th>
                </tr>
            </thead>
            <tbody>
                <?php foreach ($zones as $zone): 
                    $methods = $shippingService->getMethods($zone->id);
                    $countryList = $zone->countries ? explode(',', $zone->countries) : [];
                    $countryNames = array_map(fn($c) => $countries[$c] ?? $c, array_slice($countryList, 0, 3));
                    $regionText = implode(', ', $countryNames);
                    if (count($countryList) > 3) $regionText .= ' +' . (count($countryList) - 3);
                    if (!$regionText && !$zone->postcodes) $regionText = __('Everywhere', 'cartt');
                ?>
                <tr>
                    <td>
                        <a href="<?php echo esc_url(admin_url('admin.php?page=cartt-shipping&zone=' . $zone->id)); ?>">
                            <strong><?php echo esc_html($zone->name); ?></strong>
                        </a>
                    </td>
                    <td><?php echo esc_html($regionText); ?></td>
                    <td><?php echo count($methods); ?> <?php echo count($methods) === 1 ? __('method', 'cartt') : __('methods', 'cartt'); ?></td>
                    <td class="cartt-actions">
                        <a href="<?php echo esc_url(admin_url('admin.php?page=cartt-shipping&zone=' . $zone->id)); ?>" class="cartt-action-link">
                            <?php esc_html_e('Edit', 'cartt'); ?>
                        </a>
                        <a href="<?php echo esc_url(wp_nonce_url(admin_url('admin.php?page=cartt-shipping&delete_zone=' . $zone->id), 'delete_zone_' . $zone->id)); ?>" 
                           class="cartt-action-link cartt-action-delete"
                           onclick="return confirm('<?php esc_attr_e('Delete this zone and all its methods?', 'cartt'); ?>');">
                            <?php esc_html_e('Delete', 'cartt'); ?>
                        </a>
                    </td>
                </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
        <?php endif; ?>
    </div>
    
    <!-- Add Zone Modal -->
    <div id="zone-modal" class="cartt-modal" style="display:none;">
        <div class="cartt-modal-content">
            <h3><?php esc_html_e('Add Shipping Zone', 'cartt'); ?></h3>
            <form method="post">
                <?php wp_nonce_field('cartt_save_shipping', 'cartt_shipping_nonce'); ?>
                
                <div class="cartt-form-row">
                    <label><?php esc_html_e('Zone Name', 'cartt'); ?></label>
                    <input type="text" name="zone_name" required placeholder="<?php esc_attr_e('e.g., United States, Europe', 'cartt'); ?>">
                </div>
                
                <div class="cartt-form-row">
                    <label><?php esc_html_e('Countries', 'cartt'); ?></label>
                    <select name="zone_countries[]" multiple class="cartt-multi-select">
                        <?php foreach ($countries as $code => $name): ?>
                        <option value="<?php echo esc_attr($code); ?>"><?php echo esc_html($name); ?></option>
                        <?php endforeach; ?>
                    </select>
                </div>
                
                <div class="cartt-modal-actions">
                    <button type="button" class="button" id="cancel-zone"><?php esc_html_e('Cancel', 'cartt'); ?></button>
                    <button type="submit" name="save_zone" class="button button-primary"><?php esc_html_e('Create Zone', 'cartt'); ?></button>
                </div>
            </form>
        </div>
    </div>
    <?php endif; ?>
</div>

<style>
.cartt-back-link { display: inline-block; margin-bottom: 16px; color: #0066cc; text-decoration: none; }
.cartt-back-link:hover { text-decoration: underline; }

.cartt-form-section { background: #fff; border: 1px solid #e8e8ed; border-radius: 8px; padding: 20px; margin-bottom: 16px; }
.cartt-form-section h2 { font-size: 14px; font-weight: 600; margin: 0 0 16px; }
.cartt-section-header { display: flex; justify-content: space-between; align-items: center; margin-bottom: 16px; }
.cartt-section-header h2 { margin: 0; }

.cartt-form-row { margin-bottom: 16px; }
.cartt-form-row label { display: block; font-size: 12px; font-weight: 500; margin-bottom: 6px; }
.cartt-form-row input, .cartt-form-row select, .cartt-form-row textarea { width: 100%; padding: 8px 12px; font-size: 13px; border: 1px solid #d2d2d7; border-radius: 6px; }
.cartt-form-row input:focus, .cartt-form-row select:focus, .cartt-form-row textarea:focus { border-color: #1d1d1f; outline: none; }
.cartt-form-hint { font-size: 11px; color: #86868b; margin: 4px 0 0; }
.cartt-empty-hint { color: #86868b; font-size: 13px; }

.cartt-multi-select { height: 120px; }

.cartt-methods-table { margin-top: 0; }

.cartt-modal { position: fixed; top: 0; left: 0; right: 0; bottom: 0; background: rgba(0,0,0,0.5); display: flex; align-items: center; justify-content: center; z-index: 100000; }
.cartt-modal-content { background: #fff; border-radius: 8px; padding: 24px; width: 100%; max-width: 400px; }
.cartt-modal-content h3 { margin: 0 0 20px; font-size: 16px; }
.cartt-modal-actions { display: flex; gap: 8px; justify-content: flex-end; margin-top: 20px; }

.cartt-checkbox-label { display: flex; align-items: center; gap: 8px; font-size: 13px; cursor: pointer; }
.cartt-checkbox-label input { width: auto; margin: 0; }

.cartt-empty-state { text-align: center; padding: 60px 20px; background: #f5f5f7; border-radius: 8px; }
.cartt-empty-state p { margin: 0 0 8px; }
</style>

<script>
jQuery(document).ready(function($) {
    // Zone modal
    $('#add-zone-btn').on('click', function() {
        $('#zone-modal').show();
    });
    $('#cancel-zone').on('click', function() {
        $('#zone-modal').hide();
    });
    
    // Method modal
    $('#add-method-btn').on('click', function() {
        $('#method-modal-title').text('<?php echo esc_js(__('Add Shipping Method', 'cartt')); ?>');
        $('#method-id').val('');
        $('#method-name').val('');
        $('#method-type').val('flat_rate').trigger('change');
        $('#method-cost').val('0');
        $('#method-min-amount').val('');
        $('#method-enabled').prop('checked', true);
        $('#method-modal').show();
    });
    
    $('.edit-method-btn').on('click', function() {
        $('#method-modal-title').text('<?php echo esc_js(__('Edit Shipping Method', 'cartt')); ?>');
        $('#method-id').val($(this).data('id'));
        $('#method-name').val($(this).data('name'));
        $('#method-type').val($(this).data('type')).trigger('change');
        $('#method-cost').val($(this).data('cost'));
        $('#method-min-amount').val($(this).data('min') || '');
        $('#method-enabled').prop('checked', $(this).data('enabled') == 1);
        $('#method-modal').show();
    });
    
    $('#cancel-method').on('click', function() {
        $('#method-modal').hide();
    });
    
    // Toggle cost/min fields based on type
    $('#method-type').on('change', function() {
        var type = $(this).val();
        if (type === 'free_shipping') {
            $('#cost-row').hide();
            $('#min-amount-row').show();
        } else {
            $('#cost-row').show();
            $('#min-amount-row').hide();
        }
    });
    
    // Close modals on backdrop click
    $('.cartt-modal').on('click', function(e) {
        if (e.target === this) {
            $(this).hide();
        }
    });
});
</script>
