<?php
/**
 * Settings View
 */

if (!defined('ABSPATH')) {
    exit;
}
?>
<style>
.cartt-settings-layout {
    display: grid;
    grid-template-columns: 1fr 300px;
    gap: 24px;
    align-items: start;
    margin-top: 20px;
}
.cartt-settings-main {
    min-width: 0;
}
.cartt-settings-main .cartt-two-col {
    display: grid;
    grid-template-columns: 1fr 1fr;
    gap: 24px;
}
.cartt-settings-sidebar {
    min-width: 0;
}
@media (max-width: 1200px) {
    .cartt-settings-main .cartt-two-col {
        grid-template-columns: 1fr;
    }
}
@media (max-width: 960px) {
    .cartt-settings-layout {
        grid-template-columns: 1fr;
    }
}
</style>
<?php
// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['cartt_settings_nonce'])) {
    if (wp_verify_nonce($_POST['cartt_settings_nonce'], 'cartt_save_settings')) {
        update_option('cartt_store_type', sanitize_text_field($_POST['store_type'] ?? 'mixed'));
        update_option('cartt_store_name', sanitize_text_field($_POST['store_name'] ?? ''));
        update_option('cartt_currency', sanitize_text_field($_POST['currency'] ?? 'USD'));
        update_option('cartt_country', sanitize_text_field($_POST['country'] ?? 'US'));
        update_option('cartt_tax_enabled', isset($_POST['tax_enabled']));
        
        // Inventory settings
        update_option('cartt_low_stock_threshold', intval($_POST['low_stock_threshold'] ?? 5));
        update_option('cartt_low_stock_notifications', isset($_POST['low_stock_notifications']));
        
        // Review settings
        update_option('cartt_reviews_enabled', isset($_POST['reviews_enabled']));
        update_option('cartt_reviews_verified_only', isset($_POST['reviews_verified_only']));
        update_option('cartt_reviews_auto_approve_verified', isset($_POST['reviews_auto_approve_verified']));
        
        // Stripe settings
        if (!empty($_POST['stripe_publishable_key'])) {
            update_option('cartt_stripe_publishable_key', sanitize_text_field($_POST['stripe_publishable_key']));
        }
        if (!empty($_POST['stripe_secret_key'])) {
            update_option('cartt_stripe_secret_key', sanitize_text_field($_POST['stripe_secret_key']));
        }
        update_option('cartt_stripe_test_mode', isset($_POST['stripe_test_mode']));
        
        // PayPal settings
        update_option('cartt_paypal_enabled', isset($_POST['paypal_enabled']));
        update_option('cartt_paypal_sandbox', isset($_POST['paypal_sandbox']));
        if (!empty($_POST['paypal_client_id'])) {
            update_option('cartt_paypal_client_id', sanitize_text_field($_POST['paypal_client_id']));
        }
        if (!empty($_POST['paypal_secret'])) {
            update_option('cartt_paypal_secret', sanitize_text_field($_POST['paypal_secret']));
        }
        
        echo '<div class="notice notice-success is-dismissible"><p>' . esc_html__('Settings saved.', 'cartt') . '</p></div>';
    }
}

$store_type = get_option('cartt_store_type', 'mixed');
$store_name = get_option('cartt_store_name', get_bloginfo('name'));
$currency = get_option('cartt_currency', 'USD');
$country = get_option('cartt_country', 'US');
$tax_enabled = get_option('cartt_tax_enabled', false);
$payment_methods = get_option('cartt_payment_methods', []);

// Stripe settings
$stripe_publishable = get_option('cartt_stripe_publishable_key', '');
$stripe_secret = get_option('cartt_stripe_secret_key', '');
$stripe_test_mode = get_option('cartt_stripe_test_mode', true);

$currencies = [
    'USD' => 'USD ($)',
    'EUR' => 'EUR (€)',
    'GBP' => 'GBP (£)',
    'CAD' => 'CAD ($)',
    'AUD' => 'AUD ($)',
    'JPY' => 'JPY (¥)',
    'CNY' => 'CNY (¥)',
    'INR' => 'INR (₹)',
    'BRL' => 'BRL (R$)',
    'MXN' => 'MXN ($)',
];

$countries = [
    'US' => __('United States', 'cartt'),
    'CA' => __('Canada', 'cartt'),
    'GB' => __('United Kingdom', 'cartt'),
    'AU' => __('Australia', 'cartt'),
    'DE' => __('Germany', 'cartt'),
    'FR' => __('France', 'cartt'),
];

$store_types = [
    'physical_goods' => __('Physical Products', 'cartt'),
    'digital_goods' => __('Digital Downloads', 'cartt'),
    'services' => __('Services & Bookings', 'cartt'),
    'subscriptions' => __('Subscriptions', 'cartt'),
    'mixed' => __('Mixed / All Features', 'cartt'),
];
?>

<div class="wrap cartt-admin-wrap">
    <div class="cartt-admin-header">
        <h1><?php esc_html_e('Settings', 'cartt'); ?></h1>
    </div>

    <form method="post" action="">
        <?php wp_nonce_field('cartt_save_settings', 'cartt_settings_nonce'); ?>
        
        <div class="cartt-settings-layout" style="display: grid; grid-template-columns: 1fr 300px; gap: 24px; align-items: start;">
            <div class="cartt-settings-main">
                <!-- Two column form layout -->
                <div class="cartt-two-col" style="display: grid; grid-template-columns: 1fr 1fr; gap: 24px;">
                    
                    <!-- Left Column -->
                    <div class="cartt-col">
                        <div class="cartt-form" style="background: #fff; border: 1px solid #e8e8ed; border-radius: 8px;">
                            <div class="cartt-form-section" style="padding: 20px 24px; border-bottom: 1px solid #f0f0f0;">
                                <h3 class="cartt-form-section-title"><?php esc_html_e('General', 'cartt'); ?></h3>
                                
                                <div class="cartt-form-row">
                                    <label for="store-name"><?php esc_html_e('Store Name', 'cartt'); ?></label>
                                    <input type="text" id="store-name" name="store_name" value="<?php echo esc_attr($store_name); ?>">
                                </div>

                                <div class="cartt-form-row">
                                    <label for="store-type"><?php esc_html_e('Store Type', 'cartt'); ?></label>
                                    <select id="store-type" name="store_type">
                                        <?php foreach ($store_types as $value => $label): ?>
                                            <option value="<?php echo esc_attr($value); ?>" <?php selected($store_type, $value); ?>>
                                                <?php echo esc_html($label); ?>
                                            </option>
                                        <?php endforeach; ?>
                                    </select>
                                    <p class="cartt-form-hint"><?php esc_html_e('This controls which features and options are visible in your admin.', 'cartt'); ?></p>
                                </div>
                            </div>

                            <div class="cartt-form-section" style="padding: 20px 24px; border-bottom: 1px solid #f0f0f0;">
                                <h3 class="cartt-form-section-title"><?php esc_html_e('Currency & Location', 'cartt'); ?></h3>
                                
                                <div class="cartt-form-row">
                                    <label for="currency"><?php esc_html_e('Currency', 'cartt'); ?></label>
                                    <select id="currency" name="currency">
                                        <?php foreach ($currencies as $code => $label): ?>
                                            <option value="<?php echo esc_attr($code); ?>" <?php selected($currency, $code); ?>>
                                                <?php echo esc_html($label); ?>
                                            </option>
                                        <?php endforeach; ?>
                                    </select>
                                </div>

                                <div class="cartt-form-row">
                                    <label for="country"><?php esc_html_e('Store Location', 'cartt'); ?></label>
                                    <select id="country" name="country">
                                        <?php foreach ($countries as $code => $label): ?>
                                            <option value="<?php echo esc_attr($code); ?>" <?php selected($country, $code); ?>>
                                                <?php echo esc_html($label); ?>
                                            </option>
                                        <?php endforeach; ?>
                                    </select>
                                </div>
                            </div>

                            <div class="cartt-form-section" style="padding: 20px 24px; border-bottom: 1px solid #f0f0f0;">
                                <h3 class="cartt-form-section-title"><?php esc_html_e('Tax', 'cartt'); ?></h3>
                                
                                <div class="cartt-form-row">
                                    <label>
                                        <input type="checkbox" name="tax_enabled" <?php checked($tax_enabled); ?>>
                                        <?php esc_html_e('Enable tax calculations', 'cartt'); ?>
                                    </label>
                                    <p class="cartt-form-hint"><?php esc_html_e('Configure tax rates in the Tax menu.', 'cartt'); ?></p>
                                </div>
                            </div>

                            <div class="cartt-form-section" style="padding: 20px 24px;">
                                <h3 class="cartt-form-section-title"><?php esc_html_e('Reviews', 'cartt'); ?></h3>
                                
                                <div class="cartt-form-row">
                                    <label>
                                        <input type="checkbox" name="reviews_enabled" <?php checked(get_option('cartt_reviews_enabled', true)); ?>>
                                        <?php esc_html_e('Enable product reviews', 'cartt'); ?>
                                    </label>
                                </div>
                                
                                <div class="cartt-form-row">
                                    <label>
                                        <input type="checkbox" name="reviews_verified_only" <?php checked(get_option('cartt_reviews_verified_only', true)); ?>>
                                        <?php esc_html_e('Only verified purchasers can review', 'cartt'); ?>
                                    </label>
                                </div>
                                
                                <div class="cartt-form-row">
                                    <label>
                                        <input type="checkbox" name="reviews_auto_approve_verified" <?php checked(get_option('cartt_reviews_auto_approve_verified', true)); ?>>
                                        <?php esc_html_e('Auto-approve verified purchase reviews', 'cartt'); ?>
                                    </label>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Right Column -->
                    <div class="cartt-col">
                        <div class="cartt-form" style="background: #fff; border: 1px solid #e8e8ed; border-radius: 8px;">
                            <div class="cartt-form-section" style="padding: 20px 24px; border-bottom: 1px solid #f0f0f0;">
                                <h3 class="cartt-form-section-title"><?php esc_html_e('Inventory', 'cartt'); ?></h3>
                                
                                <div class="cartt-form-row">
                                    <label for="low-stock-threshold"><?php esc_html_e('Low Stock Threshold', 'cartt'); ?></label>
                                    <input type="number" id="low-stock-threshold" name="low_stock_threshold" 
                                           value="<?php echo esc_attr(get_option('cartt_low_stock_threshold', 5)); ?>" 
                                           min="0" style="width: 80px;">
                                    <p class="cartt-form-hint"><?php esc_html_e('Products at or below this are low stock.', 'cartt'); ?></p>
                                </div>
                                
                                <div class="cartt-form-row">
                                    <label>
                                        <input type="checkbox" name="low_stock_notifications" <?php checked(get_option('cartt_low_stock_notifications', true)); ?>>
                                        <?php esc_html_e('Enable low stock email notifications', 'cartt'); ?>
                                    </label>
                                </div>
                            </div>

                            <div class="cartt-form-section" style="padding: 20px 24px; border-bottom: 1px solid #f0f0f0;">
                                <h3 class="cartt-form-section-title"><?php esc_html_e('Stripe', 'cartt'); ?></h3>
                                
                                <div class="cartt-form-row">
                                    <label>
                                        <input type="checkbox" name="stripe_test_mode" <?php checked($stripe_test_mode); ?>>
                                        <?php esc_html_e('Test mode', 'cartt'); ?>
                                    </label>
                                </div>

                                <div class="cartt-form-row">
                                    <label for="stripe-publishable"><?php esc_html_e('Publishable Key', 'cartt'); ?></label>
                                    <input type="text" id="stripe-publishable" name="stripe_publishable_key" 
                                           value="<?php echo esc_attr($stripe_publishable); ?>" 
                                           placeholder="pk_test_...">
                                </div>

                                <div class="cartt-form-row">
                                    <label for="stripe-secret"><?php esc_html_e('Secret Key', 'cartt'); ?></label>
                                    <input type="password" id="stripe-secret" name="stripe_secret_key" 
                                           value="<?php echo esc_attr($stripe_secret); ?>" 
                                           placeholder="sk_test_...">
                                    <p class="cartt-form-hint"><?php esc_html_e('Get keys from dashboard.stripe.com', 'cartt'); ?></p>
                                </div>

                                <?php if ($stripe_publishable && $stripe_secret): ?>
                                    <p style="color: #2e7d32; font-size: 12px;">Connected</p>
                                <?php endif; ?>
                            </div>

                            <div class="cartt-form-section" style="padding: 20px 24px;">
                                <h3 class="cartt-form-section-title"><?php esc_html_e('PayPal', 'cartt'); ?></h3>
                                
                                <div class="cartt-form-row">
                                    <label>
                                        <input type="checkbox" name="paypal_enabled" <?php checked(get_option('cartt_paypal_enabled')); ?>>
                                        <?php esc_html_e('Enable PayPal', 'cartt'); ?>
                                    </label>
                                </div>

                                <div class="cartt-form-row">
                                    <label>
                                        <input type="checkbox" name="paypal_sandbox" <?php checked(get_option('cartt_paypal_sandbox', true)); ?>>
                                        <?php esc_html_e('Sandbox mode', 'cartt'); ?>
                                    </label>
                                </div>

                                <div class="cartt-form-row">
                                    <label for="paypal-client-id"><?php esc_html_e('Client ID', 'cartt'); ?></label>
                                    <input type="text" id="paypal-client-id" name="paypal_client_id" 
                                           value="<?php echo esc_attr(get_option('cartt_paypal_client_id', '')); ?>" 
                                           placeholder="AY...">
                                </div>

                                <div class="cartt-form-row">
                                    <label for="paypal-secret"><?php esc_html_e('Secret', 'cartt'); ?></label>
                                    <input type="password" id="paypal-secret" name="paypal_secret" 
                                           value="<?php echo esc_attr(get_option('cartt_paypal_secret', '')); ?>" 
                                           placeholder="EL...">
                                    <p class="cartt-form-hint"><?php esc_html_e('Get from developer.paypal.com', 'cartt'); ?></p>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                </div>

                <div class="cartt-actions" style="margin-top: 20px;">
                    <button type="submit" class="button button-primary">
                        <?php esc_html_e('Save Settings', 'cartt'); ?>
                    </button>
                </div>
            </div>

            <div class="cartt-settings-sidebar">
                <div class="cartt-sidebar-panel" style="background: #fff; border: 1px solid #e8e8ed; border-radius: 8px; margin-bottom: 12px;">
                    <div class="cartt-sidebar-panel-header" style="padding: 12px 16px; border-bottom: 1px solid #f0f0f0; font-weight: 600; font-size: 13px;">
                        <?php esc_html_e('Store Pages', 'cartt'); ?>
                    </div>
                    <div class="cartt-sidebar-panel-content" style="padding: 16px;">
                        <p class="cartt-form-hint"><?php esc_html_e('Use these shortcodes to create your store pages:', 'cartt'); ?></p>
                        <p><code>[cartt_products]</code><br><small><?php esc_html_e('Product listing', 'cartt'); ?></small></p>
                        <p><code>[cartt_cart]</code><br><small><?php esc_html_e('Shopping cart', 'cartt'); ?></small></p>
                        <p><code>[cartt_checkout]</code><br><small><?php esc_html_e('Checkout page', 'cartt'); ?></small></p>
                        <p><code>[cartt_account]</code><br><small><?php esc_html_e('Customer account', 'cartt'); ?></small></p>
                    </div>
                </div>

                <div class="cartt-sidebar-panel" style="background: #fff; border: 1px solid #e8e8ed; border-radius: 8px; margin-bottom: 12px;">
                    <div class="cartt-sidebar-panel-header" style="padding: 12px 16px; border-bottom: 1px solid #f0f0f0; font-weight: 600; font-size: 13px;">
                        <?php esc_html_e('Import', 'cartt'); ?>
                    </div>
                    <div class="cartt-sidebar-panel-content" id="cartt-migration" style="padding: 16px;">
                        <div class="cartt-migration-loading">
                            <div class="cartt-spinner" style="width:20px;height:20px;border-width:2px;"></div>
                            <span style="margin-left:8px;font-size:12px;color:#86868b;">Checking...</span>
                        </div>
                    </div>
                </div>
<script>
(function() {
    const apiUrl = '<?php echo esc_js(rest_url('cartt/v1/')); ?>';
    const nonce = '<?php echo esc_js(wp_create_nonce('wp_rest')); ?>';
    const container = document.getElementById('cartt-migration');

    async function checkMigration() {
        try {
            const response = await fetch(apiUrl + 'migration/status', {
                headers: { 'X-WP-Nonce': nonce }
            });
            const result = await response.json();
            renderMigrationUI(result.data);
        } catch (error) {
            container.innerHTML = '<p class="cartt-form-hint">Could not check WooCommerce status.</p>';
        }
    }

    function renderMigrationUI(data) {
        if (!data.has_data) {
            container.innerHTML = `
                <p class="cartt-form-hint" style="margin-bottom:12px;">No WooCommerce data found.</p>
                <button type="button" class="button" id="cartt-generate-demo" style="width:100%;">
                    Generate Demo Products
                </button>
                <div id="cartt-demo-status" style="margin-top:8px;"></div>
            `;
            document.getElementById('cartt-generate-demo').addEventListener('click', generateDemoData);
            return;
        }

        const stats = data.stats;
        container.innerHTML = `
            <p class="cartt-form-hint" style="margin-bottom:12px;">WooCommerce data available:</p>
            <div style="font-size:12px;margin-bottom:12px;">
                <div style="display:flex;justify-content:space-between;padding:4px 0;"><span>Products</span><strong>${stats.products}</strong></div>
                <div style="display:flex;justify-content:space-between;padding:4px 0;"><span>Categories</span><strong>${stats.categories}</strong></div>
                <div style="display:flex;justify-content:space-between;padding:4px 0;"><span>Customers</span><strong>${stats.customers}</strong></div>
                <div style="display:flex;justify-content:space-between;padding:4px 0;"><span>Orders</span><strong>${stats.orders}</strong></div>
            </div>
            <button type="button" class="button button-primary" id="cartt-run-migration" style="width:100%;">
                Import All
            </button>
            <div id="cartt-migration-status" style="margin-top:12px;"></div>
        `;

        document.getElementById('cartt-run-migration').addEventListener('click', runMigration);
    }

    async function runMigration() {
        const btn = document.getElementById('cartt-run-migration');
        const status = document.getElementById('cartt-migration-status');
        
        btn.disabled = true;
        btn.textContent = 'Importing...';
        status.innerHTML = '<p style="font-size:12px;color:#86868b;">This may take a few minutes...</p>';

        try {
            const response = await fetch(apiUrl + 'migration/run', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-WP-Nonce': nonce
                },
                body: JSON.stringify({ products: true, categories: true, customers: true, orders: true })
            });
            const result = await response.json();

            if (result.success) {
                const d = result.data;
                status.innerHTML = `
                    <div style="font-size:12px;color:#2e7d32;background:#e8f5e9;padding:12px;border-radius:4px;">
                        <strong>Import complete!</strong><br>
                        ${d.products} products<br>
                        ${d.categories} categories<br>
                        ${d.customers} customers<br>
                        ${d.orders} orders
                        ${d.errors.length > 0 ? '<br><span style="color:#c62828;">' + d.errors.length + ' errors</span>' : ''}
                    </div>
                `;
                btn.textContent = 'Done';
            } else {
                status.innerHTML = '<p style="font-size:12px;color:#c62828;">Import failed: ' + (result.error || 'Unknown error') + '</p>';
                btn.disabled = false;
                btn.textContent = 'Retry Import';
            }
        } catch (error) {
            status.innerHTML = '<p style="font-size:12px;color:#c62828;">Import failed. Please try again.</p>';
            btn.disabled = false;
            btn.textContent = 'Retry Import';
        }
    }

    checkMigration();

    async function generateDemoData() {
        const btn = document.getElementById('cartt-generate-demo');
        const status = document.getElementById('cartt-demo-status');
        
        btn.disabled = true;
        btn.textContent = 'Generating...';

        try {
            const response = await fetch(apiUrl + 'demo/generate', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-WP-Nonce': nonce
                }
            });
            const result = await response.json();

            if (result.success) {
                status.innerHTML = '<p style="font-size:12px;color:#2e7d32;">Created ' + result.data.count + ' demo products!</p>';
                btn.textContent = 'Done';
            } else {
                status.innerHTML = '<p style="font-size:12px;color:#c62828;">' + (result.error || 'Failed') + '</p>';
                btn.disabled = false;
                btn.textContent = 'Try Again';
            }
        } catch (error) {
            status.innerHTML = '<p style="font-size:12px;color:#c62828;">Failed to generate demo data.</p>';
            btn.disabled = false;
            btn.textContent = 'Try Again';
        }
    }
})();
</script>
            </div>
        </div>
    </form>
</div>
