<?php
/**
 * SEO Settings Admin View
 */

if (!defined('ABSPATH')) {
    exit;
}

use Cartt\Services\SEOService;

// Handle settings save
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['cartt_seo_nonce']) && wp_verify_nonce($_POST['cartt_seo_nonce'], 'cartt_save_seo')) {
    update_option('cartt_seo_enabled', !empty($_POST['seo_enabled']));
    update_option('cartt_seo_title_separator', sanitize_text_field($_POST['title_separator'] ?? '|'));
    update_option('cartt_seo_default_title', sanitize_text_field($_POST['default_title'] ?? ''));
    update_option('cartt_seo_default_description', sanitize_textarea_field($_POST['default_description'] ?? ''));
    update_option('cartt_seo_schema_enabled', !empty($_POST['schema_enabled']));
    update_option('cartt_seo_breadcrumb_enabled', !empty($_POST['breadcrumb_enabled']));
    update_option('cartt_seo_og_enabled', !empty($_POST['og_enabled']));
    update_option('cartt_seo_twitter_enabled', !empty($_POST['twitter_enabled']));
    update_option('cartt_seo_twitter_handle', sanitize_text_field($_POST['twitter_handle'] ?? ''));
    update_option('cartt_seo_sitemap_enabled', !empty($_POST['sitemap_enabled']));
    update_option('cartt_company_logo', esc_url_raw($_POST['company_logo'] ?? ''));
    
    echo '<div class="notice notice-success"><p>SEO settings saved!</p></div>';
}

$settings = [
    'seo_enabled' => get_option('cartt_seo_enabled', true),
    'title_separator' => get_option('cartt_seo_title_separator', '|'),
    'default_title' => get_option('cartt_seo_default_title', ''),
    'default_description' => get_option('cartt_seo_default_description', ''),
    'schema_enabled' => get_option('cartt_seo_schema_enabled', true),
    'breadcrumb_enabled' => get_option('cartt_seo_breadcrumb_enabled', true),
    'og_enabled' => get_option('cartt_seo_og_enabled', true),
    'twitter_enabled' => get_option('cartt_seo_twitter_enabled', true),
    'twitter_handle' => get_option('cartt_seo_twitter_handle', ''),
    'sitemap_enabled' => get_option('cartt_seo_sitemap_enabled', true),
    'company_logo' => get_option('cartt_company_logo', ''),
];
$isEmbedded = isset($GLOBALS['cartt_embedded']) && $GLOBALS['cartt_embedded'];
?>

<?php if (!$isEmbedded): ?><div class="wrap cartt-admin"><?php endif; ?>
    <h2 style="margin-bottom: 15px;">SEO Settings</h2>

    <div class="cartt-tabs">
        <a href="#general" class="cartt-tab active">General</a>
        <a href="#schema" class="cartt-tab">Schema Markup</a>
        <a href="#social" class="cartt-tab">Social Media</a>
        <a href="#sitemap" class="cartt-tab">Sitemap</a>
        <a href="#analysis" class="cartt-tab">Analysis</a>
    </div>

    <form method="post">
        <?php wp_nonce_field('cartt_save_seo', 'cartt_seo_nonce'); ?>

        <!-- General Tab -->
        <div id="general" class="cartt-tab-content active">
            <div class="cartt-card">
                <h2>General SEO Settings</h2>
                <table class="form-table">
                    <tr>
                        <th scope="row">Enable SEO Features</th>
                        <td>
                            <label>
                                <input type="checkbox" name="seo_enabled" value="1" <?php checked($settings['seo_enabled']); ?>>
                                Output meta tags and schema markup
                            </label>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">Title Separator</th>
                        <td>
                            <select name="title_separator">
                                <option value="|" <?php selected($settings['title_separator'], '|'); ?>>| (pipe)</option>
                                <option value="-" <?php selected($settings['title_separator'], '-'); ?>>- (dash)</option>
                                <option value="–" <?php selected($settings['title_separator'], '–'); ?>>– (en dash)</option>
                                <option value="»" <?php selected($settings['title_separator'], '»'); ?>>» (guillemet)</option>
                                <option value="•" <?php selected($settings['title_separator'], '•'); ?>>• (bullet)</option>
                            </select>
                            <p class="description">Used between product name and site name</p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">Default Meta Title</th>
                        <td>
                            <input type="text" name="default_title" value="<?php echo esc_attr($settings['default_title']); ?>" class="large-text">
                            <p class="description">Used for products without custom meta title. Variables: %product_name%, %site_name%, %category%</p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">Default Meta Description</th>
                        <td>
                            <textarea name="default_description" rows="3" class="large-text"><?php echo esc_textarea($settings['default_description']); ?></textarea>
                            <p class="description">Used when product has no custom description. Variables: %product_name%, %price%, %short_description%</p>
                        </td>
                    </tr>
                </table>
            </div>
        </div>

        <!-- Schema Tab -->
        <div id="schema" class="cartt-tab-content" style="display: none;">
            <div class="cartt-card">
                <h2>Schema Markup (Structured Data)</h2>
                <p class="description" style="margin-bottom: 20px;">Schema markup helps search engines understand your products and can enable rich snippets in search results.</p>
                
                <table class="form-table">
                    <tr>
                        <th scope="row">Product Schema</th>
                        <td>
                            <label>
                                <input type="checkbox" name="schema_enabled" value="1" <?php checked($settings['schema_enabled']); ?>>
                                Output Product schema with price, availability, and reviews
                            </label>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">Breadcrumb Schema</th>
                        <td>
                            <label>
                                <input type="checkbox" name="breadcrumb_enabled" value="1" <?php checked($settings['breadcrumb_enabled']); ?>>
                                Output BreadcrumbList schema for navigation
                            </label>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">Organization Logo</th>
                        <td>
                            <input type="url" name="company_logo" value="<?php echo esc_attr($settings['company_logo']); ?>" class="large-text" placeholder="https://...">
                            <p class="description">URL to your company logo for Organization schema</p>
                        </td>
                    </tr>
                </table>

                <h3 style="margin-top: 30px;">Schema Preview</h3>
                <pre style="background: #f5f5f7; padding: 15px; border-radius: 4px; overflow-x: auto; font-size: 12px;">
{
  "@context": "https://schema.org",
  "@type": "Product",
  "name": "Example Product",
  "description": "Product description...",
  "sku": "SKU-12345",
  "offers": {
    "@type": "Offer",
    "price": "29.99",
    "priceCurrency": "USD",
    "availability": "https://schema.org/InStock"
  },
  "aggregateRating": {
    "@type": "AggregateRating",
    "ratingValue": "4.5",
    "reviewCount": "12"
  }
}</pre>
            </div>
        </div>

        <!-- Social Tab -->
        <div id="social" class="cartt-tab-content" style="display: none;">
            <div class="cartt-card">
                <h2>Social Media Tags</h2>
                <table class="form-table">
                    <tr>
                        <th scope="row">Open Graph (Facebook)</th>
                        <td>
                            <label>
                                <input type="checkbox" name="og_enabled" value="1" <?php checked($settings['og_enabled']); ?>>
                                Output Open Graph meta tags for Facebook sharing
                            </label>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">Twitter Cards</th>
                        <td>
                            <label>
                                <input type="checkbox" name="twitter_enabled" value="1" <?php checked($settings['twitter_enabled']); ?>>
                                Output Twitter Card meta tags
                            </label>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">Twitter Handle</th>
                        <td>
                            <input type="text" name="twitter_handle" value="<?php echo esc_attr($settings['twitter_handle']); ?>" class="regular-text" placeholder="@yourstore">
                            <p class="description">Your Twitter/X username for attribution</p>
                        </td>
                    </tr>
                </table>
            </div>
        </div>

        <!-- Sitemap Tab -->
        <div id="sitemap" class="cartt-tab-content" style="display: none;">
            <div class="cartt-card">
                <h2>XML Sitemap</h2>
                <table class="form-table">
                    <tr>
                        <th scope="row">Enable Sitemap</th>
                        <td>
                            <label>
                                <input type="checkbox" name="sitemap_enabled" value="1" <?php checked($settings['sitemap_enabled']); ?>>
                                Generate XML sitemap for products
                            </label>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">Sitemap URL</th>
                        <td>
                            <code><?php echo home_url('/cartt-sitemap.xml'); ?></code>
                            <p class="description">Submit this URL to Google Search Console</p>
                        </td>
                    </tr>
                </table>

                <?php
                global $wpdb;
                $productCount = $wpdb->get_var("SELECT COUNT(*) FROM {$wpdb->prefix}cartt_products WHERE status = 'publish'");
                $categoryCount = $wpdb->get_var("SELECT COUNT(*) FROM {$wpdb->prefix}cartt_categories");
                ?>
                <h3 style="margin-top: 30px;">Sitemap Contents</h3>
                <ul>
                    <li><span class="material-icons-outlined" style="font-size:16px;vertical-align:middle;margin-right:4px;">inventory_2</span> <?php echo $productCount; ?> products</li>
                    <li><span class="material-icons-outlined" style="font-size:16px;vertical-align:middle;margin-right:4px;">folder</span> <?php echo $categoryCount; ?> categories</li>
                </ul>
            </div>
        </div>

        <!-- Analysis Tab -->
        <div id="analysis" class="cartt-tab-content" style="display: none;">
            <div class="cartt-card">
                <h2>SEO Analysis</h2>
                <p>Review the SEO health of your products.</p>
                
                <?php
                global $wpdb;
                $products = $wpdb->get_results(
                    "SELECT p.id, p.name, p.slug, p.description, p.featured_image, s.meta_title, s.meta_description, s.focus_keyword
                     FROM {$wpdb->prefix}cartt_products p
                     LEFT JOIN {$wpdb->prefix}cartt_product_seo s ON p.id = s.product_id
                     WHERE p.status = 'publish'
                     LIMIT 20"
                );
                ?>

                <table class="wp-list-table widefat fixed striped" style="margin-top: 20px;">
                    <thead>
                        <tr>
                            <th>Product</th>
                            <th style="width: 80px;">Score</th>
                            <th>Issues</th>
                            <th style="width: 100px;">Action</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($products as $product): 
                            $analysis = SEOService::analyzeSEO($product->id);
                        ?>
                        <tr>
                            <td><strong><?php echo esc_html($product->name); ?></strong></td>
                            <td>
                                <span class="seo-grade seo-grade-<?php echo strtolower($analysis['grade']); ?>">
                                    <?php echo esc_html($analysis['grade']); ?>
                                </span>
                            </td>
                            <td>
                                <?php if (empty($analysis['issues'])): ?>
                                    <span style="color: #28a745;">✓ All good</span>
                                <?php else: ?>
                                    <ul style="margin: 0; padding-left: 20px;">
                                        <?php foreach (array_slice($analysis['issues'], 0, 3) as $issue): ?>
                                        <li style="font-size: 12px; color: #666;"><?php echo esc_html($issue); ?></li>
                                        <?php endforeach; ?>
                                    </ul>
                                <?php endif; ?>
                            </td>
                            <td>
                                <a href="<?php echo admin_url('admin.php?page=cartt-products&action=edit&id=' . $product->id . '#seo'); ?>" class="button button-small">Edit SEO</a>
                            </td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        </div>

        <p class="submit">
            <input type="submit" class="button button-primary" value="Save SEO Settings">
        </p>
    </form>
<?php if (!$isEmbedded): ?></div><?php endif; ?>

<style>
.cartt-card { background: #fff; border: 1px solid #ccd0d4; border-radius: 4px; padding: 20px; margin-bottom: 20px; }
.cartt-card h2 { margin-top: 0; padding-bottom: 10px; border-bottom: 1px solid #eee; }
.cartt-tabs { margin-bottom: 20px; border-bottom: 1px solid #ccc; }
.cartt-tab { display: inline-block; padding: 10px 20px; text-decoration: none; color: #666; border-bottom: 2px solid transparent; margin-bottom: -1px; }
.cartt-tab.active { color: #1d1d1f; border-bottom-color: #1d1d1f; }
.cartt-tab:hover { color: #1d1d1f; }
.seo-grade { display: inline-block; width: 30px; height: 30px; line-height: 30px; text-align: center; border-radius: 50%; font-weight: bold; color: #fff; }
.seo-grade-a { background: #28a745; }
.seo-grade-b { background: #5cb85c; }
.seo-grade-c { background: #ffc107; color: #333; }
.seo-grade-d { background: #fd7e14; }
.seo-grade-f { background: #dc3545; }
</style>

<script>
jQuery(function($) {
    $('.cartt-tab').on('click', function(e) {
        e.preventDefault();
        const target = $(this).attr('href');
        
        $('.cartt-tab').removeClass('active');
        $(this).addClass('active');
        
        $('.cartt-tab-content').hide();
        $(target).show();
    });
});
</script>
