<?php
/**
 * Reports Admin Page
 */

if (!defined('ABSPATH')) {
    exit;
}

global $wpdb;

// Date range
$range = isset($_GET['range']) ? sanitize_text_field($_GET['range']) : '30days';
$ranges = [
    '7days' => __('Last 7 Days', 'cartt'),
    '30days' => __('Last 30 Days', 'cartt'),
    '90days' => __('Last 90 Days', 'cartt'),
    'year' => __('This Year', 'cartt'),
    'all' => __('All Time', 'cartt'),
];

switch ($range) {
    case '7days':
        $startDate = date('Y-m-d', strtotime('-7 days'));
        break;
    case '30days':
        $startDate = date('Y-m-d', strtotime('-30 days'));
        break;
    case '90days':
        $startDate = date('Y-m-d', strtotime('-90 days'));
        break;
    case 'year':
        $startDate = date('Y-01-01');
        break;
    default:
        $startDate = '2000-01-01';
}
$endDate = date('Y-m-d', strtotime('+1 day'));

// Fetch stats
$ordersTable = $wpdb->prefix . 'cartt_orders';
$itemsTable = $wpdb->prefix . 'cartt_order_items';
$productsTable = $wpdb->prefix . 'cartt_products';
$customersTable = $wpdb->prefix . 'cartt_customers';

// Total revenue
$revenue = $wpdb->get_var($wpdb->prepare(
    "SELECT COALESCE(SUM(total), 0) FROM $ordersTable WHERE status IN ('completed', 'processing') AND created_at >= %s AND created_at < %s",
    $startDate, $endDate
));

// Orders count
$ordersCount = $wpdb->get_var($wpdb->prepare(
    "SELECT COUNT(*) FROM $ordersTable WHERE status IN ('completed', 'processing', 'pending') AND created_at >= %s AND created_at < %s",
    $startDate, $endDate
));

// Average order value
$avgOrder = $ordersCount > 0 ? $revenue / $ordersCount : 0;

// Items sold
$itemsSold = $wpdb->get_var($wpdb->prepare(
    "SELECT COALESCE(SUM(oi.quantity), 0) FROM $itemsTable oi 
     JOIN $ordersTable o ON oi.order_id = o.id 
     WHERE o.status IN ('completed', 'processing') AND o.created_at >= %s AND o.created_at < %s",
    $startDate, $endDate
));

// New customers
$newCustomers = $wpdb->get_var($wpdb->prepare(
    "SELECT COUNT(*) FROM $customersTable WHERE created_at >= %s AND created_at < %s",
    $startDate, $endDate
));

// Revenue by day (for chart)
$revenueByDay = $wpdb->get_results($wpdb->prepare(
    "SELECT DATE(created_at) as date, SUM(total) as total, COUNT(*) as orders
     FROM $ordersTable 
     WHERE status IN ('completed', 'processing') AND created_at >= %s AND created_at < %s
     GROUP BY DATE(created_at) ORDER BY date ASC",
    $startDate, $endDate
));

// Top products
$topProducts = $wpdb->get_results($wpdb->prepare(
    "SELECT oi.product_id, oi.product_name, SUM(oi.quantity) as qty, SUM(oi.total) as revenue
     FROM $itemsTable oi
     JOIN $ordersTable o ON oi.order_id = o.id
     WHERE o.status IN ('completed', 'processing') AND o.created_at >= %s AND o.created_at < %s
     GROUP BY oi.product_id, oi.product_name
     ORDER BY revenue DESC LIMIT 5",
    $startDate, $endDate
));

// Recent orders
$recentOrders = $wpdb->get_results($wpdb->prepare(
    "SELECT * FROM $ordersTable WHERE created_at >= %s ORDER BY created_at DESC LIMIT 10",
    $startDate
));

// Orders by status
$ordersByStatus = $wpdb->get_results($wpdb->prepare(
    "SELECT status, COUNT(*) as count FROM $ordersTable WHERE created_at >= %s AND created_at < %s GROUP BY status",
    $startDate, $endDate
));

// Prepare chart data
$chartLabels = [];
$chartRevenue = [];
$chartOrders = [];

// Fill in all dates
$currentDate = new DateTime($startDate);
$endDateTime = new DateTime($endDate);
$revenueMap = [];
$ordersMap = [];

foreach ($revenueByDay as $row) {
    $revenueMap[$row->date] = (float) $row->total;
    $ordersMap[$row->date] = (int) $row->orders;
}

while ($currentDate < $endDateTime) {
    $dateStr = $currentDate->format('Y-m-d');
    $chartLabels[] = $currentDate->format('M j');
    $chartRevenue[] = $revenueMap[$dateStr] ?? 0;
    $chartOrders[] = $ordersMap[$dateStr] ?? 0;
    $currentDate->modify('+1 day');
}

// Limit chart points for readability
if (count($chartLabels) > 30) {
    $step = ceil(count($chartLabels) / 30);
    $chartLabels = array_values(array_filter($chartLabels, fn($k) => $k % $step === 0, ARRAY_FILTER_USE_KEY));
    $chartRevenue = array_values(array_filter($chartRevenue, fn($k) => $k % $step === 0, ARRAY_FILTER_USE_KEY));
    $chartOrders = array_values(array_filter($chartOrders, fn($k) => $k % $step === 0, ARRAY_FILTER_USE_KEY));
}
$isEmbedded = isset($GLOBALS['cartt_embedded']) && $GLOBALS['cartt_embedded'];
?>

<?php if (!$isEmbedded): ?><div class="wrap cartt-wrap"><?php endif; ?>
    <div class="cartt-header">
        <h2><?php esc_html_e('Reports', 'cartt'); ?></h2>
        <div class="cartt-date-range">
            <?php foreach ($ranges as $key => $label): ?>
            <a href="<?php echo esc_url(admin_url('admin.php?page=cartt-finance&tab=reports&range=' . $key)); ?>" 
               class="<?php echo $range === $key ? 'active' : ''; ?>"><?php echo esc_html($label); ?></a>
            <?php endforeach; ?>
        </div>
    </div>

    <!-- Stats Grid -->
    <div class="cartt-stats-grid">
        <div class="cartt-stat-card">
            <span class="cartt-stat-label"><?php esc_html_e('Revenue', 'cartt'); ?></span>
            <span class="cartt-stat-value">$<?php echo number_format($revenue, 2); ?></span>
        </div>
        <div class="cartt-stat-card">
            <span class="cartt-stat-label"><?php esc_html_e('Orders', 'cartt'); ?></span>
            <span class="cartt-stat-value"><?php echo number_format($ordersCount); ?></span>
        </div>
        <div class="cartt-stat-card">
            <span class="cartt-stat-label"><?php esc_html_e('Avg. Order', 'cartt'); ?></span>
            <span class="cartt-stat-value">$<?php echo number_format($avgOrder, 2); ?></span>
        </div>
        <div class="cartt-stat-card">
            <span class="cartt-stat-label"><?php esc_html_e('Items Sold', 'cartt'); ?></span>
            <span class="cartt-stat-value"><?php echo number_format($itemsSold); ?></span>
        </div>
        <div class="cartt-stat-card">
            <span class="cartt-stat-label"><?php esc_html_e('New Customers', 'cartt'); ?></span>
            <span class="cartt-stat-value"><?php echo number_format($newCustomers); ?></span>
        </div>
    </div>

    <!-- Chart -->
    <div class="cartt-chart-section">
        <h2><?php esc_html_e('Revenue Over Time', 'cartt'); ?></h2>
        <canvas id="revenue-chart" height="100"></canvas>
    </div>

    <div class="cartt-reports-grid">
        <!-- Top Products -->
        <div class="cartt-report-card">
            <h2><?php esc_html_e('Top Products', 'cartt'); ?></h2>
            <?php if (empty($topProducts)): ?>
            <p class="cartt-empty"><?php esc_html_e('No sales data yet.', 'cartt'); ?></p>
            <?php else: ?>
            <table class="cartt-mini-table">
                <thead>
                    <tr>
                        <th><?php esc_html_e('Product', 'cartt'); ?></th>
                        <th><?php esc_html_e('Sold', 'cartt'); ?></th>
                        <th><?php esc_html_e('Revenue', 'cartt'); ?></th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($topProducts as $product): ?>
                    <tr>
                        <td><?php echo esc_html($product->product_name); ?></td>
                        <td><?php echo number_format($product->qty); ?></td>
                        <td>$<?php echo number_format($product->revenue, 2); ?></td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
            <?php endif; ?>
        </div>

        <!-- Orders by Status -->
        <div class="cartt-report-card">
            <h2><?php esc_html_e('Orders by Status', 'cartt'); ?></h2>
            <?php if (empty($ordersByStatus)): ?>
            <p class="cartt-empty"><?php esc_html_e('No orders yet.', 'cartt'); ?></p>
            <?php else: ?>
            <div class="cartt-status-bars">
                <?php 
                $statusLabels = [
                    'pending' => __('Pending', 'cartt'),
                    'processing' => __('Processing', 'cartt'),
                    'completed' => __('Completed', 'cartt'),
                    'cancelled' => __('Cancelled', 'cartt'),
                    'refunded' => __('Refunded', 'cartt'),
                ];
                $statusColors = [
                    'pending' => '#f5a623',
                    'processing' => '#007aff',
                    'completed' => '#34c759',
                    'cancelled' => '#ff3b30',
                    'refunded' => '#8e8e93',
                ];
                $totalOrders = array_sum(array_column($ordersByStatus, 'count'));
                foreach ($ordersByStatus as $status): 
                    $pct = $totalOrders > 0 ? ($status->count / $totalOrders) * 100 : 0;
                ?>
                <div class="cartt-status-bar">
                    <div class="cartt-status-bar-label">
                        <span><?php echo esc_html($statusLabels[$status->status] ?? $status->status); ?></span>
                        <span><?php echo number_format($status->count); ?></span>
                    </div>
                    <div class="cartt-status-bar-track">
                        <div class="cartt-status-bar-fill" style="width:<?php echo $pct; ?>%; background:<?php echo $statusColors[$status->status] ?? '#8e8e93'; ?>;"></div>
                    </div>
                </div>
                <?php endforeach; ?>
            </div>
            <?php endif; ?>
        </div>
    </div>

    <!-- Recent Orders -->
    <div class="cartt-report-card cartt-recent-orders">
        <h2><?php esc_html_e('Recent Orders', 'cartt'); ?></h2>
        <?php if (empty($recentOrders)): ?>
        <p class="cartt-empty"><?php esc_html_e('No orders yet.', 'cartt'); ?></p>
        <?php else: ?>
        <table class="cartt-table">
            <thead>
                <tr>
                    <th><?php esc_html_e('Order', 'cartt'); ?></th>
                    <th><?php esc_html_e('Date', 'cartt'); ?></th>
                    <th><?php esc_html_e('Customer', 'cartt'); ?></th>
                    <th><?php esc_html_e('Total', 'cartt'); ?></th>
                    <th><?php esc_html_e('Status', 'cartt'); ?></th>
                </tr>
            </thead>
            <tbody>
                <?php foreach ($recentOrders as $order): ?>
                <tr>
                    <td>
                        <a href="<?php echo esc_url(admin_url('admin.php?page=cartt-orders&order=' . $order->id)); ?>">
                            #<?php echo esc_html($order->order_number); ?>
                        </a>
                    </td>
                    <td><?php echo esc_html(date_i18n(get_option('date_format'), strtotime($order->created_at))); ?></td>
                    <td><?php echo esc_html($order->customer_email); ?></td>
                    <td>$<?php echo number_format($order->total, 2); ?></td>
                    <td>
                        <span class="cartt-status cartt-status-<?php echo esc_attr($order->status); ?>">
                            <?php echo esc_html(ucfirst($order->status)); ?>
                        </span>
                    </td>
                </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
        <?php endif; ?>
    </div>
<?php if (!$isEmbedded): ?></div><?php endif; ?>

<script src="https://cdn.jsdelivr.net/npm/chart.js@4.4.0/dist/chart.umd.min.js"></script>
<script>
document.addEventListener('DOMContentLoaded', function() {
    const ctx = document.getElementById('revenue-chart');
    if (!ctx) return;
    
    new Chart(ctx, {
        type: 'line',
        data: {
            labels: <?php echo json_encode($chartLabels); ?>,
            datasets: [{
                label: 'Revenue',
                data: <?php echo json_encode($chartRevenue); ?>,
                borderColor: '#1d1d1f',
                backgroundColor: 'rgba(29, 29, 31, 0.1)',
                fill: true,
                tension: 0.3,
                pointRadius: 2,
                pointHoverRadius: 5
            }]
        },
        options: {
            responsive: true,
            maintainAspectRatio: false,
            plugins: {
                legend: { display: false },
                tooltip: {
                    callbacks: {
                        label: function(context) {
                            return '$' + context.raw.toFixed(2);
                        }
                    }
                }
            },
            scales: {
                y: {
                    beginAtZero: true,
                    ticks: {
                        callback: function(value) {
                            return '$' + value;
                        }
                    },
                    grid: { color: '#e8e8ed' }
                },
                x: {
                    grid: { display: false }
                }
            }
        }
    });
});
</script>

<style>
.cartt-date-range { display: flex; gap: 4px; }
.cartt-date-range a { padding: 6px 12px; background: #f5f5f7; border-radius: 6px; text-decoration: none; color: #1d1d1f; font-size: 13px; }
.cartt-date-range a:hover { background: #e8e8ed; }
.cartt-date-range a.active { background: #1d1d1f; color: #fff; }

.cartt-stats-grid { display: grid; grid-template-columns: repeat(5, 1fr); gap: 16px; margin: 20px 0; }
.cartt-stat-card { background: #fff; border: 1px solid #e8e8ed; border-radius: 8px; padding: 20px; text-align: center; }
.cartt-stat-label { display: block; font-size: 12px; color: #86868b; text-transform: uppercase; margin-bottom: 8px; }
.cartt-stat-value { display: block; font-size: 28px; font-weight: 600; color: #1d1d1f; }

.cartt-chart-section { background: #fff; border: 1px solid #e8e8ed; border-radius: 8px; padding: 20px; margin-bottom: 20px; }
.cartt-chart-section h2 { font-size: 14px; font-weight: 600; margin: 0 0 16px; }
.cartt-chart-section canvas { height: 250px !important; }

.cartt-reports-grid { display: grid; grid-template-columns: 1fr 1fr; gap: 20px; margin-bottom: 20px; }
.cartt-report-card { background: #fff; border: 1px solid #e8e8ed; border-radius: 8px; padding: 20px; }
.cartt-report-card h2 { font-size: 14px; font-weight: 600; margin: 0 0 16px; }
.cartt-report-card.cartt-recent-orders { grid-column: 1 / -1; }

.cartt-mini-table { width: 100%; font-size: 13px; }
.cartt-mini-table th { text-align: left; font-weight: 500; color: #86868b; padding: 8px 0; border-bottom: 1px solid #e8e8ed; }
.cartt-mini-table td { padding: 10px 0; border-bottom: 1px solid #e8e8ed; }
.cartt-mini-table td:last-child, .cartt-mini-table th:last-child { text-align: right; }

.cartt-status-bars { display: flex; flex-direction: column; gap: 12px; }
.cartt-status-bar-label { display: flex; justify-content: space-between; font-size: 13px; margin-bottom: 4px; }
.cartt-status-bar-track { height: 8px; background: #e8e8ed; border-radius: 4px; overflow: hidden; }
.cartt-status-bar-fill { height: 100%; border-radius: 4px; transition: width 0.3s; }

.cartt-empty { color: #86868b; font-size: 13px; text-align: center; padding: 20px; }

@media (max-width: 1200px) {
    .cartt-stats-grid { grid-template-columns: repeat(3, 1fr); }
}
@media (max-width: 900px) {
    .cartt-stats-grid { grid-template-columns: repeat(2, 1fr); }
    .cartt-reports-grid { grid-template-columns: 1fr; }
}
</style>
