<?php
/**
 * Products View
 */

if (!defined('ABSPATH')) {
    exit;
}

global $wpdb;

$tab = isset($_GET['tab']) ? sanitize_text_field($_GET['tab']) : 'products';

// If on a sub-tab, show that view instead
if ($tab === 'categories') {
    include CARTT_PLUGIN_DIR . 'admin/views/categories.php';
    return;
}
if ($tab === 'inventory') {
    include CARTT_PLUGIN_DIR . 'admin/views/inventory.php';
    return;
}
if ($tab === 'qa') {
    include CARTT_PLUGIN_DIR . 'admin/views/qa.php';
    return;
}

$pendingQA = \Cartt\Services\ProductQAService::getPendingCount();

$action = isset($_GET['action']) ? sanitize_text_field($_GET['action']) : 'list';
$product_id = isset($_GET['id']) ? intval($_GET['id']) : 0;
$store_type = get_option('cartt_store_type', 'mixed');

// Get product types based on store type
$product_types = [];
switch ($store_type) {
    case 'physical_goods':
        $product_types = ['physical' => __('Physical Product', 'cartt')];
        break;
    case 'digital_goods':
        $product_types = ['digital' => __('Digital Download', 'cartt')];
        break;
    case 'services':
        $product_types = ['service' => __('Service / Booking', 'cartt')];
        break;
    case 'subscriptions':
        $product_types = ['subscription' => __('Subscription', 'cartt')];
        break;
    default:
        $product_types = [
            'physical' => __('Physical Product', 'cartt'),
            'digital' => __('Digital Download', 'cartt'),
            'service' => __('Service / Booking', 'cartt'),
            'subscription' => __('Subscription', 'cartt'),
        ];
}

// Load product if editing
$product = null;
if ($product_id && in_array($action, ['edit', 'view'])) {
    $table = $wpdb->prefix . 'cartt_products';
    $product = $wpdb->get_row($wpdb->prepare("SELECT * FROM $table WHERE id = %d", $product_id));
}

// Handle CSV export
if ($action === 'export') {
    $products_table = $wpdb->prefix . 'cartt_products';
    $all_products = $wpdb->get_results("SELECT * FROM $products_table ORDER BY id ASC", ARRAY_A);
    
    header('Content-Type: text/csv; charset=utf-8');
    header('Content-Disposition: attachment; filename=cartt-products-' . date('Y-m-d') . '.csv');
    
    $output = fopen('php://output', 'w');
    
    // CSV headers
    fputcsv($output, ['ID', 'Name', 'SKU', 'Price', 'Sale Price', 'Stock', 'Type', 'Status', 'Description', 'Short Description', 'Categories', 'Tags', 'Weight', 'Length', 'Width', 'Height']);
    
    foreach ($all_products as $p) {
        fputcsv($output, [
            $p['id'],
            $p['name'],
            $p['sku'],
            $p['price'],
            $p['sale_price'],
            $p['stock_quantity'],
            $p['product_type'],
            $p['status'],
            $p['description'],
            $p['short_description'],
            $p['categories'] ?? '',
            $p['tags'] ?? '',
            $p['weight'] ?? '',
            $p['length'] ?? '',
            $p['width'] ?? '',
            $p['height'] ?? '',
        ]);
    }
    
    fclose($output);
    exit;
}

// Handle CSV import
$import_message = '';
$import_error = '';
if ($action === 'import' && isset($_FILES['csv_file']) && $_FILES['csv_file']['error'] === UPLOAD_ERR_OK) {
    if (wp_verify_nonce($_POST['import_nonce'] ?? '', 'cartt_import_products')) {
        $file = $_FILES['csv_file']['tmp_name'];
        $handle = fopen($file, 'r');
        
        // Skip header row
        $headers = fgetcsv($handle);
        
        $imported = 0;
        $updated = 0;
        $products_table = $wpdb->prefix . 'cartt_products';
        
        while (($row = fgetcsv($handle)) !== false) {
            if (count($row) < 8) continue;
            
            $data = [
                'name' => sanitize_text_field($row[1] ?? ''),
                'sku' => sanitize_text_field($row[2] ?? ''),
                'price' => floatval($row[3] ?? 0),
                'sale_price' => floatval($row[4] ?? 0) ?: null,
                'stock_quantity' => intval($row[5] ?? 0),
                'product_type' => sanitize_text_field($row[6] ?? 'physical'),
                'status' => sanitize_text_field($row[7] ?? 'draft'),
                'description' => sanitize_textarea_field($row[8] ?? ''),
                'short_description' => sanitize_textarea_field($row[9] ?? ''),
            ];
            
            if (empty($data['name'])) continue;
            
            // Check if product exists by ID or SKU
            $existing_id = intval($row[0] ?? 0);
            $existing = null;
            
            if ($existing_id) {
                $existing = $wpdb->get_var($wpdb->prepare("SELECT id FROM $products_table WHERE id = %d", $existing_id));
            } elseif (!empty($data['sku'])) {
                $existing = $wpdb->get_var($wpdb->prepare("SELECT id FROM $products_table WHERE sku = %s", $data['sku']));
            }
            
            if ($existing) {
                $wpdb->update($products_table, $data, ['id' => $existing]);
                $updated++;
            } else {
                $data['slug'] = sanitize_title($data['name']);
                $data['manage_stock'] = 1;
                $data['created_at'] = current_time('mysql');
                $wpdb->insert($products_table, $data);
                $imported++;
            }
        }
        
        fclose($handle);
        $import_message = "Import complete: $imported new products, $updated updated.";
    } else {
        $import_error = 'Invalid security token. Please try again.';
    }
}

?>

<div class="wrap cartt-admin-wrap">
    <?php if ($action === 'list'): ?>
    <!-- Tabs for Products section -->
    <nav class="nav-tab-wrapper" style="margin-bottom: 20px;">
        <a href="?page=cartt-products&tab=products" class="nav-tab <?php echo $tab === 'products' ? 'nav-tab-active' : ''; ?>">
            All Products
        </a>
        <a href="?page=cartt-products&tab=categories" class="nav-tab <?php echo $tab === 'categories' ? 'nav-tab-active' : ''; ?>">
            Categories
        </a>
        <a href="?page=cartt-products&tab=inventory" class="nav-tab <?php echo $tab === 'inventory' ? 'nav-tab-active' : ''; ?>">
            Inventory
        </a>
        <a href="?page=cartt-products&tab=qa" class="nav-tab <?php echo $tab === 'qa' ? 'nav-tab-active' : ''; ?>">
            Q&A
            <?php if ($pendingQA > 0): ?>
            <span class="awaiting-mod"><?php echo $pendingQA; ?></span>
            <?php endif; ?>
        </a>
    </nav>
    <?php endif; ?>
    
    <?php if ($action === 'list'): ?>
        <!-- Products List -->
        <div class="cartt-admin-header" style="display:flex;justify-content:space-between;align-items:center;flex-wrap:wrap;gap:12px;">
            <h1><?php esc_html_e('Products', 'cartt'); ?></h1>
            <div style="display:flex;gap:8px;">
                <a href="<?php echo esc_url(admin_url('admin.php?page=cartt-products&action=import')); ?>" class="button">Import CSV</a>
                <a href="<?php echo esc_url(admin_url('admin.php?page=cartt-products&action=export')); ?>" class="button">Export CSV</a>
                <a href="<?php echo esc_url(admin_url('admin.php?page=cartt-products&action=new')); ?>" class="button button-primary">Add Product</a>
            </div>
        </div>

        <!-- Bulk Actions Bar -->
        <div class="cartt-bulk-bar" id="bulk-actions-bar" style="display:none;background:#f6f7f7;border:1px solid #1d2327;border-radius:4px;padding:12px 16px;margin-bottom:16px;">
            <span id="selected-count">0</span> selected
            <select id="bulk-action-select" style="margin-left:12px;">
                <option value="">Bulk Actions</option>
                <option value="publish">Set Published</option>
                <option value="draft">Set Draft</option>
                <option value="delete">Delete</option>
            </select>
            <button type="button" id="apply-bulk-action" class="button">Apply</button>
            <button type="button" id="clear-selection" class="button" style="margin-left:auto;">Clear Selection</button>
        </div>

        <div class="cartt-table-wrap">
            <table class="cartt-table" id="cartt-products-table">
                <thead>
                    <tr>
                        <th style="width:30px;"><input type="checkbox" id="select-all-products"></th>
                        <th><?php esc_html_e('Product', 'cartt'); ?></th>
                        <th><?php esc_html_e('SKU', 'cartt'); ?></th>
                        <th style="width:100px;"><?php esc_html_e('Price', 'cartt'); ?></th>
                        <th style="width:80px;"><?php esc_html_e('Stock', 'cartt'); ?></th>
                        <th><?php esc_html_e('Type', 'cartt'); ?></th>
                        <th><?php esc_html_e('Status', 'cartt'); ?></th>
                        <th><?php esc_html_e('Actions', 'cartt'); ?></th>
                    </tr>
                </thead>
                <tbody>
                    <?php
                    $table = $wpdb->prefix . 'cartt_products';
                    $products = $wpdb->get_results("SELECT * FROM $table ORDER BY created_at DESC LIMIT 50");
                    $currency = \Cartt\Services\CurrencyService::getSymbol();
                    
                    if ($products):
                        foreach ($products as $p):
                    ?>
                        <tr data-product-id="<?php echo esc_attr($p->id); ?>">
                            <td><input type="checkbox" class="product-checkbox" value="<?php echo esc_attr($p->id); ?>"></td>
                            <td>
                                <strong>
                                    <a href="<?php echo esc_url(admin_url('admin.php?page=cartt-products&action=edit&id=' . $p->id)); ?>">
                                        <?php echo esc_html($p->name); ?>
                                    </a>
                                </strong>
                            </td>
                            <td><?php echo esc_html($p->sku ?: '—'); ?></td>
                            <td>
                                <span class="inline-edit-value" data-field="price" data-id="<?php echo $p->id; ?>">
                                    <?php echo $p->price ? $currency . number_format($p->price, 2) : '—'; ?>
                                </span>
                                <input type="number" step="0.01" class="inline-edit-input" style="display:none;width:80px;" value="<?php echo esc_attr($p->price); ?>">
                            </td>
                            <td>
                                <?php if ($p->manage_stock): ?>
                                <span class="inline-edit-value" data-field="stock" data-id="<?php echo $p->id; ?>">
                                    <?php echo (int)$p->stock_quantity; ?>
                                </span>
                                <input type="number" class="inline-edit-input" style="display:none;width:60px;" value="<?php echo (int)$p->stock_quantity; ?>">
                                <?php else: ?>
                                —
                                <?php endif; ?>
                            </td>
                            <td><?php echo esc_html(ucfirst($p->product_type)); ?></td>
                            <td>
                                <select class="inline-status-select" data-id="<?php echo $p->id; ?>" style="font-size:12px;padding:2px 4px;">
                                    <option value="published" <?php selected($p->status, 'published'); ?>>Published</option>
                                    <option value="draft" <?php selected($p->status, 'draft'); ?>>Draft</option>
                                </select>
                            </td>
                            <td>
                                <a href="<?php echo esc_url(admin_url('admin.php?page=cartt-products&action=edit&id=' . $p->id)); ?>" class="button button-small">
                                    <?php esc_html_e('Edit', 'cartt'); ?>
                                </a>
                                <button type="button" class="button button-small cartt-delete-product" data-product-id="<?php echo esc_attr($p->id); ?>">
                                    <?php esc_html_e('Delete', 'cartt'); ?>
                                </button>
                            </td>
                        </tr>
                    <?php 
                        endforeach;
                    else:
                    ?>
                        <tr>
                            <td colspan="8">
                                <div class="cartt-empty-state" style="text-align:center;padding:60px 20px;">
                                    <div style="margin-bottom:24px;"><span class="material-icons-outlined" style="font-size:64px;color:#1d1d1f;">inventory_2</span></div>
                                    <h3 style="margin:0 0 12px;"><?php esc_html_e('No products yet', 'cartt'); ?></h3>
                                    <p style="color:#666;margin:0 0 20px;"><?php esc_html_e('Add your first product to start selling, or import from a CSV file.', 'cartt'); ?></p>
                                    <a href="<?php echo esc_url(admin_url('admin.php?page=cartt-products&action=new')); ?>" class="button button-primary button-hero">
                                        <?php esc_html_e('Add Your First Product', 'cartt'); ?>
                                    </a>
                                    <br><br>
                                    <a href="<?php echo esc_url(admin_url('admin.php?page=cartt-products&action=import')); ?>">
                                        Or import from CSV
                                    </a>
                                </div>
                            </td>
                        </tr>
                    <?php endif; ?>
                </tbody>
            </table>
        </div>

        <script>
        jQuery(function($) {
            // Select all checkbox
            $('#select-all-products').on('change', function() {
                $('.product-checkbox').prop('checked', this.checked);
                updateBulkBar();
            });
            
            // Individual checkbox
            $(document).on('change', '.product-checkbox', updateBulkBar);
            
            function updateBulkBar() {
                const count = $('.product-checkbox:checked').length;
                if (count > 0) {
                    $('#bulk-actions-bar').show();
                    $('#selected-count').text(count);
                } else {
                    $('#bulk-actions-bar').hide();
                }
            }
            
            // Clear selection
            $('#clear-selection').on('click', function() {
                $('.product-checkbox, #select-all-products').prop('checked', false);
                updateBulkBar();
            });
            
            // Apply bulk action
            $('#apply-bulk-action').on('click', function() {
                const action = $('#bulk-action-select').val();
                if (!action) return;
                
                const ids = $('.product-checkbox:checked').map(function() { return $(this).val(); }).get();
                if (ids.length === 0) return;
                
                if (action === 'delete' && !confirm('Delete ' + ids.length + ' products?')) return;
                
                $.post(ajaxurl, {
                    action: 'cartt_bulk_products',
                    bulk_action: action,
                    ids: ids,
                    nonce: '<?php echo wp_create_nonce('cartt_bulk'); ?>'
                }, function(r) {
                    if (r.success) {
                        carttToast(r.data.message, 'success');
                        setTimeout(() => location.reload(), 1000);
                    } else {
                        carttToast(r.data.message || 'Error', 'error');
                    }
                });
            });
            
            // Inline status change
            $(document).on('change', '.inline-status-select', function() {
                const id = $(this).data('id');
                const status = $(this).val();
                $.post(ajaxurl, {
                    action: 'cartt_update_product_status',
                    id: id,
                    status: status,
                    nonce: '<?php echo wp_create_nonce('cartt_inline'); ?>'
                }, function(r) {
                    carttToast(r.success ? 'Status updated' : 'Error', r.success ? 'success' : 'error');
                });
            });
            
            // Inline editing - click to edit
            $(document).on('click', '.inline-edit-value', function() {
                const $span = $(this);
                const $input = $span.next('.inline-edit-input');
                $span.hide();
                $input.show().focus().select();
            });
            
            // Inline editing - save on blur/enter
            $(document).on('blur keydown', '.inline-edit-input', function(e) {
                if (e.type === 'keydown' && e.key !== 'Enter') return;
                const $input = $(this);
                const $span = $input.prev('.inline-edit-value');
                const field = $span.data('field');
                const id = $span.data('id');
                const value = $input.val();
                
                $input.hide();
                $span.show();
                
                $.post(ajaxurl, {
                    action: 'cartt_inline_edit_product',
                    id: id,
                    field: field,
                    value: value,
                    nonce: '<?php echo wp_create_nonce('cartt_inline'); ?>'
                }, function(r) {
                    if (r.success) {
                        if (field === 'price') {
                            $span.text(value ? '<?php echo $currency; ?>' + parseFloat(value).toFixed(2) : '—');
                        } else {
                            $span.text(value);
                        }
                        carttToast('Saved', 'success');
                    }
                });
            });
        });
        </script>

    <?php elseif ($action === 'import'): ?>
        <!-- Import Products -->
        <div class="cartt-admin-header">
            <h1>Import Products</h1>
            <a href="<?php echo esc_url(admin_url('admin.php?page=cartt-products')); ?>" class="button">Back to Products</a>
        </div>

        <?php if ($import_message): ?>
        <div class="notice notice-success"><p><?php echo esc_html($import_message); ?></p></div>
        <?php endif; ?>
        <?php if ($import_error): ?>
        <div class="notice notice-error"><p><?php echo esc_html($import_error); ?></p></div>
        <?php endif; ?>

        <div class="cartt-card" style="max-width:600px;">
            <h2 style="margin-top:0;">Upload CSV File</h2>
            <form method="post" enctype="multipart/form-data">
                <?php wp_nonce_field('cartt_import_products', 'import_nonce'); ?>
                <p>
                    <input type="file" name="csv_file" accept=".csv" required>
                </p>
                <p class="description">
                    CSV should have columns: ID, Name, SKU, Price, Sale Price, Stock, Type, Status, Description, Short Description
                </p>
                <p>
                    <button type="submit" class="button button-primary">Import Products</button>
                    <a href="<?php echo admin_url('admin.php?page=cartt-products&action=export'); ?>" class="button">Download Template</a>
                </p>
            </form>
        </div>

        <div class="cartt-card" style="max-width:600px;margin-top:20px;">
            <h2 style="margin-top:0;">CSV Format Guide</h2>
            <table class="widefat" style="font-size:13px;">
                <thead>
                    <tr>
                        <th>Column</th>
                        <th>Required</th>
                        <th>Description</th>
                    </tr>
                </thead>
                <tbody>
                    <tr><td>ID</td><td>No</td><td>Leave blank for new products, include to update existing</td></tr>
                    <tr><td>Name</td><td>Yes</td><td>Product name</td></tr>
                    <tr><td>SKU</td><td>No</td><td>Stock keeping unit, used to match existing products</td></tr>
                    <tr><td>Price</td><td>Yes</td><td>Regular price (number only, no currency symbol)</td></tr>
                    <tr><td>Sale Price</td><td>No</td><td>Sale price if on sale</td></tr>
                    <tr><td>Stock</td><td>No</td><td>Stock quantity</td></tr>
                    <tr><td>Type</td><td>No</td><td>physical, digital, service, or subscription</td></tr>
                    <tr><td>Status</td><td>No</td><td>published or draft</td></tr>
                    <tr><td>Description</td><td>No</td><td>Full product description</td></tr>
                    <tr><td>Short Description</td><td>No</td><td>Brief summary</td></tr>
                </tbody>
            </table>
        </div>

    <?php else: ?>
        <!-- Product Form (New / Edit) -->
        <div class="cartt-admin-header">
            <h1>
                <?php echo $product ? esc_html__('Edit Product', 'cartt') : esc_html__('Add Product', 'cartt'); ?>
            </h1>
            <a href="<?php echo esc_url(admin_url('admin.php?page=cartt-products')); ?>" class="button">
                <?php esc_html_e('Back to Products', 'cartt'); ?>
            </a>
        </div>

        <form id="cartt-product-form" data-product-id="<?php echo esc_attr($product_id); ?>">
            <div class="cartt-columns">
                <div class="cartt-main">
                    <div class="cartt-form">
                        <div class="cartt-form-section">
                            <div class="cartt-form-row">
                                <label for="product-name"><?php esc_html_e('Product Name', 'cartt'); ?></label>
                                <input type="text" id="product-name" name="name" value="<?php echo esc_attr($product->name ?? ''); ?>" required style="max-width: 100%;">
                            </div>

                            <div class="cartt-form-row">
                                <label for="product-description"><?php esc_html_e('Description', 'cartt'); ?></label>
                                <textarea id="product-description" name="description" rows="6" style="max-width: 100%;"><?php echo esc_textarea($product->description ?? ''); ?></textarea>
                            </div>

                            <div class="cartt-form-row">
                                <label for="product-short-description"><?php esc_html_e('Short Description', 'cartt'); ?></label>
                                <textarea id="product-short-description" name="short_description" rows="3" style="max-width: 100%;"><?php echo esc_textarea($product->short_description ?? ''); ?></textarea>
                            </div>
                        </div>

                        <div class="cartt-form-section">
                            <div class="cartt-form-row">
                                <label for="product-price"><?php esc_html_e('Price', 'cartt'); ?></label>
                                <input type="number" id="product-price" name="price" value="<?php echo esc_attr($product->price ?? ''); ?>" step="0.01" min="0">
                            </div>

                            <div class="cartt-form-row">
                                <label for="product-sale-price"><?php esc_html_e('Sale Price', 'cartt'); ?></label>
                                <input type="number" id="product-sale-price" name="sale_price" value="<?php echo esc_attr($product->sale_price ?? ''); ?>" step="0.01" min="0">
                                <p class="cartt-form-hint"><?php esc_html_e('Leave empty if not on sale.', 'cartt'); ?></p>
                            </div>
                        </div>

                        <?php if (in_array($store_type, ['physical_goods', 'mixed'])): ?>
                        <div class="cartt-form-section cartt-inventory-section">
                            <h3 class="cartt-form-section-title"><?php esc_html_e('Inventory', 'cartt'); ?></h3>
                            
                            <div class="cartt-form-row">
                                <label for="product-sku"><?php esc_html_e('SKU', 'cartt'); ?></label>
                                <input type="text" id="product-sku" name="sku" value="<?php echo esc_attr($product->sku ?? ''); ?>">
                            </div>

                            <div class="cartt-form-row">
                                <label>
                                    <input type="checkbox" id="product-manage-stock" name="manage_stock" <?php checked($product->manage_stock ?? false); ?>>
                                    <?php esc_html_e('Track stock quantity', 'cartt'); ?>
                                </label>
                            </div>

                            <div class="cartt-form-row">
                                <label for="product-stock"><?php esc_html_e('Stock Quantity', 'cartt'); ?></label>
                                <input type="number" id="product-stock" name="stock_quantity" value="<?php echo esc_attr($product->stock_quantity ?? ''); ?>" min="0">
                            </div>
                        </div>
                        <?php endif; ?>

                        <?php if (in_array($store_type, ['physical_goods', 'mixed'])): ?>
                        <!-- Variations Section -->
                        <div class="cartt-form-section cartt-variations-section">
                            <h3 class="cartt-form-section-title"><?php esc_html_e('Variations', 'cartt'); ?></h3>
                            <p class="cartt-form-hint"><?php esc_html_e('Add size, color, or other options. Each combination creates a variation.', 'cartt'); ?></p>
                            
                            <div class="cartt-attributes-list" id="attributes-list">
                                <?php 
                                $attributes = [];
                                if ($product && $product_id) {
                                    $variationService = new \Cartt\Services\VariationService();
                                    $attributes = $variationService->getProductAttributes($product_id);
                                }
                                foreach ($attributes as $attrName => $attrValues): 
                                ?>
                                <div class="cartt-attribute-row" data-attribute="<?php echo esc_attr($attrName); ?>">
                                    <input type="text" class="cartt-attr-name" value="<?php echo esc_attr($attrName); ?>" placeholder="<?php esc_attr_e('e.g. Size, Color', 'cartt'); ?>">
                                    <input type="text" class="cartt-attr-values" value="<?php echo esc_attr(implode(', ', $attrValues)); ?>" placeholder="<?php esc_attr_e('e.g. Small, Medium, Large', 'cartt'); ?>">
                                    <button type="button" class="button cartt-remove-attr">&times;</button>
                                </div>
                                <?php endforeach; ?>
                            </div>
                            
                            <button type="button" class="button" id="add-attribute">
                                <?php esc_html_e('+ Add Attribute', 'cartt'); ?>
                            </button>
                            
                            <?php if ($product_id): ?>
                            <button type="button" class="button" id="generate-variations" style="margin-left: 8px;">
                                <?php esc_html_e('Generate Variations', 'cartt'); ?>
                            </button>
                            <?php endif; ?>
                            
                            <?php if ($product_id): 
                                $variationService = new \Cartt\Services\VariationService();
                                $variations = $variationService->getAllVariations($product_id);
                                if (!empty($variations)):
                            ?>
                            <div class="cartt-variations-table" style="margin-top: 16px;">
                                <table class="cartt-table cartt-table-compact">
                                    <thead>
                                        <tr>
                                            <th><?php esc_html_e('Variation', 'cartt'); ?></th>
                                            <th><?php esc_html_e('SKU', 'cartt'); ?></th>
                                            <th><?php esc_html_e('Price', 'cartt'); ?></th>
                                            <th><?php esc_html_e('Stock', 'cartt'); ?></th>
                                            <th></th>
                                        </tr>
                                    </thead>
                                    <tbody id="variations-tbody">
                                        <?php foreach ($variations as $v): 
                                            $attrStr = implode(' / ', array_values($v->attributes));
                                        ?>
                                        <tr data-variation-id="<?php echo esc_attr($v->id); ?>">
                                            <td><?php echo esc_html($attrStr); ?></td>
                                            <td><input type="text" class="var-sku" value="<?php echo esc_attr($v->sku); ?>" style="width:80px;"></td>
                                            <td><input type="number" class="var-price" value="<?php echo esc_attr($v->price); ?>" step="0.01" style="width:70px;"></td>
                                            <td><input type="number" class="var-stock" value="<?php echo esc_attr($v->stock_quantity); ?>" style="width:60px;"></td>
                                            <td><button type="button" class="button-link delete-variation" style="color:#c62828;">&times;</button></td>
                                        </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                                <button type="button" class="button" id="save-variations" style="margin-top: 8px;">
                                    <?php esc_html_e('Save Variations', 'cartt'); ?>
                                </button>
                            </div>
                            <?php endif; endif; ?>
                        </div>
                        <?php endif; ?>
                    </div>
                </div>

                <div class="cartt-sidebar">
                    <div class="cartt-sidebar-panel">
                        <div class="cartt-sidebar-panel-header">
                            <?php esc_html_e('Publish', 'cartt'); ?>
                        </div>
                        <div class="cartt-sidebar-panel-content">
                            <div class="cartt-form-row">
                                <label for="product-status"><?php esc_html_e('Status', 'cartt'); ?></label>
                                <select id="product-status" name="status" style="width: 100%;">
                                    <option value="draft" <?php selected($product->status ?? 'draft', 'draft'); ?>><?php esc_html_e('Draft', 'cartt'); ?></option>
                                    <option value="publish" <?php selected($product->status ?? '', 'publish'); ?>><?php esc_html_e('Published', 'cartt'); ?></option>
                                </select>
                            </div>

                            <?php if (count($product_types) > 1): ?>
                            <div class="cartt-form-row">
                                <label for="product-type"><?php esc_html_e('Product Type', 'cartt'); ?></label>
                                <select id="product-type" name="product_type" style="width: 100%;">
                                    <?php foreach ($product_types as $value => $label): ?>
                                        <option value="<?php echo esc_attr($value); ?>" <?php selected($product->product_type ?? 'physical', $value); ?>>
                                            <?php echo esc_html($label); ?>
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                            <?php else: ?>
                                <input type="hidden" id="product-type" name="product_type" value="<?php echo esc_attr(array_key_first($product_types)); ?>">
                            <?php endif; ?>

                            <div class="cartt-actions">
                                <button type="submit" class="button button-primary">
                                    <?php echo $product ? esc_html__('Update Product', 'cartt') : esc_html__('Create Product', 'cartt'); ?>
                                </button>
                            </div>
                        </div>
                    </div>

                    <div class="cartt-sidebar-panel">
                        <div class="cartt-sidebar-panel-header">
                            <?php esc_html_e('Product Image', 'cartt'); ?>
                        </div>
                        <div class="cartt-sidebar-panel-content">
                            <div class="cartt-image-upload" id="cartt-featured-image">
                                <?php if (!empty($product->featured_image)): 
                                    $img_url = wp_get_attachment_image_url($product->featured_image, 'thumbnail');
                                ?>
                                    <img src="<?php echo esc_url($img_url); ?>" class="cartt-image-upload-preview" id="cartt-image-preview">
                                    <input type="hidden" id="product-featured-image" name="featured_image" value="<?php echo esc_attr($product->featured_image); ?>">
                                    <button type="button" class="button button-small" id="cartt-remove-image"><?php esc_html_e('Remove', 'cartt'); ?></button>
                                <?php else: ?>
                                    <div class="cartt-image-upload-text"><?php esc_html_e('Click to add image', 'cartt'); ?></div>
                                    <input type="hidden" id="product-featured-image" name="featured_image" value="">
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>

                    <!-- Product Gallery -->
                    <div class="cartt-sidebar-panel">
                        <div class="cartt-sidebar-panel-header">
                            <?php esc_html_e('Gallery', 'cartt'); ?>
                        </div>
                        <div class="cartt-sidebar-panel-content">
                            <p class="cartt-form-hint" style="margin-top:0;"><?php esc_html_e('Additional product images', 'cartt'); ?></p>
                            <div class="cartt-gallery-images" id="product-gallery">
                                <?php 
                                $gallery = !empty($product->gallery_images) ? json_decode($product->gallery_images, true) : [];
                                foreach ($gallery as $imgId): 
                                    $imgUrl = wp_get_attachment_image_url($imgId, 'thumbnail');
                                    if (!$imgUrl) continue;
                                ?>
                                <div class="cartt-gallery-item" data-id="<?php echo esc_attr($imgId); ?>">
                                    <img src="<?php echo esc_url($imgUrl); ?>">
                                    <button type="button" class="cartt-gallery-remove">&times;</button>
                                </div>
                                <?php endforeach; ?>
                            </div>
                            <input type="hidden" name="gallery_images" id="gallery-images-input" value="<?php echo esc_attr($product->gallery_images ?? '[]'); ?>">
                            <button type="button" class="button button-small" id="add-gallery-images">
                                <?php esc_html_e('Add Images', 'cartt'); ?>
                            </button>
                        </div>
                    </div>

                    <!-- Categories -->
                    <div class="cartt-sidebar-panel">
                        <div class="cartt-sidebar-panel-header">
                            <?php esc_html_e('Categories', 'cartt'); ?>
                        </div>
                        <div class="cartt-sidebar-panel-content">
                            <?php 
                            $allCategories = \Cartt\Services\CategoryService::getAllCategories();
                            $productCategories = $product_id ? \Cartt\Services\CategoryService::getProductCategories($product_id) : [];
                            $productCatIds = array_map(fn($c) => $c->id, $productCategories);
                            ?>
                            <?php if (empty($allCategories)): ?>
                            <p class="cartt-form-hint" style="margin:0;">
                                <?php esc_html_e('No categories yet.', 'cartt'); ?>
                                <a href="<?php echo esc_url(admin_url('admin.php?page=cartt-categories')); ?>"><?php esc_html_e('Create one', 'cartt'); ?></a>
                            </p>
                            <?php else: ?>
                            <div class="cartt-category-checklist">
                                <?php foreach ($allCategories as $cat): ?>
                                <label class="cartt-cat-checkbox">
                                    <input type="checkbox" name="categories[]" value="<?php echo esc_attr($cat->id); ?>" <?php checked(in_array($cat->id, $productCatIds)); ?>>
                                    <?php echo esc_html($cat->name); ?>
                                </label>
                                <?php endforeach; ?>
                            </div>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>
            </div>
        </form>
    <?php endif; ?>
</div>

<style>
/* Variations Section */
.cartt-variations-section { margin-top: 20px; }
.cartt-attributes-list { margin-bottom: 12px; }
.cartt-attribute-row { display: flex; gap: 8px; margin-bottom: 8px; }
.cartt-attr-name { width: 120px; }
.cartt-attr-values { flex: 1; }
.cartt-remove-attr { padding: 4px 10px !important; }
.cartt-table-compact { font-size: 13px; }
.cartt-table-compact input { font-size: 12px; padding: 4px 6px; }
.cartt-table-compact td, .cartt-table-compact th { padding: 8px 6px; }

/* Gallery */
.cartt-gallery-images { display: flex; flex-wrap: wrap; gap: 8px; margin-bottom: 12px; }
.cartt-gallery-item { position: relative; width: 60px; height: 60px; }
.cartt-gallery-item img { width: 100%; height: 100%; object-fit: cover; border-radius: 4px; }
.cartt-gallery-remove { position: absolute; top: -6px; right: -6px; width: 18px; height: 18px; border-radius: 50%; background: #c62828; color: #fff; border: none; font-size: 14px; line-height: 1; cursor: pointer; padding: 0; }

/* Category Checklist */
.cartt-category-checklist { max-height: 200px; overflow-y: auto; border: 1px solid #e8e8ed; border-radius: 4px; padding: 8px; }
.cartt-cat-checkbox { display: block; padding: 4px 0; font-size: 13px; cursor: pointer; }
.cartt-cat-checkbox input { margin-right: 6px; }
</style>

<script>
jQuery(document).ready(function($) {
    // Add attribute row
    $('#add-attribute').on('click', function() {
        var html = '<div class="cartt-attribute-row">' +
            '<input type="text" class="cartt-attr-name" placeholder="<?php esc_attr_e('e.g. Size, Color', 'cartt'); ?>">' +
            '<input type="text" class="cartt-attr-values" placeholder="<?php esc_attr_e('e.g. Small, Medium, Large', 'cartt'); ?>">' +
            '<button type="button" class="button cartt-remove-attr">&times;</button>' +
            '</div>';
        $('#attributes-list').append(html);
    });
    
    // Remove attribute row
    $(document).on('click', '.cartt-remove-attr', function() {
        $(this).closest('.cartt-attribute-row').remove();
    });
    
    // Generate variations
    $('#generate-variations').on('click', function() {
        var productId = $('#cartt-product-form').data('product-id');
        var attributes = {};
        
        $('.cartt-attribute-row').each(function() {
            var name = $(this).find('.cartt-attr-name').val().trim();
            var values = $(this).find('.cartt-attr-values').val().trim();
            if (name && values) {
                attributes[name] = values.split(',').map(function(v) { return v.trim(); }).filter(function(v) { return v; });
            }
        });
        
        if (Object.keys(attributes).length === 0) {
            alert('<?php esc_attr_e('Please add at least one attribute with values.', 'cartt'); ?>');
            return;
        }
        
        var basePrice = $('#product-price').val() || 0;
        
        $.ajax({
            url: ajaxurl,
            method: 'POST',
            data: {
                action: 'cartt_generate_variations',
                product_id: productId,
                attributes: JSON.stringify(attributes),
                base_price: basePrice,
                _wpnonce: '<?php echo wp_create_nonce('cartt_variations'); ?>'
            },
            success: function(response) {
                if (response.success) {
                    location.reload();
                } else {
                    alert(response.data || 'Error generating variations');
                }
            }
        });
    });
    
    // Save variations
    $('#save-variations').on('click', function() {
        var variations = [];
        $('#variations-tbody tr').each(function() {
            variations.push({
                id: $(this).data('variation-id'),
                sku: $(this).find('.var-sku').val(),
                price: $(this).find('.var-price').val(),
                stock_quantity: $(this).find('.var-stock').val()
            });
        });
        
        $.ajax({
            url: ajaxurl,
            method: 'POST',
            data: {
                action: 'cartt_save_variations',
                variations: JSON.stringify(variations),
                _wpnonce: '<?php echo wp_create_nonce('cartt_variations'); ?>'
            },
            success: function(response) {
                if (response.success) {
                    alert('<?php esc_attr_e('Variations saved!', 'cartt'); ?>');
                } else {
                    alert(response.data || 'Error saving variations');
                }
            }
        });
    });
    
    // Delete variation
    $(document).on('click', '.delete-variation', function() {
        if (!confirm('<?php esc_attr_e('Delete this variation?', 'cartt'); ?>')) return;
        
        var $row = $(this).closest('tr');
        var variationId = $row.data('variation-id');
        
        $.ajax({
            url: ajaxurl,
            method: 'POST',
            data: {
                action: 'cartt_delete_variation',
                variation_id: variationId,
                _wpnonce: '<?php echo wp_create_nonce('cartt_variations'); ?>'
            },
            success: function(response) {
                if (response.success) {
                    $row.remove();
                }
            }
        });
    });
    
    // Gallery management
    $('#add-gallery-images').on('click', function() {
        var frame = wp.media({
            title: '<?php esc_attr_e('Select Gallery Images', 'cartt'); ?>',
            button: { text: '<?php esc_attr_e('Add to Gallery', 'cartt'); ?>' },
            multiple: true
        });
        
        frame.on('select', function() {
            var attachments = frame.state().get('selection').toJSON();
            var gallery = JSON.parse($('#gallery-images-input').val() || '[]');
            
            attachments.forEach(function(attachment) {
                if (gallery.indexOf(attachment.id) === -1) {
                    gallery.push(attachment.id);
                    var thumbUrl = attachment.sizes.thumbnail ? attachment.sizes.thumbnail.url : attachment.url;
                    $('#product-gallery').append(
                        '<div class="cartt-gallery-item" data-id="' + attachment.id + '">' +
                        '<img src="' + thumbUrl + '">' +
                        '<button type="button" class="cartt-gallery-remove">&times;</button>' +
                        '</div>'
                    );
                }
            });
            
            $('#gallery-images-input').val(JSON.stringify(gallery));
        });
        
        frame.open();
    });
    
    $(document).on('click', '.cartt-gallery-remove', function(e) {
        e.preventDefault();
        var $item = $(this).closest('.cartt-gallery-item');
        var imgId = $item.data('id');
        var gallery = JSON.parse($('#gallery-images-input').val() || '[]');
        
        gallery = gallery.filter(function(id) { return id !== imgId; });
        $('#gallery-images-input').val(JSON.stringify(gallery));
        $item.remove();
    });
});
</script>
