<?php
/**
 * Loyalty Program Admin View
 */

if (!defined('ABSPATH')) {
    exit;
}

use Cartt\Services\LoyaltyService;
use Cartt\Services\CurrencyService;

$settings = LoyaltyService::getSettings();
$tiers = LoyaltyService::getTiers();
$leaderboard = LoyaltyService::getLeaderboard(20);
$currencySymbol = CurrencyService::getSymbol();
$currencyCode = get_option('cartt_currency', 'USD');

// Handle settings save
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['cartt_loyalty_nonce']) && wp_verify_nonce($_POST['cartt_loyalty_nonce'], 'cartt_save_loyalty')) {
    update_option('cartt_loyalty_enabled', !empty($_POST['enabled']));
    update_option('cartt_loyalty_points_per_currency', floatval($_POST['points_per_currency'] ?? 1));
    update_option('cartt_loyalty_points_value', floatval($_POST['points_value'] ?? 0.01));
    update_option('cartt_loyalty_min_redeem', intval($_POST['min_redeem'] ?? 100));
    update_option('cartt_loyalty_review_points', intval($_POST['review_points'] ?? 50));
    update_option('cartt_loyalty_signup_points', intval($_POST['signup_points'] ?? 100));
    update_option('cartt_loyalty_referral_points', intval($_POST['referral_points'] ?? 200));
    update_option('cartt_loyalty_expiry_days', intval($_POST['expiry_days'] ?? 365));
    
    echo '<div class="notice notice-success"><p>Settings saved!</p></div>';
    $settings = LoyaltyService::getSettings();
}

// Support legacy option name
$pointsPerCurrency = $settings['points_per_currency'] ?? $settings['points_per_dollar'] ?? 1;

// Handle tier settings save
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['cartt_tiers_nonce']) && wp_verify_nonce($_POST['cartt_tiers_nonce'], 'cartt_save_tiers')) {
    $newTiers = [];
    foreach (['bronze', 'silver', 'gold', 'platinum'] as $tierKey) {
        $newTiers[$tierKey] = [
            'name' => ucfirst($tierKey),
            'min_points' => intval($_POST["tier_{$tierKey}_min"] ?? 0),
            'multiplier' => floatval($_POST["tier_{$tierKey}_multiplier"] ?? 1)
        ];
    }
    update_option('cartt_loyalty_tiers', $newTiers);
    echo '<div class="notice notice-success"><p>Tier settings saved!</p></div>';
    $tiers = $newTiers;
}
?>

<div class="wrap cartt-admin">
    <h1>Loyalty Program</h1>

    <div class="cartt-tabs">
        <a href="#settings" class="cartt-tab active">Settings</a>
        <a href="#leaderboard" class="cartt-tab">Leaderboard</a>
        <a href="#tiers" class="cartt-tab">Tiers</a>
    </div>

    <!-- Settings Tab -->
    <div id="settings" class="cartt-tab-content active">
        <form method="post">
            <?php wp_nonce_field('cartt_save_loyalty', 'cartt_loyalty_nonce'); ?>
            
            <div class="cartt-card">
                <h2>General Settings</h2>
                <table class="form-table">
                    <tr>
                        <th scope="row">Enable Loyalty Program</th>
                        <td>
                            <label>
                                <input type="checkbox" name="enabled" value="1" <?php checked($settings['enabled']); ?>>
                                Allow customers to earn and redeem points
                            </label>
                        </td>
                    </tr>
                </table>
            </div>

            <div class="cartt-card" style="margin-top: 20px;">
                <h2>Earning Points</h2>
                <table class="form-table">
                    <tr>
                        <th scope="row">Points per <?php echo esc_html($currencySymbol); ?>1 Spent</th>
                        <td>
                            <input type="number" name="points_per_currency" value="<?php echo esc_attr($pointsPerCurrency); ?>" step="0.1" min="0" style="width: 80px;">
                            <p class="description">Base points earned per <?php echo esc_html($currencySymbol); ?>1 spent (before tier multiplier)</p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">Points for Signup</th>
                        <td>
                            <input type="number" name="signup_points" value="<?php echo esc_attr($settings['points_for_signup']); ?>" min="0" style="width: 80px;">
                            <p class="description">Welcome bonus for new customers</p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">Points for Review</th>
                        <td>
                            <input type="number" name="review_points" value="<?php echo esc_attr($settings['points_for_review']); ?>" min="0" style="width: 80px;">
                            <p class="description">Bonus for leaving a product review</p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">Points for Referral</th>
                        <td>
                            <input type="number" name="referral_points" value="<?php echo esc_attr($settings['points_for_referral']); ?>" min="0" style="width: 80px;">
                            <p class="description">Bonus when a referred friend makes a purchase</p>
                        </td>
                    </tr>
                </table>
            </div>

            <div class="cartt-card" style="margin-top: 20px;">
                <h2>Redeeming Points</h2>
                <table class="form-table">
                    <tr>
                        <th scope="row">Point Value</th>
                        <td>
                            <span>$</span>
                            <input type="number" name="points_value" value="<?php echo esc_attr($settings['points_value']); ?>" step="0.001" min="0" style="width: 100px;">
                            <span>per point</span>
                            <p class="description">Example: 0.01 = 100 points = $1.00 discount</p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">Minimum Points to Redeem</th>
                        <td>
                            <input type="number" name="min_redeem" value="<?php echo esc_attr($settings['min_redeem']); ?>" min="0" style="width: 80px;">
                            <p class="description">Minimum points required before redemption</p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">Points Expiry</th>
                        <td>
                            <input type="number" name="expiry_days" value="<?php echo esc_attr($settings['points_expiry_days']); ?>" min="0" style="width: 80px;"> days
                            <p class="description">0 = points never expire</p>
                        </td>
                    </tr>
                </table>
            </div>

            <p class="submit">
                <input type="submit" class="button button-primary" value="Save Settings">
            </p>
        </form>
    </div>

    <!-- Leaderboard Tab -->
    <div id="leaderboard" class="cartt-tab-content" style="display: none;">
        <div class="cartt-card">
            <h2>Top Customers</h2>
            <table class="wp-list-table widefat fixed striped">
                <thead>
                    <tr>
                        <th style="width: 50px;">Rank</th>
                        <th>Customer</th>
                        <th>Current Points</th>
                        <th>Lifetime Points</th>
                        <th>Tier</th>
                    </tr>
                </thead>
                <tbody>
                    <?php if (empty($leaderboard)): ?>
                    <tr>
                        <td colspan="5" style="text-align: center; padding: 40px;">No loyalty data yet.</td>
                    </tr>
                    <?php else: ?>
                    <?php foreach ($leaderboard as $i => $customer): ?>
                    <tr>
                        <td>
                            <?php if ($i < 3): ?>
                                <span style="font-size: 20px;"><?php echo ['🥇', '🥈', '🥉'][$i]; ?></span>
                            <?php else: ?>
                                <?php echo $i + 1; ?>
                            <?php endif; ?>
                        </td>
                        <td><?php echo esc_html($customer->first_name . ' ' . $customer->last_name); ?></td>
                        <td><strong><?php echo number_format($customer->points); ?></strong></td>
                        <td><?php echo number_format($customer->lifetime_points); ?></td>
                        <td>
                            <span class="cartt-tier cartt-tier-<?php echo esc_attr($customer->tier); ?>">
                                <?php echo esc_html(ucfirst($customer->tier)); ?>
                            </span>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                    <?php endif; ?>
                </tbody>
            </table>
        </div>
    </div>

    <!-- Tiers Tab -->
    <div id="tiers" class="cartt-tab-content" style="display: none;">
        <form method="post" id="tiers-form">
            <?php wp_nonce_field('cartt_save_tiers', 'cartt_tiers_nonce'); ?>
            <input type="hidden" name="save_tiers" value="1">
            
            <div class="cartt-card">
                <h2>Loyalty Tiers</h2>
                <p class="description">Customers automatically advance through tiers based on lifetime points earned. Customize the requirements and rewards for each tier.</p>
                
                <div class="cartt-tiers-grid" style="display: grid; grid-template-columns: repeat(4, 1fr); gap: 20px; margin-top: 20px;">
                    <?php foreach ($tiers as $key => $tier): ?>
                    <div class="cartt-tier-card cartt-tier-card-<?php echo esc_attr($key); ?>">
                        <h3><?php echo esc_html($tier['name']); ?></h3>
                        <div style="margin-top: 15px;">
                            <label style="font-size: 12px; color: #666;">Min Points Required</label>
                            <input type="number" name="tier_<?php echo esc_attr($key); ?>_min" 
                                   value="<?php echo esc_attr($tier['min_points']); ?>" 
                                   min="0" style="width: 100%; margin-top: 4px;"
                                   <?php echo $key === 'bronze' ? 'readonly' : ''; ?>>
                        </div>
                        <div style="margin-top: 10px;">
                            <label style="font-size: 12px; color: #666;">Points Multiplier</label>
                            <input type="number" name="tier_<?php echo esc_attr($key); ?>_multiplier" 
                                   value="<?php echo esc_attr($tier['multiplier']); ?>" 
                                   step="0.1" min="1" style="width: 100%; margin-top: 4px;">
                        </div>
                    </div>
                    <?php endforeach; ?>
                </div>
                
                <div style="margin-top: 20px;">
                    <button type="submit" class="button button-primary">Save Tier Settings</button>
                </div>
            </div>
        </form>
    </div>
</div>

<style>
.cartt-card { background: #fff; border: 1px solid #ccd0d4; border-radius: 4px; padding: 20px; }
.cartt-card h2 { margin-top: 0; padding-bottom: 10px; border-bottom: 1px solid #eee; }
.cartt-tabs { margin-bottom: 20px; border-bottom: 1px solid #ccc; }
.cartt-tab { display: inline-block; padding: 10px 20px; text-decoration: none; color: #666; border-bottom: 2px solid transparent; margin-bottom: -1px; }
.cartt-tab.active { color: #1d1d1f; border-bottom-color: #1d1d1f; }
.cartt-tab:hover { color: #1d1d1f; }
.cartt-tier { display: inline-block; padding: 3px 10px; border-radius: 3px; font-size: 12px; font-weight: 500; }
.cartt-tier-bronze { background: #cd7f32; color: #fff; }
.cartt-tier-silver { background: #c0c0c0; color: #333; }
.cartt-tier-gold { background: #ffd700; color: #333; }
.cartt-tier-platinum { background: #e5e4e2; color: #333; }
.cartt-tier-card { background: #f9f9f9; border-radius: 8px; padding: 20px; text-align: center; }
.cartt-tier-card h3 { margin: 0 0 10px; }
.cartt-tier-card .tier-requirement { font-size: 14px; color: #666; }
.cartt-tier-card .tier-multiplier { font-size: 24px; font-weight: bold; margin-top: 10px; }
.cartt-tier-card-bronze { border: 2px solid #cd7f32; }
.cartt-tier-card-silver { border: 2px solid #c0c0c0; }
.cartt-tier-card-gold { border: 2px solid #ffd700; }
.cartt-tier-card-platinum { border: 2px solid #e5e4e2; }
</style>

<script>
jQuery(function($) {
    $('.cartt-tab').on('click', function(e) {
        e.preventDefault();
        const target = $(this).attr('href');
        
        $('.cartt-tab').removeClass('active');
        $(this).addClass('active');
        
        $('.cartt-tab-content').hide();
        $(target).show();
    });
});
</script>
