<?php
/**
 * Inventory Management Admin View
 */

if (!defined('ABSPATH')) {
    exit;
}

use Cartt\Services\InventoryService;

$report = InventoryService::getStockReport();
$lowStock = InventoryService::getLowStockProducts();
$outOfStock = InventoryService::getOutOfStockProducts();
$settings = InventoryService::getSettings();

// Handle settings save
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['cartt_inventory_nonce']) && wp_verify_nonce($_POST['cartt_inventory_nonce'], 'cartt_save_inventory')) {
    update_option('cartt_manage_stock', !empty($_POST['manage_stock']));
    update_option('cartt_low_stock_threshold', intval($_POST['low_stock_threshold'] ?? 5));
    update_option('cartt_out_of_stock_visibility', sanitize_text_field($_POST['out_of_stock_visibility'] ?? 'show'));
    update_option('cartt_backorders_allowed', !empty($_POST['backorders_allowed']));
    update_option('cartt_notify_low_stock', !empty($_POST['notify_low_stock']));
    update_option('cartt_notify_out_of_stock', !empty($_POST['notify_out_of_stock']));
    update_option('cartt_stock_display', sanitize_text_field($_POST['stock_display'] ?? 'always'));
    
    echo '<div class="notice notice-success"><p>Settings saved!</p></div>';
    $settings = InventoryService::getSettings();
}
?>

<?php
$pendingQA = \Cartt\Services\ProductQAService::getPendingCount();
?>

<div class="wrap cartt-admin">
    <!-- Tabs for Products section -->
    <nav class="nav-tab-wrapper" style="margin-bottom: 20px;">
        <a href="?page=cartt-products&tab=products" class="nav-tab">
            All Products
        </a>
        <a href="?page=cartt-products&tab=categories" class="nav-tab">
            Categories
        </a>
        <a href="?page=cartt-products&tab=inventory" class="nav-tab nav-tab-active">
            Inventory
        </a>
        <a href="?page=cartt-products&tab=qa" class="nav-tab">
            Q&A
            <?php if ($pendingQA > 0): ?>
            <span class="awaiting-mod"><?php echo $pendingQA; ?></span>
            <?php endif; ?>
        </a>
    </nav>

    <h1>Inventory Management</h1>

    <div class="cartt-tabs">
        <a href="#overview" class="cartt-tab active">Overview</a>
        <a href="#low-stock" class="cartt-tab">Low Stock <span class="count">(<?php echo count($lowStock); ?>)</span></a>
        <a href="#out-of-stock" class="cartt-tab">Out of Stock <span class="count">(<?php echo count($outOfStock); ?>)</span></a>
        <a href="#settings" class="cartt-tab">Settings</a>
    </div>

    <!-- Overview Tab -->
    <div id="overview" class="cartt-tab-content active">
        <div class="cartt-stats-row" style="display: grid; grid-template-columns: repeat(5, 1fr); gap: 20px; margin-bottom: 30px;">
            <div class="cartt-stat-card">
                <div class="stat-value"><?php echo number_format($report['total_products']); ?></div>
                <div class="stat-label">Total Products</div>
            </div>
            <div class="cartt-stat-card">
                <div class="stat-value" style="color: #28a745;"><?php echo number_format($report['in_stock']); ?></div>
                <div class="stat-label">In Stock</div>
            </div>
            <div class="cartt-stat-card">
                <div class="stat-value" style="color: #ffc107;"><?php echo number_format($report['low_stock']); ?></div>
                <div class="stat-label">Low Stock</div>
            </div>
            <div class="cartt-stat-card">
                <div class="stat-value" style="color: #dc3545;"><?php echo number_format($report['out_of_stock']); ?></div>
                <div class="stat-label">Out of Stock</div>
            </div>
            <div class="cartt-stat-card">
                <div class="stat-value" style="color: #6c757d;"><?php echo number_format($report['backorder']); ?></div>
                <div class="stat-label">On Backorder</div>
            </div>
        </div>

        <div class="cartt-card">
            <h2>Quick Actions</h2>
            <p>
                <a href="<?php echo admin_url('admin.php?page=cartt-products&stock_status=low'); ?>" class="button">View Low Stock Products</a>
                <a href="<?php echo admin_url('admin.php?page=cartt-products&stock_status=out'); ?>" class="button">View Out of Stock Products</a>
                <button type="button" class="button" id="export-inventory-btn">Export Inventory CSV</button>
            </p>
        </div>
    </div>

    <!-- Low Stock Tab -->
    <div id="low-stock" class="cartt-tab-content" style="display: none;">
        <div class="cartt-card">
            <h2>Low Stock Alert (<?php echo $settings['low_stock_threshold']; ?> or fewer)</h2>
            <?php if (empty($lowStock)): ?>
            <p style="padding: 20px; text-align: center; color: #666;">No low stock items</p>
            <?php else: ?>
            <table class="wp-list-table widefat fixed striped">
                <thead>
                    <tr>
                        <th>Product</th>
                        <th>SKU</th>
                        <th style="width: 100px;">Stock</th>
                        <th style="width: 150px;">Actions</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($lowStock as $product): ?>
                    <tr>
                        <td>
                            <strong><a href="<?php echo admin_url('admin.php?page=cartt-products&action=edit&id=' . $product->id); ?>"><?php echo esc_html($product->name); ?></a></strong>
                        </td>
                        <td><?php echo esc_html($product->sku ?: '—'); ?></td>
                        <td>
                            <span style="color: #ffc107; font-weight: bold;"><?php echo intval($product->stock_quantity); ?></span>
                        </td>
                        <td>
                            <button type="button" class="button button-small quick-stock-btn" data-id="<?php echo $product->id; ?>" data-stock="<?php echo $product->stock_quantity; ?>">Update Stock</button>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
            <?php endif; ?>
        </div>
    </div>

    <!-- Out of Stock Tab -->
    <div id="out-of-stock" class="cartt-tab-content" style="display: none;">
        <div class="cartt-card">
            <h2>Out of Stock Products</h2>
            <?php if (empty($outOfStock)): ?>
            <p style="padding: 20px; text-align: center; color: #666;">No out of stock items</p>
            <?php else: ?>
            <table class="wp-list-table widefat fixed striped">
                <thead>
                    <tr>
                        <th>Product</th>
                        <th>SKU</th>
                        <th style="width: 100px;">Status</th>
                        <th style="width: 150px;">Actions</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($outOfStock as $product): ?>
                    <tr>
                        <td>
                            <strong><a href="<?php echo admin_url('admin.php?page=cartt-products&action=edit&id=' . $product->id); ?>"><?php echo esc_html($product->name); ?></a></strong>
                        </td>
                        <td><?php echo esc_html($product->sku ?: '—'); ?></td>
                        <td>
                            <span style="color: #dc3545; font-weight: bold;">Out of Stock</span>
                        </td>
                        <td>
                            <button type="button" class="button button-small quick-stock-btn" data-id="<?php echo $product->id; ?>" data-stock="0">Restock</button>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
            <?php endif; ?>
        </div>
    </div>

    <!-- Settings Tab -->
    <div id="settings" class="cartt-tab-content" style="display: none;">
        <form method="post">
            <?php wp_nonce_field('cartt_save_inventory', 'cartt_inventory_nonce'); ?>
            
            <div class="cartt-card">
                <h2>Stock Management</h2>
                <table class="form-table">
                    <tr>
                        <th scope="row">Enable Stock Management</th>
                        <td>
                            <label>
                                <input type="checkbox" name="manage_stock" value="1" <?php checked($settings['manage_stock']); ?>>
                                Track inventory levels for products
                            </label>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">Low Stock Threshold</th>
                        <td>
                            <input type="number" name="low_stock_threshold" value="<?php echo esc_attr($settings['low_stock_threshold']); ?>" min="0" style="width: 80px;">
                            <p class="description">Products with stock at or below this level will trigger low stock alerts</p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">Out of Stock Visibility</th>
                        <td>
                            <select name="out_of_stock_visibility">
                                <option value="show" <?php selected($settings['out_of_stock_visibility'], 'show'); ?>>Show products</option>
                                <option value="hide" <?php selected($settings['out_of_stock_visibility'], 'hide'); ?>>Hide products</option>
                            </select>
                            <p class="description">Whether to show out of stock products in catalog</p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">Allow Backorders</th>
                        <td>
                            <label>
                                <input type="checkbox" name="backorders_allowed" value="1" <?php checked($settings['backorders_allowed']); ?>>
                                Allow customers to order products that are out of stock
                            </label>
                        </td>
                    </tr>
                </table>
            </div>

            <div class="cartt-card" style="margin-top: 20px;">
                <h2>Notifications</h2>
                <table class="form-table">
                    <tr>
                        <th scope="row">Low Stock Email</th>
                        <td>
                            <label>
                                <input type="checkbox" name="notify_low_stock" value="1" <?php checked($settings['notify_low_stock']); ?>>
                                Send email when product reaches low stock threshold
                            </label>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">Out of Stock Email</th>
                        <td>
                            <label>
                                <input type="checkbox" name="notify_out_of_stock" value="1" <?php checked($settings['notify_out_of_stock']); ?>>
                                Send email when product goes out of stock
                            </label>
                        </td>
                    </tr>
                </table>
            </div>

            <div class="cartt-card" style="margin-top: 20px;">
                <h2>Display Settings</h2>
                <table class="form-table">
                    <tr>
                        <th scope="row">Stock Display</th>
                        <td>
                            <select name="stock_display">
                                <option value="always" <?php selected($settings['stock_display'], 'always'); ?>>Always show stock quantity</option>
                                <option value="low" <?php selected($settings['stock_display'], 'low'); ?>>Only show when low stock</option>
                                <option value="never" <?php selected($settings['stock_display'], 'never'); ?>>Never show stock quantity</option>
                            </select>
                        </td>
                    </tr>
                </table>
            </div>

            <p class="submit">
                <input type="submit" class="button button-primary" value="Save Settings">
            </p>
        </form>
    </div>
</div>

<!-- Quick Stock Modal -->
<div id="quick-stock-modal" class="cartt-modal" style="display: none;">
    <div class="cartt-modal-content" style="max-width: 400px;">
        <div class="cartt-modal-header">
            <h2>Update Stock</h2>
            <button type="button" class="cartt-modal-close">&times;</button>
        </div>
        <div class="cartt-modal-body">
            <input type="hidden" id="stock-product-id">
            <p>
                <label for="stock-quantity"><strong>New Stock Quantity:</strong></label><br>
                <input type="number" id="stock-quantity" min="0" style="width: 100%; margin-top: 5px;">
            </p>
        </div>
        <div class="cartt-modal-footer">
            <button type="button" class="button" onclick="document.getElementById('quick-stock-modal').style.display='none'">Cancel</button>
            <button type="button" class="button button-primary" id="save-stock-btn">Update</button>
        </div>
    </div>
</div>

<style>
.cartt-card { background: #fff; border: 1px solid #ccd0d4; border-radius: 4px; padding: 20px; }
.cartt-card h2 { margin-top: 0; padding-bottom: 10px; border-bottom: 1px solid #eee; }
.cartt-stat-card { background: #fff; border: 1px solid #ccd0d4; border-radius: 8px; padding: 20px; text-align: center; }
.cartt-stat-card .stat-value { font-size: 32px; font-weight: 600; color: #1d1d1f; }
.cartt-stat-card .stat-label { font-size: 13px; color: #666; margin-top: 5px; }
.cartt-tabs { margin-bottom: 20px; border-bottom: 1px solid #ccc; }
.cartt-tab { display: inline-block; padding: 10px 20px; text-decoration: none; color: #666; border-bottom: 2px solid transparent; margin-bottom: -1px; }
.cartt-tab.active { color: #1d1d1f; border-bottom-color: #1d1d1f; }
.cartt-tab:hover { color: #1d1d1f; }
.cartt-tab .count { background: #eee; padding: 2px 6px; border-radius: 10px; font-size: 11px; }
.cartt-modal { position: fixed; top: 0; left: 0; right: 0; bottom: 0; background: rgba(0,0,0,0.6); z-index: 100000; display: flex; align-items: center; justify-content: center; }
.cartt-modal-content { background: #fff; border-radius: 4px; }
.cartt-modal-header { padding: 15px 20px; border-bottom: 1px solid #ddd; display: flex; justify-content: space-between; align-items: center; }
.cartt-modal-header h2 { margin: 0; }
.cartt-modal-close { background: none; border: none; font-size: 24px; cursor: pointer; }
.cartt-modal-body { padding: 20px; }
.cartt-modal-footer { padding: 15px 20px; border-top: 1px solid #ddd; text-align: right; }
</style>

<script>
jQuery(function($) {
    const nonce = '<?php echo wp_create_nonce('cartt_inventory'); ?>';

    // Tabs
    $('.cartt-tab').on('click', function(e) {
        e.preventDefault();
        const target = $(this).attr('href');
        
        $('.cartt-tab').removeClass('active');
        $(this).addClass('active');
        
        $('.cartt-tab-content').hide();
        $(target).show();
    });

    // Quick stock modal
    $('.quick-stock-btn').on('click', function() {
        const id = $(this).data('id');
        const stock = $(this).data('stock');
        
        $('#stock-product-id').val(id);
        $('#stock-quantity').val(stock);
        $('#quick-stock-modal').show();
    });

    $('.cartt-modal-close').on('click', function() {
        $(this).closest('.cartt-modal').hide();
    });

    $('#save-stock-btn').on('click', function() {
        const id = $('#stock-product-id').val();
        const quantity = $('#stock-quantity').val();

        $.post(ajaxurl, {
            action: 'cartt_update_stock',
            nonce: nonce,
            product_id: id,
            quantity: quantity
        }, function(response) {
            if (response.success) {
                location.reload();
            } else {
                alert('Error: ' + response.data.message);
            }
        });
    });

    // Export inventory
    $('#export-inventory-btn').on('click', function() {
        window.location.href = ajaxurl + '?action=cartt_export_inventory&nonce=' + nonce;
    });
});
</script>
