<?php
/**
 * Gift Cards Admin View
 */

if (!defined('ABSPATH')) {
    exit;
}

use Cartt\Services\GiftCardService;
use Cartt\Services\CurrencyService;

$action = isset($_GET['action']) ? sanitize_text_field($_GET['action']) : 'list';
$giftCards = GiftCardService::getAll();
$designs = GiftCardService::getDesigns();
?>

<div class="wrap cartt-admin">
    <h1 class="wp-heading-inline">Gift Cards</h1>
    <button type="button" class="page-title-action" id="create-gift-card-btn">Create Gift Card</button>
    <hr class="wp-header-end">

    <!-- Stats -->
    <div class="cartt-stats-row" style="display: grid; grid-template-columns: repeat(4, 1fr); gap: 20px; margin-bottom: 30px;">
        <?php
        $totalActive = 0;
        $totalBalance = 0;
        $totalRedeemed = 0;
        foreach ($giftCards as $gc) {
            if ($gc->status === 'active') {
                $totalActive++;
                $totalBalance += $gc->current_balance;
            }
            $totalRedeemed += ($gc->initial_balance - $gc->current_balance);
        }
        ?>
        <div class="cartt-stat-card">
            <div class="stat-value"><?php echo count($giftCards); ?></div>
            <div class="stat-label">Total Gift Cards</div>
        </div>
        <div class="cartt-stat-card">
            <div class="stat-value"><?php echo $totalActive; ?></div>
            <div class="stat-label">Active Cards</div>
        </div>
        <div class="cartt-stat-card">
            <div class="stat-value"><?php echo esc_html(CurrencyService::formatPrice($totalBalance)); ?></div>
            <div class="stat-label">Outstanding Balance</div>
        </div>
        <div class="cartt-stat-card">
            <div class="stat-value"><?php echo esc_html(CurrencyService::formatPrice($totalRedeemed)); ?></div>
            <div class="stat-label">Total Redeemed</div>
        </div>
    </div>

    <!-- Gift Cards Table -->
    <div class="cartt-card">
        <table class="wp-list-table widefat fixed striped">
            <thead>
                <tr>
                    <th>Code</th>
                    <th>Initial</th>
                    <th>Balance</th>
                    <th>Recipient</th>
                    <th>Status</th>
                    <th>Created</th>
                    <th>Expires</th>
                    <th>Actions</th>
                </tr>
            </thead>
            <tbody>
                <?php if (empty($giftCards)): ?>
                <tr>
                    <td colspan="8" style="text-align: center; padding: 40px;">No gift cards yet.</td>
                </tr>
                <?php else: ?>
                <?php foreach ($giftCards as $gc): ?>
                <tr data-id="<?php echo esc_attr($gc->id); ?>">
                    <td><code style="font-size: 13px;"><?php echo esc_html($gc->code); ?></code></td>
                    <td><?php echo esc_html(CurrencyService::formatPrice($gc->initial_balance)); ?></td>
                    <td><strong><?php echo esc_html(CurrencyService::formatPrice($gc->current_balance)); ?></strong></td>
                    <td>
                        <?php if ($gc->recipient_email): ?>
                            <?php echo esc_html($gc->recipient_name ?: $gc->recipient_email); ?>
                            <?php if (!$gc->is_delivered): ?>
                                <span class="dashicons dashicons-email-alt" title="Not delivered yet" style="color: #1d2327;"></span>
                            <?php endif; ?>
                        <?php else: ?>
                            <span style="color: #999;">—</span>
                        <?php endif; ?>
                    </td>
                    <td>
                        <span class="cartt-status cartt-status-<?php echo esc_attr($gc->status); ?>">
                            <?php echo esc_html(ucfirst($gc->status)); ?>
                        </span>
                    </td>
                    <td><?php echo date('M j, Y', strtotime($gc->created_at)); ?></td>
                    <td><?php echo $gc->expires_at ? date('M j, Y', strtotime($gc->expires_at)) : '—'; ?></td>
                    <td>
                        <button type="button" class="button button-small view-gc-btn" data-id="<?php echo esc_attr($gc->id); ?>">View</button>
                        <?php if ($gc->recipient_email && !$gc->is_delivered): ?>
                        <button type="button" class="button button-small send-gc-btn" data-id="<?php echo esc_attr($gc->id); ?>">Send</button>
                        <?php endif; ?>
                    </td>
                </tr>
                <?php endforeach; ?>
                <?php endif; ?>
            </tbody>
        </table>
    </div>
</div>

<!-- Create Gift Card Modal -->
<div id="create-gc-modal" class="cartt-modal" style="display: none;">
    <div class="cartt-modal-content" style="max-width: 500px;">
        <div class="cartt-modal-header">
            <h2>Create Gift Card</h2>
            <button type="button" class="cartt-modal-close">&times;</button>
        </div>
        <div class="cartt-modal-body">
            <form id="create-gc-form">
                <table class="form-table">
                    <tr>
                        <th><label for="gc-amount">Amount *</label></th>
                        <td><input type="number" id="gc-amount" step="0.01" min="1" required style="width: 150px;"></td>
                    </tr>
                    <tr>
                        <th><label for="gc-recipient-email">Recipient Email</label></th>
                        <td>
                            <input type="email" id="gc-recipient-email" class="regular-text">
                            <p class="description">Leave empty for a general gift card</p>
                        </td>
                    </tr>
                    <tr>
                        <th><label for="gc-recipient-name">Recipient Name</label></th>
                        <td><input type="text" id="gc-recipient-name" class="regular-text"></td>
                    </tr>
                    <tr>
                        <th><label for="gc-sender-name">From (Sender)</label></th>
                        <td><input type="text" id="gc-sender-name" class="regular-text"></td>
                    </tr>
                    <tr>
                        <th><label for="gc-message">Message</label></th>
                        <td><textarea id="gc-message" rows="3" class="large-text"></textarea></td>
                    </tr>
                    <tr>
                        <th><label for="gc-design">Design</label></th>
                        <td>
                            <select id="gc-design">
                                <?php foreach ($designs as $key => $name): ?>
                                <option value="<?php echo esc_attr($key); ?>"><?php echo esc_html($name); ?></option>
                                <?php endforeach; ?>
                            </select>
                        </td>
                    </tr>
                    <tr>
                        <th><label for="gc-expires">Expires In</label></th>
                        <td>
                            <select id="gc-expires">
                                <option value="">Never</option>
                                <option value="30">30 days</option>
                                <option value="90">90 days</option>
                                <option value="180">6 months</option>
                                <option value="365">1 year</option>
                            </select>
                        </td>
                    </tr>
                    <tr>
                        <th>Options</th>
                        <td>
                            <label><input type="checkbox" id="gc-send-now" checked> Send email immediately</label>
                        </td>
                    </tr>
                </table>
            </form>
        </div>
        <div class="cartt-modal-footer">
            <button type="button" class="button" onclick="document.getElementById('create-gc-modal').style.display='none'">Cancel</button>
            <button type="button" class="button button-primary" id="save-gc-btn">Create Gift Card</button>
        </div>
    </div>
</div>

<style>
.cartt-card { background: #fff; border: 1px solid #ccd0d4; border-radius: 4px; padding: 0; }
.cartt-stat-card { background: #fff; border: 1px solid #ccd0d4; border-radius: 8px; padding: 20px; text-align: center; }
.cartt-stat-card .stat-value { font-size: 32px; font-weight: 600; color: #1d1d1f; }
.cartt-stat-card .stat-label { font-size: 13px; color: #666; margin-top: 5px; }
.cartt-status { display: inline-block; padding: 3px 10px; border-radius: 3px; font-size: 12px; }
.cartt-status-active { background: #d4edda; color: #155724; }
.cartt-status-used { background: #e2e3e5; color: #383d41; }
.cartt-status-expired { background: #f8d7da; color: #721c24; }
.cartt-status-disabled { background: #fff3cd; color: #856404; }
.cartt-modal { position: fixed; top: 0; left: 0; right: 0; bottom: 0; background: rgba(0,0,0,0.6); z-index: 100000; display: flex; align-items: center; justify-content: center; }
.cartt-modal-content { background: #fff; border-radius: 4px; max-height: 90vh; overflow-y: auto; width: 90%; }
.cartt-modal-header { padding: 15px 20px; border-bottom: 1px solid #ddd; display: flex; justify-content: space-between; align-items: center; }
.cartt-modal-header h2 { margin: 0; }
.cartt-modal-close { background: none; border: none; font-size: 24px; cursor: pointer; }
.cartt-modal-body { padding: 20px; }
.cartt-modal-footer { padding: 15px 20px; border-top: 1px solid #ddd; text-align: right; }
</style>

<script>
jQuery(function($) {
    const nonce = '<?php echo wp_create_nonce('cartt_gift_cards'); ?>';

    $('#create-gift-card-btn').on('click', function() {
        $('#create-gc-modal').show();
    });

    $('.cartt-modal-close').on('click', function() {
        $(this).closest('.cartt-modal').hide();
    });

    $('#save-gc-btn').on('click', function() {
        const amount = $('#gc-amount').val();
        if (!amount || amount <= 0) {
            alert('Please enter a valid amount');
            return;
        }

        const data = {
            action: 'cartt_create_gift_card',
            nonce: nonce,
            amount: amount,
            recipient_email: $('#gc-recipient-email').val(),
            recipient_name: $('#gc-recipient-name').val(),
            sender_name: $('#gc-sender-name').val(),
            message: $('#gc-message').val(),
            design: $('#gc-design').val(),
            expires_days: $('#gc-expires').val(),
            send_now: $('#gc-send-now').is(':checked') ? 1 : 0
        };

        $.post(ajaxurl, data, function(response) {
            if (response.success) {
                location.reload();
            } else {
                alert('Error: ' + response.data.message);
            }
        });
    });

    $('.send-gc-btn').on('click', function() {
        const id = $(this).data('id');
        $.post(ajaxurl, {
            action: 'cartt_send_gift_card',
            nonce: nonce,
            id: id
        }, function(response) {
            if (response.success) {
                alert('Gift card sent!');
                location.reload();
            } else {
                alert('Error: ' + response.data.message);
            }
        });
    });
});
</script>
