<?php
/**
 * Dropshipping Admin View
 * 
 * @package Cartt
 * @since 1.4.0
 */

defined('ABSPATH') || exit;

// Get suppliers with error handling
$suppliers = [];
try {
    if (class_exists('Cartt\\Services\\DropshippingService')) {
        $dropship_service = \Cartt\Services\DropshippingService::instance();
        $suppliers = $dropship_service->list_suppliers();
    }
} catch (Exception $e) {
    // Service not available
}

$settings = get_option('cartt_dropshipping_settings', [
    'auto_fulfill' => true,
    'sync_inventory' => true,
    'sync_interval' => 'hourly',
    'default_markup' => 30,
    'markup_type' => 'percent',
    'import_images' => true,
    'import_description' => true
]);

// Get linked products count with error suppression
global $wpdb;
$products_table = $wpdb->prefix . 'cartt_supplier_products';
$total_linked = 0;
$pending_orders = 0;

$wpdb->suppress_errors(true);
$total_linked = (int) $wpdb->get_var("SELECT COUNT(*) FROM {$products_table}");
$pending_orders = (int) $wpdb->get_var($wpdb->prepare(
    "SELECT COUNT(DISTINCT order_id) FROM {$wpdb->prefix}cartt_order_items oi
     JOIN {$products_table} dp ON dp.product_id = oi.product_id
     JOIN {$wpdb->prefix}cartt_orders o ON o.id = oi.order_id
     WHERE o.status = %s AND dp.auto_order = 1",
    'processing'
));
$wpdb->suppress_errors(false);

$isEmbedded = isset($GLOBALS['cartt_embedded']) && $GLOBALS['cartt_embedded'];
// When embedded, use subtab; otherwise use tab
$current_tab = isset($_GET['subtab']) && $isEmbedded ? sanitize_key($_GET['subtab']) : (isset($_GET['tab']) ? sanitize_key($_GET['tab']) : 'suppliers');
?>

<?php if (!$isEmbedded): ?>
<div class="wrap cartt-admin">
    <h1 class="wp-heading-inline">Dropshipping</h1>
    <a href="#" class="page-title-action" id="add-supplier-btn">Add Supplier</a>
    <hr class="wp-header-end">
<?php endif; ?>

    <div class="cartt-stats-grid" style="display: grid; grid-template-columns: repeat(4, 1fr); gap: 15px; margin-bottom: 20px;">
        <div class="cartt-stat-card" style="background: #fff; padding: 20px; border-radius: 8px; box-shadow: 0 1px 3px rgba(0,0,0,0.1);">
            <div style="color: #666; font-size: 13px; margin-bottom: 5px;">Active Suppliers</div>
            <div style="font-size: 28px; font-weight: 600; color: #1d2327;"><?php echo count(array_filter($suppliers, fn($s) => $s['status'] === 'active')); ?></div>
        </div>
        <div class="cartt-stat-card" style="background: #fff; padding: 20px; border-radius: 8px; box-shadow: 0 1px 3px rgba(0,0,0,0.1);">
            <div style="color: #666; font-size: 13px; margin-bottom: 5px;">Linked Products</div>
            <div style="font-size: 28px; font-weight: 600; color: #50575e;"><?php echo $total_linked; ?></div>
        </div>
        <div class="cartt-stat-card" style="background: #fff; padding: 20px; border-radius: 8px; box-shadow: 0 1px 3px rgba(0,0,0,0.1);">
            <div style="color: #666; font-size: 13px; margin-bottom: 5px;">Pending Fulfillment</div>
            <div style="font-size: 28px; font-weight: 600; color: #dba617;"><?php echo $pending_orders; ?></div>
        </div>
        <div class="cartt-stat-card" style="background: #fff; padding: 20px; border-radius: 8px; box-shadow: 0 1px 3px rgba(0,0,0,0.1);">
            <div style="color: #666; font-size: 13px; margin-bottom: 5px;">Last Sync</div>
            <div style="font-size: 16px; font-weight: 600; color: #1d2327;"><?php echo get_option('cartt_dropship_last_sync', 'Never'); ?></div>
        </div>
    </div>

    <?php 
    $baseUrl = $isEmbedded ? '?page=cartt-marketplace&tab=dropshipping&subtab=' : '?page=cartt-dropshipping&tab=';
    ?>
    <nav class="nav-tab-wrapper">
        <a href="<?php echo $baseUrl; ?>suppliers" class="nav-tab <?php echo $current_tab === 'suppliers' ? 'nav-tab-active' : ''; ?>">Suppliers</a>
        <a href="<?php echo $baseUrl; ?>products" class="nav-tab <?php echo $current_tab === 'products' ? 'nav-tab-active' : ''; ?>">Product Mappings</a>
        <a href="<?php echo $baseUrl; ?>orders" class="nav-tab <?php echo $current_tab === 'orders' ? 'nav-tab-active' : ''; ?>">Dropship Orders</a>
        <a href="<?php echo $baseUrl; ?>import" class="nav-tab <?php echo $current_tab === 'import' ? 'nav-tab-active' : ''; ?>">Import Products</a>
        <a href="<?php echo $baseUrl; ?>settings" class="nav-tab <?php echo $current_tab === 'settings' ? 'nav-tab-active' : ''; ?>">Settings</a>
    </nav>

    <div style="background: #fff; padding: 20px; border: 1px solid #c3c4c7; border-top: none;">
        <?php if ($current_tab === 'suppliers'): ?>
            <table class="wp-list-table widefat fixed striped">
                <thead>
                    <tr>
                        <th style="width: 200px;">Supplier</th>
                        <th style="width: 120px;">Type</th>
                        <th style="width: 100px;">Products</th>
                        <th style="width: 100px;">Auto-Order</th>
                        <th style="width: 100px;">Auto-Sync</th>
                        <th style="width: 100px;">Status</th>
                        <th style="width: 150px;">Last Synced</th>
                        <th style="width: 150px;">Actions</th>
                    </tr>
                </thead>
                <tbody>
                    <?php if (empty($suppliers)): ?>
                        <tr>
                            <td colspan="8" style="text-align: center; padding: 40px;">
                                <p style="color: #666; margin-bottom: 15px;">No suppliers configured yet.</p>
                                <button type="button" class="button button-primary" id="add-first-supplier">Add Your First Supplier</button>
                            </td>
                        </tr>
                    <?php else: ?>
                        <?php foreach ($suppliers as $supplier): ?>
                            <tr data-supplier-id="<?php echo esc_attr($supplier['id']); ?>">
                                <td>
                                    <strong><?php echo esc_html($supplier['name']); ?></strong>
                                    <?php if (!empty($supplier['store_id'])): ?>
                                        <br><small style="color: #666;">Store: <?php echo esc_html($supplier['store_id']); ?></small>
                                    <?php endif; ?>
                                </td>
                                <td>
                                    <?php
                                    $type_labels = [
                                        'aliexpress' => '<span style="background: #ff6600; color: #fff; padding: 2px 8px; border-radius: 3px; font-size: 11px;">AliExpress</span>',
                                        'printful' => '<span style="background: #2b3948; color: #fff; padding: 2px 8px; border-radius: 3px; font-size: 11px;">Printful</span>',
                                        'printify' => '<span style="background: #29b474; color: #fff; padding: 2px 8px; border-radius: 3px; font-size: 11px;">Printify</span>',
                                        'cjdropshipping' => '<span style="background: #e74c3c; color: #fff; padding: 2px 8px; border-radius: 3px; font-size: 11px;">CJ</span>',
                                        'custom' => '<span style="background: #666; color: #fff; padding: 2px 8px; border-radius: 3px; font-size: 11px;">Custom</span>'
                                    ];
                                    echo $type_labels[$supplier['type']] ?? esc_html($supplier['type']);
                                    ?>
                                </td>
                                <td><?php echo (int) $supplier['product_count']; ?></td>
                                <td>
                                    <?php if ($supplier['auto_order']): ?>
                                        <span style="color: #00a32a;">Enabled</span>
                                    <?php else: ?>
                                        <span style="color: #666;">Disabled</span>
                                    <?php endif; ?>
                                </td>
                                <td>
                                    <?php if ($supplier['auto_sync']): ?>
                                        <span style="color: #00a32a;">Enabled</span>
                                    <?php else: ?>
                                        <span style="color: #666;">Disabled</span>
                                    <?php endif; ?>
                                </td>
                                <td>
                                    <?php
                                    $status_styles = [
                                        'active' => 'background: #d4edda; color: #155724;',
                                        'inactive' => 'background: #f8f9fa; color: #666;',
                                        'error' => 'background: #f8d7da; color: #721c24;'
                                    ];
                                    $style = $status_styles[$supplier['status']] ?? '';
                                    ?>
                                    <span style="<?php echo $style; ?> padding: 3px 8px; border-radius: 3px; font-size: 12px;">
                                        <?php echo ucfirst(esc_html($supplier['status'])); ?>
                                    </span>
                                </td>
                                <td>
                                    <?php echo $supplier['last_synced'] ? esc_html(human_time_diff(strtotime($supplier['last_synced']))) . ' ago' : 'Never'; ?>
                                </td>
                                <td>
                                    <button type="button" class="button button-small sync-supplier" data-id="<?php echo esc_attr($supplier['id']); ?>">Sync</button>
                                    <button type="button" class="button button-small edit-supplier" data-id="<?php echo esc_attr($supplier['id']); ?>">Edit</button>
                                    <button type="button" class="button button-small button-link-delete delete-supplier" data-id="<?php echo esc_attr($supplier['id']); ?>">Delete</button>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </tbody>
            </table>

        <?php elseif ($current_tab === 'products'): ?>
            <?php
            $linked_products = $wpdb->get_results("
                SELECT 
                    sp.*,
                    p.name as product_name,
                    p.price as sell_price,
                    p.sku,
                    s.name as supplier_name,
                    s.type as supplier_type
                FROM {$products_table} sp
                JOIN {$wpdb->prefix}cartt_products p ON p.id = sp.product_id
                JOIN {$wpdb->prefix}cartt_suppliers s ON s.id = sp.supplier_id
                ORDER BY sp.created_at DESC
                LIMIT 100
            ", ARRAY_A);
            ?>

            <div style="margin-bottom: 15px;">
                <button type="button" class="button" id="link-product-btn">Link Product to Supplier</button>
                <button type="button" class="button" id="bulk-sync-btn">Sync All Inventory</button>
            </div>

            <table class="wp-list-table widefat fixed striped">
                <thead>
                    <tr>
                        <th>Product</th>
                        <th style="width: 150px;">Supplier</th>
                        <th style="width: 120px;">Supplier SKU</th>
                        <th style="width: 100px;">Cost</th>
                        <th style="width: 100px;">Sell Price</th>
                        <th style="width: 100px;">Profit</th>
                        <th style="width: 80px;">Margin</th>
                        <th style="width: 100px;">Auto-Order</th>
                        <th style="width: 120px;">Actions</th>
                    </tr>
                </thead>
                <tbody>
                    <?php if (empty($linked_products)): ?>
                        <tr>
                            <td colspan="9" style="text-align: center; padding: 40px;">
                                <p style="color: #666;">No products linked to suppliers yet.</p>
                            </td>
                        </tr>
                    <?php else: ?>
                        <?php foreach ($linked_products as $product): 
                            $total_cost = (float)$product['cost_price'] + (float)$product['shipping_cost'];
                            $profit = (float)$product['sell_price'] - $total_cost;
                            $margin = $product['sell_price'] > 0 ? ($profit / $product['sell_price']) * 100 : 0;
                            $margin_color = $margin >= 30 ? '#00a32a' : ($margin >= 15 ? '#dba617' : '#dc3545');
                        ?>
                            <tr data-link-id="<?php echo esc_attr($product['id']); ?>">
                                <td>
                                    <strong><?php echo esc_html($product['product_name']); ?></strong>
                                    <?php if ($product['sku']): ?>
                                        <br><small style="color: #666;">SKU: <?php echo esc_html($product['sku']); ?></small>
                                    <?php endif; ?>
                                </td>
                                <td><?php echo esc_html($product['supplier_name']); ?></td>
                                <td><?php echo esc_html($product['supplier_sku'] ?: '-'); ?></td>
                                <td>
                                    $<?php echo number_format($product['cost_price'], 2); ?>
                                    <?php if ($product['shipping_cost'] > 0): ?>
                                        <br><small style="color: #666;">+$<?php echo number_format($product['shipping_cost'], 2); ?> ship</small>
                                    <?php endif; ?>
                                </td>
                                <td>$<?php echo number_format($product['sell_price'], 2); ?></td>
                                <td style="color: <?php echo $margin_color; ?>; font-weight: 600;">
                                    $<?php echo number_format($profit, 2); ?>
                                </td>
                                <td>
                                    <span style="color: <?php echo $margin_color; ?>; font-weight: 600;">
                                        <?php echo number_format($margin, 1); ?>%
                                    </span>
                                </td>
                                <td>
                                    <?php if ($product['auto_order']): ?>
                                        <span style="color: #00a32a;">Yes</span>
                                    <?php else: ?>
                                        <span style="color: #666;">No</span>
                                    <?php endif; ?>
                                </td>
                                <td>
                                    <button type="button" class="button button-small edit-link" data-id="<?php echo esc_attr($product['id']); ?>">Edit</button>
                                    <button type="button" class="button button-small button-link-delete unlink-product" data-id="<?php echo esc_attr($product['id']); ?>">Unlink</button>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </tbody>
            </table>

        <?php elseif ($current_tab === 'orders'): ?>
            <?php
            $dropship_orders = $wpdb->get_results("
                SELECT 
                    o.id,
                    o.order_number,
                    o.status,
                    o.total,
                    o.created_at,
                    GROUP_CONCAT(DISTINCT s.name) as suppliers,
                    SUM(sp.cost_price * oi.quantity) as total_cost
                FROM {$wpdb->prefix}cartt_orders o
                JOIN {$wpdb->prefix}cartt_order_items oi ON oi.order_id = o.id
                JOIN {$products_table} sp ON sp.product_id = oi.product_id
                JOIN {$wpdb->prefix}cartt_suppliers s ON s.id = sp.supplier_id
                GROUP BY o.id
                ORDER BY o.created_at DESC
                LIMIT 50
            ", ARRAY_A);
            ?>

            <table class="wp-list-table widefat fixed striped">
                <thead>
                    <tr>
                        <th style="width: 120px;">Order</th>
                        <th>Suppliers</th>
                        <th style="width: 100px;">Order Total</th>
                        <th style="width: 100px;">Cost</th>
                        <th style="width: 100px;">Profit</th>
                        <th style="width: 120px;">Status</th>
                        <th style="width: 150px;">Date</th>
                        <th style="width: 150px;">Actions</th>
                    </tr>
                </thead>
                <tbody>
                    <?php if (empty($dropship_orders)): ?>
                        <tr>
                            <td colspan="8" style="text-align: center; padding: 40px;">
                                <p style="color: #666;">No dropship orders yet.</p>
                            </td>
                        </tr>
                    <?php else: ?>
                        <?php foreach ($dropship_orders as $order): 
                            $profit = (float)$order['total'] - (float)$order['total_cost'];
                        ?>
                            <tr>
                                <td>
                                    <a href="?page=cartt-orders&action=view&id=<?php echo esc_attr($order['id']); ?>">
                                        <strong>#<?php echo esc_html($order['order_number']); ?></strong>
                                    </a>
                                </td>
                                <td><?php echo esc_html($order['suppliers']); ?></td>
                                <td>$<?php echo number_format($order['total'], 2); ?></td>
                                <td>$<?php echo number_format($order['total_cost'], 2); ?></td>
                                <td style="color: <?php echo $profit > 0 ? '#00a32a' : '#dc3545'; ?>; font-weight: 600;">
                                    $<?php echo number_format($profit, 2); ?>
                                </td>
                                <td>
                                    <?php
                                    $status_colors = [
                                        'pending' => '#f0ad4e',
                                        'processing' => '#0275d8',
                                        'shipped' => '#5bc0de',
                                        'completed' => '#5cb85c',
                                        'cancelled' => '#d9534f'
                                    ];
                                    $color = $status_colors[$order['status']] ?? '#666';
                                    ?>
                                    <span style="background: <?php echo $color; ?>; color: #fff; padding: 3px 8px; border-radius: 3px; font-size: 12px;">
                                        <?php echo ucfirst(esc_html($order['status'])); ?>
                                    </span>
                                </td>
                                <td><?php echo esc_html(date('M j, Y g:ia', strtotime($order['created_at']))); ?></td>
                                <td>
                                    <button type="button" class="button button-small view-order" data-id="<?php echo esc_attr($order['id']); ?>">View</button>
                                    <?php if ($order['status'] === 'processing'): ?>
                                        <button type="button" class="button button-small button-primary fulfill-order" data-id="<?php echo esc_attr($order['id']); ?>">Fulfill</button>
                                    <?php endif; ?>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </tbody>
            </table>

        <?php elseif ($current_tab === 'import'): ?>
            <div style="max-width: 800px;">
                <h3>Import from AliExpress</h3>
                <p style="color: #666; margin-bottom: 20px;">Paste an AliExpress product URL to import it to your store.</p>
                
                <div style="display: flex; gap: 10px; margin-bottom: 30px;">
                    <input type="url" id="aliexpress-url" class="regular-text" placeholder="https://www.aliexpress.com/item/..." style="flex: 1;">
                    <button type="button" class="button button-primary" id="import-aliexpress">Import Product</button>
                </div>

                <hr style="margin: 30px 0;">

                <h3>Import from Printful</h3>
                <p style="color: #666; margin-bottom: 20px;">Sync products from your Printful store.</p>
                
                <?php 
                $printful_supplier = array_filter($suppliers, fn($s) => $s['type'] === 'printful');
                if (empty($printful_supplier)): 
                ?>
                    <div style="background: #fff3cd; border: 1px solid #ffc107; padding: 15px; border-radius: 5px;">
                        <p style="margin: 0;">No Printful supplier configured. <a href="#" id="add-printful-link">Add Printful supplier</a> first.</p>
                    </div>
                <?php else: ?>
                    <button type="button" class="button" id="sync-printful">Sync Printful Products</button>
                <?php endif; ?>

                <hr style="margin: 30px 0;">

                <h3>Import from Printify</h3>
                <p style="color: #666; margin-bottom: 20px;">Sync products from your Printify store.</p>
                
                <?php 
                $printify_supplier = array_filter($suppliers, fn($s) => $s['type'] === 'printify');
                if (empty($printify_supplier)): 
                ?>
                    <div style="background: #fff3cd; border: 1px solid #ffc107; padding: 15px; border-radius: 5px;">
                        <p style="margin: 0;">No Printify supplier configured. <a href="#" id="add-printify-link">Add Printify supplier</a> first.</p>
                    </div>
                <?php else: ?>
                    <button type="button" class="button" id="sync-printify">Sync Printify Products</button>
                <?php endif; ?>

                <hr style="margin: 30px 0;">

                <h3>Bulk Import (CSV)</h3>
                <p style="color: #666; margin-bottom: 20px;">Upload a CSV file with product mappings.</p>
                
                <div style="border: 2px dashed #c3c4c7; border-radius: 8px; padding: 40px; text-align: center; margin-bottom: 15px;">
                    <input type="file" id="csv-upload" accept=".csv" style="display: none;">
                    <p style="margin-bottom: 15px;">Drag & drop CSV file here or</p>
                    <button type="button" class="button" onclick="document.getElementById('csv-upload').click()">Choose File</button>
                </div>
                
                <p><a href="#" id="download-template">Download CSV template</a></p>
            </div>

        <?php elseif ($current_tab === 'settings'): ?>
            <form method="post" action="" id="dropshipping-settings-form" style="max-width: 600px;">
                <?php wp_nonce_field('cartt_dropshipping_settings', 'cartt_nonce'); ?>
                
                <h3>Fulfillment Settings</h3>
                
                <table class="form-table">
                    <tr>
                        <th scope="row">Auto-Fulfill Orders</th>
                        <td>
                            <label>
                                <input type="checkbox" name="auto_fulfill" value="1" <?php checked($settings['auto_fulfill'] ?? false); ?>>
                                Automatically forward orders to suppliers when payment is received
                            </label>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">Inventory Sync</th>
                        <td>
                            <label>
                                <input type="checkbox" name="sync_inventory" value="1" <?php checked($settings['sync_inventory'] ?? false); ?>>
                                Automatically sync inventory levels from suppliers
                            </label>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">Sync Interval</th>
                        <td>
                            <select name="sync_interval">
                                <option value="hourly" <?php selected($settings['sync_interval'] ?? '', 'hourly'); ?>>Hourly</option>
                                <option value="twicedaily" <?php selected($settings['sync_interval'] ?? '', 'twicedaily'); ?>>Twice Daily</option>
                                <option value="daily" <?php selected($settings['sync_interval'] ?? '', 'daily'); ?>>Daily</option>
                            </select>
                        </td>
                    </tr>
                </table>

                <h3>Pricing Settings</h3>
                
                <table class="form-table">
                    <tr>
                        <th scope="row">Default Markup</th>
                        <td>
                            <input type="number" name="default_markup" value="<?php echo esc_attr($settings['default_markup'] ?? 30); ?>" min="0" step="0.1" class="small-text">
                            <select name="markup_type">
                                <option value="percent" <?php selected($settings['markup_type'] ?? '', 'percent'); ?>>%</option>
                                <option value="fixed" <?php selected($settings['markup_type'] ?? '', 'fixed'); ?>>$ Fixed</option>
                            </select>
                            <p class="description">Applied when importing new products</p>
                        </td>
                    </tr>
                </table>

                <h3>Import Settings</h3>
                
                <table class="form-table">
                    <tr>
                        <th scope="row">Import Images</th>
                        <td>
                            <label>
                                <input type="checkbox" name="import_images" value="1" <?php checked($settings['import_images'] ?? true); ?>>
                                Download and import product images
                            </label>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">Import Description</th>
                        <td>
                            <label>
                                <input type="checkbox" name="import_description" value="1" <?php checked($settings['import_description'] ?? true); ?>>
                                Import product descriptions
                            </label>
                        </td>
                    </tr>
                </table>

                <h3>API Credentials</h3>
                
                <table class="form-table">
                    <tr>
                        <th scope="row">AliExpress API Key</th>
                        <td>
                            <input type="password" name="aliexpress_api_key" value="<?php echo esc_attr(get_option('cartt_aliexpress_api_key', '')); ?>" class="regular-text">
                            <p class="description">Required for AliExpress product import</p>
                        </td>
                    </tr>
                </table>

                <p class="submit">
                    <button type="submit" class="button button-primary">Save Settings</button>
                </p>
            </form>
        <?php endif; ?>
    </div>
<?php if (!$isEmbedded): ?>
</div>
<?php endif; ?>

<!-- Add/Edit Supplier Modal -->
<div id="supplier-modal" class="cartt-modal" style="display: none;">
    <div class="cartt-modal-content" style="max-width: 500px;">
        <div class="cartt-modal-header">
            <h2 id="supplier-modal-title">Add Supplier</h2>
            <button type="button" class="cartt-modal-close">&times;</button>
        </div>
        <div class="cartt-modal-body">
            <form id="supplier-form">
                <input type="hidden" name="supplier_id" id="supplier-id">
                
                <table class="form-table">
                    <tr>
                        <th><label for="supplier-name">Supplier Name *</label></th>
                        <td><input type="text" id="supplier-name" name="name" class="regular-text" required></td>
                    </tr>
                    <tr>
                        <th><label for="supplier-type">Type *</label></th>
                        <td>
                            <select id="supplier-type" name="type" required>
                                <option value="">Select type...</option>
                                <option value="aliexpress">AliExpress</option>
                                <option value="printful">Printful</option>
                                <option value="printify">Printify</option>
                                <option value="cjdropshipping">CJ Dropshipping</option>
                                <option value="custom">Custom / Other</option>
                            </select>
                        </td>
                    </tr>
                    <tr>
                        <th><label for="supplier-api-key">API Key</label></th>
                        <td><input type="password" id="supplier-api-key" name="api_key" class="regular-text"></td>
                    </tr>
                    <tr>
                        <th><label for="supplier-api-secret">API Secret</label></th>
                        <td><input type="password" id="supplier-api-secret" name="api_secret" class="regular-text"></td>
                    </tr>
                    <tr>
                        <th><label for="supplier-store-id">Store ID</label></th>
                        <td><input type="text" id="supplier-store-id" name="store_id" class="regular-text"></td>
                    </tr>
                    <tr>
                        <th>Options</th>
                        <td>
                            <label style="display: block; margin-bottom: 8px;">
                                <input type="checkbox" name="auto_order" id="supplier-auto-order" value="1">
                                Auto-forward orders to supplier
                            </label>
                            <label>
                                <input type="checkbox" name="auto_sync" id="supplier-auto-sync" value="1">
                                Auto-sync inventory
                            </label>
                        </td>
                    </tr>
                </table>
            </form>
        </div>
        <div class="cartt-modal-footer">
            <button type="button" class="button cartt-modal-close">Cancel</button>
            <button type="button" class="button button-primary" id="save-supplier">Save Supplier</button>
        </div>
    </div>
</div>

<!-- Link Product Modal -->
<div id="link-product-modal" class="cartt-modal" style="display: none;">
    <div class="cartt-modal-content" style="max-width: 500px;">
        <div class="cartt-modal-header">
            <h2>Link Product to Supplier</h2>
            <button type="button" class="cartt-modal-close">&times;</button>
        </div>
        <div class="cartt-modal-body">
            <form id="link-product-form">
                <table class="form-table">
                    <tr>
                        <th><label for="link-product-id">Product *</label></th>
                        <td>
                            <select id="link-product-id" name="product_id" required style="width: 100%;">
                                <option value="">Select product...</option>
                            </select>
                        </td>
                    </tr>
                    <tr>
                        <th><label for="link-supplier-id">Supplier *</label></th>
                        <td>
                            <select id="link-supplier-id" name="supplier_id" required>
                                <option value="">Select supplier...</option>
                                <?php foreach ($suppliers as $supplier): ?>
                                    <option value="<?php echo esc_attr($supplier['id']); ?>"><?php echo esc_html($supplier['name']); ?></option>
                                <?php endforeach; ?>
                            </select>
                        </td>
                    </tr>
                    <tr>
                        <th><label for="link-supplier-sku">Supplier SKU</label></th>
                        <td><input type="text" id="link-supplier-sku" name="supplier_sku" class="regular-text"></td>
                    </tr>
                    <tr>
                        <th><label for="link-cost-price">Cost Price *</label></th>
                        <td><input type="number" id="link-cost-price" name="cost_price" step="0.01" min="0" class="regular-text" required></td>
                    </tr>
                    <tr>
                        <th><label for="link-shipping-cost">Shipping Cost</label></th>
                        <td><input type="number" id="link-shipping-cost" name="shipping_cost" step="0.01" min="0" value="0" class="regular-text"></td>
                    </tr>
                    <tr>
                        <th><label for="link-processing-days">Processing Days</label></th>
                        <td><input type="number" id="link-processing-days" name="processing_days" min="1" value="3" class="small-text"></td>
                    </tr>
                    <tr>
                        <th>Auto-Order</th>
                        <td>
                            <label>
                                <input type="checkbox" name="auto_order" id="link-auto-order" value="1" checked>
                                Automatically order from supplier when product is purchased
                            </label>
                        </td>
                    </tr>
                </table>
            </form>
        </div>
        <div class="cartt-modal-footer">
            <button type="button" class="button cartt-modal-close">Cancel</button>
            <button type="button" class="button button-primary" id="save-link">Link Product</button>
        </div>
    </div>
</div>

<style>
.cartt-modal {
    position: fixed;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: rgba(0,0,0,0.6);
    z-index: 100000;
    display: flex;
    align-items: center;
    justify-content: center;
}
.cartt-modal-content {
    background: #fff;
    border-radius: 8px;
    width: 90%;
    max-height: 90vh;
    overflow: auto;
}
.cartt-modal-header {
    padding: 15px 20px;
    border-bottom: 1px solid #ddd;
    display: flex;
    justify-content: space-between;
    align-items: center;
}
.cartt-modal-header h2 {
    margin: 0;
    font-size: 18px;
}
.cartt-modal-close {
    background: none;
    border: none;
    font-size: 24px;
    cursor: pointer;
    color: #666;
}
.cartt-modal-body {
    padding: 20px;
}
.cartt-modal-footer {
    padding: 15px 20px;
    border-top: 1px solid #ddd;
    text-align: right;
}
.cartt-modal-footer .button {
    margin-left: 10px;
}
</style>

<script>
jQuery(document).ready(function($) {
    // Add supplier button
    $('#add-supplier-btn, #add-first-supplier').on('click', function(e) {
        e.preventDefault();
        $('#supplier-modal-title').text('Add Supplier');
        $('#supplier-form')[0].reset();
        $('#supplier-id').val('');
        $('#supplier-modal').show();
    });

    // Edit supplier
    $('.edit-supplier').on('click', function() {
        const id = $(this).data('id');
        $('#supplier-modal-title').text('Edit Supplier');
        $('#supplier-id').val(id);
        // Load supplier data via AJAX and populate form
        $('#supplier-modal').show();
    });

    // Save supplier
    $('#save-supplier').on('click', function() {
        const $btn = $(this);
        const formData = new FormData($('#supplier-form')[0]);
        formData.append('action', 'cartt_save_supplier');
        formData.append('nonce', '<?php echo wp_create_nonce('cartt_admin_nonce'); ?>');

        $btn.prop('disabled', true).text('Saving...');

        $.ajax({
            url: ajaxurl,
            type: 'POST',
            data: formData,
            processData: false,
            contentType: false,
            success: function(response) {
                if (response.success) {
                    location.reload();
                } else {
                    alert(response.data?.message || 'Failed to save supplier');
                    $btn.prop('disabled', false).text('Save Supplier');
                }
            },
            error: function() {
                alert('An error occurred');
                $btn.prop('disabled', false).text('Save Supplier');
            }
        });
    });

    // Delete supplier
    $('.delete-supplier').on('click', function() {
        if (!confirm('Are you sure you want to delete this supplier? All product links will be removed.')) return;
        
        const id = $(this).data('id');
        const $row = $(this).closest('tr');

        $.post(ajaxurl, {
            action: 'cartt_delete_supplier',
            nonce: '<?php echo wp_create_nonce('cartt_admin_nonce'); ?>',
            supplier_id: id
        }, function(response) {
            if (response.success) {
                $row.fadeOut(function() { $(this).remove(); });
            } else {
                alert(response.data?.message || 'Failed to delete supplier');
            }
        });
    });

    // Sync supplier
    $('.sync-supplier').on('click', function() {
        const $btn = $(this);
        const id = $(this).data('id');

        $btn.prop('disabled', true).text('Syncing...');

        $.post(ajaxurl, {
            action: 'cartt_sync_supplier',
            nonce: '<?php echo wp_create_nonce('cartt_admin_nonce'); ?>',
            supplier_id: id
        }, function(response) {
            if (response.success) {
                alert('Sync completed: ' + (response.data?.message || 'Success'));
                location.reload();
            } else {
                alert(response.data?.message || 'Sync failed');
            }
            $btn.prop('disabled', false).text('Sync');
        });
    });

    // Link product modal
    $('#link-product-btn').on('click', function() {
        $('#link-product-form')[0].reset();
        $('#link-product-modal').show();
    });

    // Close modals
    $('.cartt-modal-close').on('click', function() {
        $(this).closest('.cartt-modal').hide();
    });

    // Import AliExpress
    $('#import-aliexpress').on('click', function() {
        const url = $('#aliexpress-url').val();
        if (!url) {
            alert('Please enter an AliExpress URL');
            return;
        }

        const $btn = $(this);
        $btn.prop('disabled', true).text('Importing...');

        $.post(ajaxurl, {
            action: 'cartt_import_aliexpress',
            nonce: '<?php echo wp_create_nonce('cartt_admin_nonce'); ?>',
            url: url
        }, function(response) {
            if (response.success) {
                alert('Product imported successfully!');
                $('#aliexpress-url').val('');
            } else {
                alert(response.data?.message || 'Import failed');
            }
            $btn.prop('disabled', false).text('Import Product');
        });
    });

    // Settings form
    $('#dropshipping-settings-form').on('submit', function(e) {
        e.preventDefault();
        const $form = $(this);
        const $btn = $form.find('button[type="submit"]');

        $btn.prop('disabled', true).text('Saving...');

        $.post(ajaxurl, {
            action: 'cartt_save_dropshipping_settings',
            nonce: '<?php echo wp_create_nonce('cartt_admin_nonce'); ?>',
            settings: $form.serialize()
        }, function(response) {
            if (response.success) {
                alert('Settings saved!');
            } else {
                alert(response.data?.message || 'Failed to save settings');
            }
            $btn.prop('disabled', false).text('Save Settings');
        });
    });
});
</script>
