<?php
/**
 * Currency Management Admin View
 */

if (!defined('ABSPATH')) {
    exit;
}

use Cartt\Services\CurrencyService;

// Handle initialization if needed
CurrencyService::initializeDefaults();

$currencies = CurrencyService::getAllCurrencies();
$availableCurrencies = CurrencyService::getAvailableCurrencyCodes();
$apiKey = get_option('cartt_exchange_rate_api', '');
$autoDetect = get_option('cartt_auto_detect_currency', false);
$isEmbedded = isset($GLOBALS['cartt_embedded']) && $GLOBALS['cartt_embedded'];
?>

<?php if (!$isEmbedded): ?><div class="wrap cartt-admin"><?php endif; ?>
    <h2 class="wp-heading-inline" style="margin-bottom: 15px;">Currency Settings</h2>
    <button type="button" class="page-title-action" id="add-currency-btn">Add Currency</button>

    <!-- API Settings -->
    <div class="cartt-card" style="margin-bottom: 20px;">
        <h2>Exchange Rate API</h2>
        <p class="description">Configure automatic exchange rate updates using <a href="https://www.exchangerate-api.com/" target="_blank">ExchangeRate-API</a> (free tier available).</p>
        
        <table class="form-table">
            <tr>
                <th scope="row">API Key</th>
                <td>
                    <input type="text" id="exchange-rate-api-key" class="regular-text" value="<?php echo esc_attr($apiKey); ?>" placeholder="Your API key">
                    <button type="button" class="button" id="save-api-key-btn">Save Key</button>
                    <button type="button" class="button button-primary" id="update-rates-btn" <?php echo empty($apiKey) ? 'disabled' : ''; ?>>Update Rates Now</button>
                </td>
            </tr>
            <tr>
                <th scope="row">Auto-detect Currency</th>
                <td>
                    <label>
                        <input type="checkbox" id="auto-detect-currency" <?php checked($autoDetect); ?>>
                        Automatically detect visitor's currency based on browser locale
                    </label>
                </td>
            </tr>
        </table>
    </div>

    <!-- Currencies List -->
    <div class="cartt-card">
        <h2>Enabled Currencies</h2>
        
        <table class="wp-list-table widefat fixed striped" id="currencies-table">
            <thead>
                <tr>
                    <th style="width: 60px;">Code</th>
                    <th style="width: 60px;">Symbol</th>
                    <th>Name</th>
                    <th style="width: 100px;">Rate</th>
                    <th style="width: 60px;">Dec.</th>
                    <th style="width: 80px;">Position</th>
                    <th style="width: 60px;">Default</th>
                    <th style="width: 60px;">Active</th>
                    <th style="width: 100px;">Actions</th>
                </tr>
            </thead>
            <tbody>
                <?php if (empty($currencies)): ?>
                <tr class="no-currencies">
                    <td colspan="9" style="text-align: center; padding: 20px;">
                        No currencies configured. Click "Add Currency" to get started.
                    </td>
                </tr>
                <?php else: ?>
                <?php foreach ($currencies as $currency): ?>
                <tr data-id="<?php echo esc_attr($currency->id); ?>">
                    <td><strong><?php echo esc_html($currency->code); ?></strong></td>
                    <td><?php echo esc_html($currency->symbol); ?></td>
                    <td><?php echo esc_html($currency->name); ?></td>
                    <td>
                        <input type="number" step="0.0001" class="currency-rate small-text" 
                               value="<?php echo esc_attr(round($currency->exchange_rate, 4)); ?>"
                               <?php echo $currency->is_default ? 'disabled' : ''; ?>>
                    </td>
                    <td><?php echo esc_html($currency->decimals); ?></td>
                    <td><?php echo esc_html($currency->symbol_position); ?></td>
                    <td style="text-align: center;">
                        <?php if ($currency->is_default): ?>
                            <span class="dashicons dashicons-star-filled" style="color: #1d2327;"></span>
                        <?php else: ?>
                            <button type="button" class="button button-small set-default-btn" title="Set as default">
                                <span class="dashicons dashicons-star-empty"></span>
                            </button>
                        <?php endif; ?>
                    </td>
                    <td style="text-align: center;">
                        <input type="checkbox" class="currency-enabled" <?php checked($currency->enabled); ?>>
                    </td>
                    <td>
                        <button type="button" class="button button-small edit-currency-btn">Edit</button>
                        <?php if (!$currency->is_default): ?>
                        <button type="button" class="button button-small button-link-delete delete-currency-btn">Delete</button>
                        <?php endif; ?>
                    </td>
                </tr>
                <?php endforeach; ?>
                <?php endif; ?>
            </tbody>
        </table>
    </div>

    <!-- Add Quick Currencies -->
    <div class="cartt-card" style="margin-top: 20px;">
        <h2>Quick Add Currencies</h2>
        <p class="description">Click to quickly add popular currencies:</p>
        <div style="display: flex; flex-wrap: wrap; gap: 8px; margin-top: 10px;">
            <?php 
            $popularCurrencies = ['USD', 'EUR', 'GBP', 'CAD', 'AUD', 'JPY', 'CHF', 'CNY', 'INR', 'MXN', 'BRL', 'KRW'];
            $existingCodes = array_column($currencies, 'code');
            foreach ($popularCurrencies as $code): 
                if (in_array($code, $existingCodes)) continue;
                $info = $availableCurrencies[$code] ?? null;
                if (!$info) continue;
            ?>
            <button type="button" class="button quick-add-currency" 
                    data-code="<?php echo esc_attr($code); ?>"
                    data-name="<?php echo esc_attr($info['name']); ?>"
                    data-symbol="<?php echo esc_attr($info['symbol']); ?>"
                    data-decimals="<?php echo esc_attr($info['decimals']); ?>">
                <?php echo esc_html($code); ?> (<?php echo esc_html($info['symbol']); ?>)
            </button>
            <?php endforeach; ?>
        </div>
    </div>
<?php if (!$isEmbedded): ?></div><?php endif; ?>

<!-- Add/Edit Currency Modal -->
<div id="currency-modal" class="cartt-modal" style="display: none;">
    <div class="cartt-modal-content" style="max-width: 500px;">
        <div class="cartt-modal-header">
            <h2 id="currency-modal-title">Add Currency</h2>
            <button type="button" class="cartt-modal-close">&times;</button>
        </div>
        <div class="cartt-modal-body">
            <form id="currency-form">
                <input type="hidden" id="currency-edit-id">
                
                <table class="form-table">
                    <tr>
                        <th scope="row"><label for="currency-code">Currency Code</label></th>
                        <td>
                            <select id="currency-code" style="width: 100%;" required>
                                <option value="">Select currency...</option>
                                <?php foreach ($availableCurrencies as $code => $info): ?>
                                <option value="<?php echo esc_attr($code); ?>" 
                                        data-name="<?php echo esc_attr($info['name']); ?>"
                                        data-symbol="<?php echo esc_attr($info['symbol']); ?>"
                                        data-decimals="<?php echo esc_attr($info['decimals']); ?>">
                                    <?php echo esc_html($code . ' - ' . $info['name']); ?>
                                </option>
                                <?php endforeach; ?>
                            </select>
                            <input type="text" id="currency-code-display" style="display: none; width: 100%;" disabled>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><label for="currency-name">Name</label></th>
                        <td><input type="text" id="currency-name" class="regular-text" required></td>
                    </tr>
                    <tr>
                        <th scope="row"><label for="currency-symbol">Symbol</label></th>
                        <td><input type="text" id="currency-symbol" style="width: 80px;" required></td>
                    </tr>
                    <tr>
                        <th scope="row"><label for="currency-exchange-rate">Exchange Rate</label></th>
                        <td>
                            <input type="number" id="currency-exchange-rate" step="0.000001" value="1" style="width: 150px;" required>
                            <p class="description">Rate relative to base currency</p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><label for="currency-decimals">Decimals</label></th>
                        <td>
                            <select id="currency-decimals">
                                <option value="0">0</option>
                                <option value="2" selected>2</option>
                                <option value="3">3</option>
                            </select>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><label for="currency-position">Symbol Position</label></th>
                        <td>
                            <select id="currency-position">
                                <option value="left">Left ($99.99)</option>
                                <option value="right">Right (99.99$)</option>
                                <option value="left_space">Left with space ($ 99.99)</option>
                                <option value="right_space">Right with space (99.99 $)</option>
                            </select>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><label for="currency-thousand">Thousand Separator</label></th>
                        <td>
                            <select id="currency-thousand">
                                <option value=",">,</option>
                                <option value=".">.</option>
                                <option value=" ">Space</option>
                                <option value="">None</option>
                            </select>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><label for="currency-decimal">Decimal Separator</label></th>
                        <td>
                            <select id="currency-decimal">
                                <option value=".">.</option>
                                <option value=",">,</option>
                            </select>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">Options</th>
                        <td>
                            <label><input type="checkbox" id="currency-enabled" checked> Enabled</label><br>
                            <label><input type="checkbox" id="currency-auto-update"> Auto-update exchange rate</label><br>
                            <label><input type="checkbox" id="currency-is-default"> Set as default currency</label>
                        </td>
                    </tr>
                </table>
            </form>
        </div>
        <div class="cartt-modal-footer">
            <button type="button" class="button" id="cancel-currency-btn">Cancel</button>
            <button type="button" class="button button-primary" id="save-currency-btn">Save Currency</button>
        </div>
    </div>
</div>

<style>
.cartt-card {
    background: #fff;
    border: 1px solid #ccd0d4;
    border-radius: 4px;
    padding: 20px;
}
.cartt-card h2 {
    margin-top: 0;
    padding-bottom: 10px;
    border-bottom: 1px solid #eee;
}
.cartt-modal {
    position: fixed;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: rgba(0,0,0,0.6);
    z-index: 100000;
    display: flex;
    align-items: center;
    justify-content: center;
}
.cartt-modal-content {
    background: #fff;
    border-radius: 4px;
    max-height: 90vh;
    overflow-y: auto;
    width: 90%;
}
.cartt-modal-header {
    padding: 15px 20px;
    border-bottom: 1px solid #ddd;
    display: flex;
    justify-content: space-between;
    align-items: center;
}
.cartt-modal-header h2 {
    margin: 0;
}
.cartt-modal-close {
    background: none;
    border: none;
    font-size: 24px;
    cursor: pointer;
    color: #666;
}
.cartt-modal-body {
    padding: 20px;
}
.cartt-modal-footer {
    padding: 15px 20px;
    border-top: 1px solid #ddd;
    text-align: right;
}
.cartt-modal-footer .button {
    margin-left: 10px;
}
#currencies-table .currency-rate {
    width: 100px;
}
.button .dashicons {
    vertical-align: middle;
    line-height: 26px;
}
</style>

<script>
jQuery(function($) {
    const nonce = '<?php echo wp_create_nonce('cartt_currency'); ?>';
    
    // Save API key
    $('#save-api-key-btn').on('click', function() {
        const apiKey = $('#exchange-rate-api-key').val();
        $.post(ajaxurl, {
            action: 'cartt_save_option',
            nonce: nonce,
            option: 'cartt_exchange_rate_api',
            value: apiKey
        }, function() {
            $('#update-rates-btn').prop('disabled', !apiKey);
            alert('API key saved!');
        });
    });
    
    // Toggle auto-detect
    $('#auto-detect-currency').on('change', function() {
        $.post(ajaxurl, {
            action: 'cartt_save_option',
            nonce: nonce,
            option: 'cartt_auto_detect_currency',
            value: $(this).is(':checked') ? '1' : ''
        });
    });
    
    // Update exchange rates
    $('#update-rates-btn').on('click', function() {
        const btn = $(this);
        btn.prop('disabled', true).text('Updating...');
        
        $.post(ajaxurl, {
            action: 'cartt_update_exchange_rates',
            nonce: nonce
        }, function(response) {
            btn.prop('disabled', false).text('Update Rates Now');
            if (response.success) {
                alert(response.data.message);
                location.reload();
            } else {
                alert('Error: ' + response.data.message);
            }
        });
    });
    
    // Open add currency modal
    $('#add-currency-btn').on('click', function() {
        $('#currency-modal-title').text('Add Currency');
        $('#currency-form')[0].reset();
        $('#currency-edit-id').val('');
        $('#currency-code').show().prop('disabled', false);
        $('#currency-code-display').hide();
        $('#currency-modal').show();
    });
    
    // Quick add currency
    $('.quick-add-currency').on('click', function() {
        const btn = $(this);
        const data = {
            action: 'cartt_save_currency',
            nonce: nonce,
            code: btn.data('code'),
            name: btn.data('name'),
            symbol: btn.data('symbol'),
            decimals: btn.data('decimals'),
            exchange_rate: 1,
            symbol_position: 'left',
            thousand_separator: ',',
            decimal_separator: '.',
            enabled: 1,
            auto_update: 1
        };
        
        btn.prop('disabled', true);
        $.post(ajaxurl, data, function(response) {
            if (response.success) {
                location.reload();
            } else {
                alert('Error: ' + response.data.message);
                btn.prop('disabled', false);
            }
        });
    });
    
    // Currency code selection
    $('#currency-code').on('change', function() {
        const opt = $(this).find(':selected');
        if (opt.val()) {
            $('#currency-name').val(opt.data('name'));
            $('#currency-symbol').val(opt.data('symbol'));
            $('#currency-decimals').val(opt.data('decimals'));
        }
    });
    
    // Close modal
    $('.cartt-modal-close, #cancel-currency-btn').on('click', function() {
        $('#currency-modal').hide();
    });
    
    // Save currency
    $('#save-currency-btn').on('click', function() {
        const editId = $('#currency-edit-id').val();
        const code = editId ? $('#currency-code-display').val() : $('#currency-code').val();
        
        if (!code) {
            alert('Please select a currency');
            return;
        }
        
        const data = {
            action: 'cartt_save_currency',
            nonce: nonce,
            code: code,
            name: $('#currency-name').val(),
            symbol: $('#currency-symbol').val(),
            exchange_rate: $('#currency-exchange-rate').val(),
            decimals: $('#currency-decimals').val(),
            symbol_position: $('#currency-position').val(),
            thousand_separator: $('#currency-thousand').val(),
            decimal_separator: $('#currency-decimal').val(),
            enabled: $('#currency-enabled').is(':checked') ? 1 : 0,
            auto_update: $('#currency-auto-update').is(':checked') ? 1 : 0,
            is_default: $('#currency-is-default').is(':checked') ? 1 : 0
        };
        
        $.post(ajaxurl, data, function(response) {
            if (response.success) {
                location.reload();
            } else {
                alert('Error: ' + response.data.message);
            }
        });
    });
    
    // Edit currency
    $(document).on('click', '.edit-currency-btn', function() {
        const row = $(this).closest('tr');
        const id = row.data('id');
        
        // Fetch current values from row
        const code = row.find('td:eq(0)').text().trim();
        const symbol = row.find('td:eq(1)').text().trim();
        const name = row.find('td:eq(2)').text().trim();
        const rate = row.find('.currency-rate').val();
        const decimals = row.find('td:eq(4)').text().trim();
        const position = row.find('td:eq(5)').text().trim();
        const autoUpdate = row.find('.currency-auto-update').is(':checked');
        const isDefault = row.find('td:eq(7) .dashicons-star-filled').length > 0;
        const enabled = row.find('.currency-enabled').is(':checked');
        
        $('#currency-modal-title').text('Edit Currency');
        $('#currency-edit-id').val(id);
        $('#currency-code').hide();
        $('#currency-code-display').val(code).show();
        $('#currency-name').val(name);
        $('#currency-symbol').val(symbol);
        $('#currency-exchange-rate').val(rate);
        $('#currency-decimals').val(decimals);
        $('#currency-position').val(position);
        $('#currency-enabled').prop('checked', enabled);
        $('#currency-auto-update').prop('checked', autoUpdate);
        $('#currency-is-default').prop('checked', isDefault);
        
        $('#currency-modal').show();
    });
    
    // Delete currency
    $(document).on('click', '.delete-currency-btn', function() {
        if (!confirm('Delete this currency?')) return;
        
        const row = $(this).closest('tr');
        const id = row.data('id');
        
        $.post(ajaxurl, {
            action: 'cartt_delete_currency',
            nonce: nonce,
            id: id
        }, function(response) {
            if (response.success) {
                row.fadeOut(function() { $(this).remove(); });
            } else {
                alert('Error: ' + response.data.message);
            }
        });
    });
    
    // Set as default
    $(document).on('click', '.set-default-btn', function() {
        const row = $(this).closest('tr');
        const id = row.data('id');
        const code = row.find('td:eq(0)').text().trim();
        
        $.post(ajaxurl, {
            action: 'cartt_save_currency',
            nonce: nonce,
            code: code,
            name: row.find('td:eq(2)').text().trim(),
            symbol: row.find('td:eq(1)').text().trim(),
            exchange_rate: row.find('.currency-rate').val(),
            decimals: row.find('td:eq(4)').text().trim(),
            symbol_position: row.find('td:eq(5)').text().trim(),
            enabled: 1,
            is_default: 1
        }, function(response) {
            if (response.success) {
                location.reload();
            }
        });
    });
    
    // Inline rate update
    $(document).on('change', '.currency-rate', function() {
        const row = $(this).closest('tr');
        const code = row.find('td:eq(0)').text().trim();
        
        $.post(ajaxurl, {
            action: 'cartt_save_currency',
            nonce: nonce,
            code: code,
            name: row.find('td:eq(2)').text().trim(),
            symbol: row.find('td:eq(1)').text().trim(),
            exchange_rate: $(this).val(),
            decimals: row.find('td:eq(4)').text().trim(),
            symbol_position: row.find('td:eq(5)').text().trim(),
            enabled: row.find('.currency-enabled').is(':checked') ? 1 : 0,
            auto_update: row.find('.currency-auto-update').is(':checked') ? 1 : 0
        });
    });
    
    // Inline enabled toggle
    $(document).on('change', '.currency-enabled', function() {
        const row = $(this).closest('tr');
        const code = row.find('td:eq(0)').text().trim();
        
        $.post(ajaxurl, {
            action: 'cartt_save_currency',
            nonce: nonce,
            code: code,
            name: row.find('td:eq(2)').text().trim(),
            symbol: row.find('td:eq(1)').text().trim(),
            exchange_rate: row.find('.currency-rate').val(),
            decimals: row.find('td:eq(4)').text().trim(),
            symbol_position: row.find('td:eq(5)').text().trim(),
            enabled: $(this).is(':checked') ? 1 : 0,
            auto_update: row.find('.currency-auto-update').is(':checked') ? 1 : 0
        });
    });
    
    // Inline auto-update toggle
    $(document).on('change', '.currency-auto-update', function() {
        const row = $(this).closest('tr');
        const code = row.find('td:eq(0)').text().trim();
        
        $.post(ajaxurl, {
            action: 'cartt_save_currency',
            nonce: nonce,
            code: code,
            name: row.find('td:eq(2)').text().trim(),
            symbol: row.find('td:eq(1)').text().trim(),
            exchange_rate: row.find('.currency-rate').val(),
            decimals: row.find('td:eq(4)').text().trim(),
            symbol_position: row.find('td:eq(5)').text().trim(),
            enabled: row.find('.currency-enabled').is(':checked') ? 1 : 0,
            auto_update: $(this).is(':checked') ? 1 : 0
        });
    });
});
</script>
