<?php
/**
 * Commissions & Payouts Admin View
 * @since 1.4.0
 */

if (!defined('ABSPATH')) {
    exit;
}

global $wpdb;

// When embedded, use subtab; otherwise use tab
$isEmbeddedCheck = isset($GLOBALS['cartt_embedded']) && $GLOBALS['cartt_embedded'];
$tab = isset($_GET['subtab']) && $isEmbeddedCheck ? sanitize_text_field($_GET['subtab']) : (isset($_GET['tab']) ? sanitize_text_field($_GET['tab']) : 'overview');
$filterStatus = isset($_GET['status']) ? sanitize_text_field($_GET['status']) : '';

// Initialize with defaults
$stats = (object) [
    'total_platform_fees' => 0,
    'total_vendor_earnings' => 0,
    'pending_amount' => 0,
    'available_amount' => 0,
    'paid_amount' => 0
];
$pendingPayouts = [];
$recentPayouts = [];
$vendorBalances = [];

// Get data with error suppression for missing tables
$wpdb->suppress_errors(true);

// Get overall stats
$stats_result = $wpdb->get_row(
    "SELECT 
        SUM(platform_fee) as total_platform_fees,
        SUM(vendor_amount) as total_vendor_earnings,
        SUM(CASE WHEN status = 'pending' THEN vendor_amount ELSE 0 END) as pending_amount,
        SUM(CASE WHEN status = 'available' THEN vendor_amount ELSE 0 END) as available_amount,
        SUM(CASE WHEN status = 'paid' THEN vendor_amount ELSE 0 END) as paid_amount
    FROM {$wpdb->prefix}cartt_vendor_commissions"
);
if ($stats_result) {
    $stats = $stats_result;
}

// Get pending payouts
$pendingPayouts_result = $wpdb->get_results(
    "SELECT p.*, v.store_name, v.email as vendor_email
    FROM {$wpdb->prefix}cartt_vendor_payouts p
    JOIN {$wpdb->prefix}cartt_vendors v ON p.vendor_id = v.id
    WHERE p.status IN ('pending', 'processing')
    ORDER BY p.requested_at DESC"
);
if ($pendingPayouts_result) {
    $pendingPayouts = $pendingPayouts_result;
}

// Get recent payouts
$recentPayouts_result = $wpdb->get_results(
    "SELECT p.*, v.store_name, v.email as vendor_email
    FROM {$wpdb->prefix}cartt_vendor_payouts p
    JOIN {$wpdb->prefix}cartt_vendors v ON p.vendor_id = v.id
    ORDER BY p.requested_at DESC
    LIMIT 20"
);
if ($recentPayouts_result) {
    $recentPayouts = $recentPayouts_result;
}

// Get vendor balances
$vendorBalances_result = $wpdb->get_results(
    "SELECT 
        v.id, v.store_name, v.email,
        COALESCE(SUM(CASE WHEN c.status = 'available' THEN c.vendor_amount ELSE 0 END), 0) as available_balance,
        COALESCE(SUM(c.vendor_amount), 0) as total_earned,
        COALESCE(SUM(CASE WHEN c.status = 'paid' THEN c.vendor_amount ELSE 0 END), 0) as total_paid
    FROM {$wpdb->prefix}cartt_vendors v
    LEFT JOIN {$wpdb->prefix}cartt_vendor_commissions c ON v.id = c.vendor_id
    WHERE v.status = 'active'
    GROUP BY v.id
    ORDER BY available_balance DESC"
);
if ($vendorBalances_result) {
    $vendorBalances = $vendorBalances_result;
}

$wpdb->suppress_errors(false);

$paymentMethods = [
    'paypal' => 'PayPal',
    'stripe' => 'Stripe Connect',
    'bank' => 'Bank Transfer'
];

$payoutStatuses = [
    'pending' => 'Pending',
    'processing' => 'Processing',
    'completed' => 'Completed',
    'rejected' => 'Rejected'
];
$isEmbedded = isset($GLOBALS['cartt_embedded']) && $GLOBALS['cartt_embedded'];
?>

<?php if (!$isEmbedded): ?>
<div class="wrap cartt-admin">
    <h1 class="wp-heading-inline">Commissions & Payouts</h1>
    <hr class="wp-header-end">
<?php endif; ?>

    <!-- Stats -->
    <div class="cartt-stats-row" style="display: grid; grid-template-columns: repeat(5, 1fr); gap: 20px; margin-bottom: 30px;">
        <div class="cartt-stat-card">
            <div class="stat-value"><?php echo \Cartt\Services\CurrencyService::formatPrice($stats->total_platform_fees ?? 0); ?></div>
            <div class="stat-label">Platform Fees Earned</div>
        </div>
        <div class="cartt-stat-card">
            <div class="stat-value"><?php echo \Cartt\Services\CurrencyService::formatPrice($stats->total_vendor_earnings ?? 0); ?></div>
            <div class="stat-label">Total Vendor Earnings</div>
        </div>
        <div class="cartt-stat-card" style="<?php echo ($stats->pending_amount ?? 0) > 0 ? 'border-color: #f0b849;' : ''; ?>">
            <div class="stat-value"><?php echo \Cartt\Services\CurrencyService::formatPrice($stats->pending_amount ?? 0); ?></div>
            <div class="stat-label">Pending (Hold Period)</div>
        </div>
        <div class="cartt-stat-card" style="border-color: #4CAF50;">
            <div class="stat-value"><?php echo \Cartt\Services\CurrencyService::formatPrice($stats->available_amount ?? 0); ?></div>
            <div class="stat-label">Available for Payout</div>
        </div>
        <div class="cartt-stat-card">
            <div class="stat-value"><?php echo \Cartt\Services\CurrencyService::formatPrice($stats->paid_amount ?? 0); ?></div>
            <div class="stat-label">Paid Out</div>
        </div>
    </div>

    <!-- Tabs -->
    <?php 
    $baseUrl = $isEmbedded ? '?page=cartt-marketplace&tab=commissions&subtab=' : '?page=cartt-commissions&tab=';
    ?>
    <h2 class="nav-tab-wrapper">
        <a href="<?php echo $baseUrl; ?>overview" class="nav-tab <?php echo $tab === 'overview' ? 'nav-tab-active' : ''; ?>">Overview</a>
        <a href="<?php echo $baseUrl; ?>payouts" class="nav-tab <?php echo $tab === 'payouts' ? 'nav-tab-active' : ''; ?>">
            Payout Requests
            <?php if (count($pendingPayouts) > 0): ?>
            <span class="awaiting-mod count-<?php echo count($pendingPayouts); ?>"><span class="pending-count"><?php echo count($pendingPayouts); ?></span></span>
            <?php endif; ?>
        </a>
        <a href="<?php echo $baseUrl; ?>balances" class="nav-tab <?php echo $tab === 'balances' ? 'nav-tab-active' : ''; ?>">Vendor Balances</a>
        <a href="<?php echo $baseUrl; ?>settings" class="nav-tab <?php echo $tab === 'settings' ? 'nav-tab-active' : ''; ?>">Settings</a>
    </h2>

    <div style="background: #fff; padding: 20px; border: 1px solid #c3c4c7; border-top: none;">
        <?php if ($tab === 'overview' || $tab === ''): ?>
        <!-- Recent Commission Activity -->
        <div class="cartt-card">
            <h2>Recent Payout Activity</h2>
            
            <?php if (empty($recentPayouts)): ?>
            <p style="color: #666; padding: 20px 0; text-align: center;">No payout activity yet.</p>
            <?php else: ?>
            <table class="wp-list-table widefat fixed striped">
                <thead>
                    <tr>
                        <th>Vendor</th>
                        <th>Amount</th>
                        <th>Method</th>
                        <th>Status</th>
                        <th>Requested</th>
                        <th>Processed</th>
                        <th>Actions</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($recentPayouts as $payout): ?>
                    <tr>
                        <td>
                            <strong><?php echo esc_html($payout->store_name); ?></strong>
                            <br><small><?php echo esc_html($payout->vendor_email); ?></small>
                        </td>
                        <td><strong><?php echo \Cartt\Services\CurrencyService::formatPrice($payout->amount); ?></strong></td>
                        <td><?php echo esc_html($paymentMethods[$payout->payment_method] ?? $payout->payment_method); ?></td>
                        <td>
                            <?php
                            $statusColors = [
                                'pending' => '#f0b849',
                                'processing' => '#2196F3',
                                'completed' => '#4CAF50',
                                'rejected' => '#f44336'
                            ];
                            ?>
                            <span style="background: <?php echo $statusColors[$payout->status] ?? '#ccc'; ?>; color: #fff; padding: 3px 8px; border-radius: 3px; font-size: 11px;">
                                <?php echo esc_html(ucfirst($payout->status)); ?>
                            </span>
                        </td>
                        <td><?php echo date('M j, Y', strtotime($payout->requested_at)); ?></td>
                        <td><?php echo $payout->processed_at ? date('M j, Y', strtotime($payout->processed_at)) : '-'; ?></td>
                        <td>
                            <?php if ($payout->status === 'pending'): ?>
                            <button type="button" class="button button-small button-primary" onclick="carttProcessPayout(<?php echo $payout->id; ?>)">Process</button>
                            <button type="button" class="button button-small" onclick="carttRejectPayout(<?php echo $payout->id; ?>)">Reject</button>
                            <?php elseif ($payout->transaction_id): ?>
                            <span style="font-size: 11px; color: #666;">Txn: <?php echo esc_html($payout->transaction_id); ?></span>
                            <?php endif; ?>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
            <?php endif; ?>
        </div>
        
        <?php elseif ($tab === 'payouts'): ?>
        <!-- Pending Payout Requests -->
        <div class="cartt-card">
            <h2>Pending Payout Requests</h2>
            
            <?php if (empty($pendingPayouts)): ?>
            <p style="color: #666; padding: 20px 0; text-align: center;">No pending payout requests.</p>
            <?php else: ?>
            <table class="wp-list-table widefat fixed striped">
                <thead>
                    <tr>
                        <th>Vendor</th>
                        <th>Amount</th>
                        <th>Payment Method</th>
                        <th>Payment Details</th>
                        <th>Requested</th>
                        <th>Actions</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($pendingPayouts as $payout): ?>
                    <tr>
                        <td>
                            <strong><?php echo esc_html($payout->store_name); ?></strong>
                            <br><small><?php echo esc_html($payout->vendor_email); ?></small>
                        </td>
                        <td><strong style="font-size: 16px;"><?php echo \Cartt\Services\CurrencyService::formatPrice($payout->amount); ?></strong></td>
                        <td><?php echo esc_html($paymentMethods[$payout->payment_method] ?? $payout->payment_method); ?></td>
                        <td>
                            <?php if ($payout->payment_email): ?>
                            <?php echo esc_html($payout->payment_email); ?>
                            <?php elseif ($payout->payment_details): ?>
                            <small><?php echo esc_html($payout->payment_details); ?></small>
                            <?php endif; ?>
                        </td>
                        <td>
                            <?php echo date('M j, Y', strtotime($payout->requested_at)); ?>
                            <br><small><?php echo human_time_diff(strtotime($payout->requested_at)); ?> ago</small>
                        </td>
                        <td>
                            <button type="button" class="button button-primary" onclick="carttProcessPayout(<?php echo $payout->id; ?>)">Process Payment</button>
                            <button type="button" class="button" onclick="carttRejectPayout(<?php echo $payout->id; ?>)">Reject</button>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
            <?php endif; ?>
        </div>
        
        <?php elseif ($tab === 'balances'): ?>
        <!-- Vendor Balances -->
        <div class="cartt-card">
            <h2>Vendor Balances</h2>
            
            <table class="wp-list-table widefat fixed striped">
                <thead>
                    <tr>
                        <th>Vendor</th>
                        <th>Available Balance</th>
                        <th>Total Earned</th>
                        <th>Total Paid Out</th>
                        <th>Actions</th>
                    </tr>
                </thead>
                <tbody>
                    <?php if (empty($vendorBalances)): ?>
                    <tr>
                        <td colspan="5" style="text-align: center; padding: 60px 20px;">
                            <div style="margin-bottom: 16px;"><span class="material-icons-outlined" style="font-size: 48px; color: #ccc;">payments</span></div>
                            <h3 style="margin: 0 0 8px;">No vendor earnings yet</h3>
                            <p style="color: #666; margin: 0;">Commissions appear here when vendors make sales.</p>
                        </td>
                    </tr>
                    <?php else: ?>
                    <?php foreach ($vendorBalances as $vendor): ?>
                    <tr>
                        <td>
                            <strong><?php echo esc_html($vendor->store_name); ?></strong>
                            <br><small><?php echo esc_html($vendor->email); ?></small>
                        </td>
                        <td>
                            <strong style="color: <?php echo $vendor->available_balance > 0 ? '#4CAF50' : '#666'; ?>;">
                                <?php echo \Cartt\Services\CurrencyService::formatPrice($vendor->available_balance); ?>
                            </strong>
                        </td>
                        <td><?php echo \Cartt\Services\CurrencyService::formatPrice($vendor->total_earned); ?></td>
                        <td><?php echo \Cartt\Services\CurrencyService::formatPrice($vendor->total_paid); ?></td>
                        <td>
                            <?php $vendorViewUrl = $isEmbedded ? '?page=cartt-marketplace&tab=vendors&action=view&id=' . $vendor->id : admin_url('admin.php?page=cartt-vendors&action=view&id=' . $vendor->id); ?>
                            <a href="<?php echo $vendorViewUrl; ?>" class="button button-small">View Vendor</a>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                    <?php endif; ?>
                </tbody>
            </table>
        </div>
        
        <?php elseif ($tab === 'settings'): ?>
        <!-- Commission Settings -->
        <div class="cartt-card">
            <h2>Commission Settings</h2>
            
            <form method="post" id="commission-settings-form">
                <table class="form-table">
                    <tr>
                        <th scope="row">Platform Commission Rate</th>
                        <td>
                            <input type="number" name="cartt_platform_commission_rate" value="<?php echo esc_attr(get_option('cartt_platform_commission_rate', 15)); ?>" min="0" max="100" step="0.01" class="small-text"> %
                            <p class="description">Default commission rate for all vendors. Individual vendors can have custom rates.</p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">Payout Hold Period</th>
                        <td>
                            <input type="number" name="cartt_payout_hold_days" value="<?php echo esc_attr(get_option('cartt_payout_hold_days', 14)); ?>" min="0" max="90" class="small-text"> days
                            <p class="description">Number of days to hold earnings before they become available for payout (for refund protection).</p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">Minimum Payout Amount</th>
                        <td>
                            <input type="number" name="cartt_minimum_payout" value="<?php echo esc_attr(get_option('cartt_minimum_payout', 50)); ?>" min="0" step="0.01" class="small-text">
                            <p class="description">Minimum balance required before vendors can request a payout.</p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">Available Payment Methods</th>
                        <td>
                            <?php $enabledMethods = get_option('cartt_payout_methods', ['paypal']); ?>
                            <label><input type="checkbox" name="cartt_payout_methods[]" value="paypal" <?php checked(in_array('paypal', $enabledMethods)); ?>> PayPal</label><br>
                            <label><input type="checkbox" name="cartt_payout_methods[]" value="stripe" <?php checked(in_array('stripe', $enabledMethods)); ?>> Stripe Connect</label><br>
                            <label><input type="checkbox" name="cartt_payout_methods[]" value="bank" <?php checked(in_array('bank', $enabledMethods)); ?>> Bank Transfer</label>
                        </td>
                    </tr>
                </table>
                
                <p class="submit">
                    <button type="submit" class="button button-primary">Save Settings</button>
                </p>
            </form>
        </div>
        <?php endif; ?>
    </div>
<?php if (!$isEmbedded): ?>
</div>
<?php endif; ?>

<script>
var carttCommissionsNonce = '<?php echo wp_create_nonce('cartt_commissions'); ?>';

function carttProcessPayout(id) {
    var transactionId = prompt('Enter transaction ID (PayPal transaction ID, Stripe payout ID, or bank reference):');
    if (!transactionId) return;
    
    jQuery.post(ajaxurl, {
        action: 'cartt_process_payout',
        nonce: carttCommissionsNonce,
        payout_id: id,
        transaction_id: transactionId
    }, function(response) {
        if (response.success) {
            location.reload();
        } else {
            alert(response.data.message || 'Failed to process payout');
        }
    });
}

function carttRejectPayout(id) {
    var reason = prompt('Rejection reason:');
    if (!reason) return;
    
    jQuery.post(ajaxurl, {
        action: 'cartt_reject_payout',
        nonce: carttCommissionsNonce,
        payout_id: id,
        reason: reason
    }, function(response) {
        if (response.success) {
            location.reload();
        } else {
            alert(response.data.message || 'Failed to reject payout');
        }
    });
}

// Save settings via AJAX
jQuery('#commission-settings-form').on('submit', function(e) {
    e.preventDefault();
    
    var formData = jQuery(this).serialize();
    
    jQuery.post(ajaxurl, {
        action: 'cartt_save_option',
        nonce: '<?php echo wp_create_nonce('cartt_save_option'); ?>',
        options: formData
    }, function(response) {
        if (response.success) {
            alert('Settings saved!');
        } else {
            alert('Failed to save settings');
        }
    });
});
</script>
