<?php
/**
 * Checkout Builder Admin Page
 */

if (!defined('ABSPATH')) {
    exit;
}

global $wpdb;
$config = \Cartt\Services\CheckoutConfig::get();
$fields = $config['fields'] ?? \Cartt\Services\CheckoutConfig::getDefaultFields();
$orderBumps = $config['order_bumps'] ?? [];

// Get products for bump selector
$products = $wpdb->get_results(
    "SELECT id, name, price, sale_price FROM {$wpdb->prefix}cartt_products WHERE status = 'publish' ORDER BY name ASC"
);

// Handle bump actions
if (isset($_GET['delete_bump']) && isset($_GET['_wpnonce'])) {
    if (wp_verify_nonce($_GET['_wpnonce'], 'delete_bump_' . $_GET['delete_bump'])) {
        \Cartt\Services\CheckoutConfig::removeOrderBump(sanitize_text_field($_GET['delete_bump']));
        wp_redirect(admin_url('admin.php?page=cartt-checkout'));
        exit;
    }
}

// Handle save
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['cartt_checkout_nonce'])) {
    if (wp_verify_nonce($_POST['cartt_checkout_nonce'], 'cartt_save_checkout')) {
        
        // Save new bump
        if (!empty($_POST['bump_product_id'])) {
            \Cartt\Services\CheckoutConfig::addOrderBump([
                'product_id' => intval($_POST['bump_product_id']),
                'title' => sanitize_text_field($_POST['bump_title'] ?? ''),
                'description' => sanitize_text_field($_POST['bump_description'] ?? ''),
                'discount_type' => sanitize_text_field($_POST['bump_discount_type'] ?? 'percent'),
                'discount_value' => floatval($_POST['bump_discount_value'] ?? 0),
                'enabled' => true,
            ]);
        }
        
        $newConfig = [
            'layout' => sanitize_text_field($_POST['layout'] ?? 'one-page'),
            'style' => sanitize_text_field($_POST['style'] ?? 'modern'),
            'guest_checkout' => isset($_POST['guest_checkout']),
            'email_first' => isset($_POST['email_first']),
            'address_autocomplete' => isset($_POST['address_autocomplete']),
            'google_places_key' => sanitize_text_field($_POST['google_places_key'] ?? ''),
            'colors' => [
                'button' => sanitize_hex_color($_POST['color_button'] ?? '#1d1d1f'),
                'button_text' => sanitize_hex_color($_POST['color_button_text'] ?? '#ffffff'),
                'accent' => sanitize_hex_color($_POST['color_accent'] ?? '#1d1d1f'),
                'border' => sanitize_hex_color($_POST['color_border'] ?? '#d2d2d7'),
            ],
            'express_payments' => [
                'enabled' => isset($_POST['express_enabled']),
                'apple_pay' => isset($_POST['apple_pay']),
                'google_pay' => isset($_POST['google_pay']),
                'link' => isset($_POST['stripe_link']),
            ],
            'fields' => $fields, // Will be updated via JS/API
            'order_bumps' => $config['order_bumps'] ?? [], // Preserve bumps
        ];
        
        // Update field settings from POST
        if (!empty($_POST['fields']) && is_array($_POST['fields'])) {
            foreach ($_POST['fields'] as $fieldId => $fieldData) {
                if (isset($fields[$fieldId])) {
                    $fields[$fieldId]['enabled'] = isset($fieldData['enabled']);
                    $fields[$fieldId]['required'] = isset($fieldData['required']);
                    if (!empty($fieldData['label'])) {
                        $fields[$fieldId]['label'] = sanitize_text_field($fieldData['label']);
                    }
                    if (isset($fieldData['width'])) {
                        $fields[$fieldId]['width'] = sanitize_text_field($fieldData['width']);
                    }
                }
            }
            $newConfig['fields'] = $fields;
        }

        // Update field order
        if (!empty($_POST['field_order'])) {
            $order = explode(',', sanitize_text_field($_POST['field_order']));
            $i = 0;
            foreach ($order as $fieldId) {
                if (isset($newConfig['fields'][$fieldId])) {
                    $newConfig['fields'][$fieldId]['order'] = $i++;
                }
            }
        }

        \Cartt\Services\CheckoutConfig::save(array_merge($config, $newConfig));
        $config = \Cartt\Services\CheckoutConfig::get();
        $fields = $config['fields'];
        
        echo '<div class="notice notice-success is-dismissible"><p>' . esc_html__('Checkout settings saved.', 'cartt') . '</p></div>';
    }
}

// Get abandoned cart stats
$abandonedService = new \Cartt\Services\AbandonedCartService();
$abandonedStats = $abandonedService->getStats();

// Sort fields by order
uasort($fields, fn($a, $b) => ($a['order'] ?? 0) - ($b['order'] ?? 0));

$checkoutPageId = get_option('cartt_checkout_page_id');
$previewUrl = $checkoutPageId ? get_permalink($checkoutPageId) : home_url('/checkout/');
?>

<div class="wrap cartt-wrap">
    <div class="cartt-header">
        <h1><?php esc_html_e('Checkout Builder', 'cartt'); ?></h1>
        <div class="cartt-header-actions">
            <a href="<?php echo esc_url($previewUrl); ?>" target="_blank" class="button">
                <?php esc_html_e('Preview', 'cartt'); ?>
            </a>
        </div>
    </div>

    <form method="post" id="cartt-checkout-form">
        <?php wp_nonce_field('cartt_save_checkout', 'cartt_checkout_nonce'); ?>
        <input type="hidden" name="field_order" id="field-order" value="">

        <div class="cartt-checkout-builder">
            <div class="cartt-builder-main">
                <!-- Layout Selection -->
                <div class="cartt-builder-section">
                    <h2><?php esc_html_e('Layout', 'cartt'); ?></h2>
                    <div class="cartt-layout-options">
                        <label class="cartt-layout-option <?php echo ($config['layout'] ?? 'one-page') === 'one-page' ? 'active' : ''; ?>">
                            <input type="radio" name="layout" value="one-page" <?php checked($config['layout'] ?? 'one-page', 'one-page'); ?>>
                            <div class="cartt-layout-preview">
                                <div class="cartt-layout-icon">
                                    <svg width="48" height="48" viewBox="0 0 48 48" fill="none" stroke="currentColor" stroke-width="1.5">
                                        <rect x="8" y="6" width="32" height="36" rx="2"/>
                                        <line x1="14" y1="14" x2="34" y2="14"/>
                                        <line x1="14" y1="22" x2="34" y2="22"/>
                                        <line x1="14" y1="30" x2="26" y2="30"/>
                                    </svg>
                                </div>
                                <span class="cartt-layout-name"><?php esc_html_e('One Page', 'cartt'); ?></span>
                                <span class="cartt-layout-desc"><?php esc_html_e('Everything on single page', 'cartt'); ?></span>
                            </div>
                        </label>
                        
                        <label class="cartt-layout-option <?php echo ($config['layout'] ?? '') === 'multi-step' ? 'active' : ''; ?>">
                            <input type="radio" name="layout" value="multi-step" <?php checked($config['layout'] ?? '', 'multi-step'); ?>>
                            <div class="cartt-layout-preview">
                                <div class="cartt-layout-icon">
                                    <svg width="48" height="48" viewBox="0 0 48 48" fill="none" stroke="currentColor" stroke-width="1.5">
                                        <rect x="4" y="10" width="12" height="28" rx="1"/>
                                        <rect x="18" y="10" width="12" height="28" rx="1"/>
                                        <rect x="32" y="10" width="12" height="28" rx="1"/>
                                        <circle cx="10" cy="38" r="2" fill="currentColor"/>
                                        <circle cx="24" cy="38" r="2"/>
                                        <circle cx="38" cy="38" r="2"/>
                                    </svg>
                                </div>
                                <span class="cartt-layout-name"><?php esc_html_e('Multi-Step', 'cartt'); ?></span>
                                <span class="cartt-layout-desc"><?php esc_html_e('Guided step-by-step flow', 'cartt'); ?></span>
                            </div>
                        </label>
                        
                        <label class="cartt-layout-option <?php echo ($config['layout'] ?? '') === 'express' ? 'active' : ''; ?>">
                            <input type="radio" name="layout" value="express" <?php checked($config['layout'] ?? '', 'express'); ?>>
                            <div class="cartt-layout-preview">
                                <div class="cartt-layout-icon">
                                    <svg width="48" height="48" viewBox="0 0 48 48" fill="none" stroke="currentColor" stroke-width="1.5">
                                        <rect x="8" y="12" width="32" height="24" rx="2"/>
                                        <path d="M8 20h32"/>
                                        <line x1="14" y1="28" x2="22" y2="28"/>
                                    </svg>
                                </div>
                                <span class="cartt-layout-name"><?php esc_html_e('Express', 'cartt'); ?></span>
                                <span class="cartt-layout-desc"><?php esc_html_e('Minimal fields, fast checkout', 'cartt'); ?></span>
                            </div>
                        </label>
                    </div>
                </div>

                <!-- Style Templates with Live Preview and Colors -->
                <div class="cartt-builder-section">
                    <h2><?php esc_html_e('Style', 'cartt'); ?></h2>
                    
                    <div class="cartt-style-container">
                        <div class="cartt-style-controls">
                            <div class="cartt-style-presets">
                                <label class="cartt-style-option <?php echo ($config['style'] ?? 'modern') === 'modern' ? 'active' : ''; ?>">
                                    <input type="radio" name="style" value="modern" <?php checked($config['style'] ?? 'modern', 'modern'); ?>>
                                    <span><?php esc_html_e('Modern', 'cartt'); ?></span>
                                </label>
                                <label class="cartt-style-option <?php echo ($config['style'] ?? '') === 'minimal' ? 'active' : ''; ?>">
                                    <input type="radio" name="style" value="minimal" <?php checked($config['style'] ?? '', 'minimal'); ?>>
                                    <span><?php esc_html_e('Minimal', 'cartt'); ?></span>
                                </label>
                                <label class="cartt-style-option <?php echo ($config['style'] ?? '') === 'classic' ? 'active' : ''; ?>">
                                    <input type="radio" name="style" value="classic" <?php checked($config['style'] ?? '', 'classic'); ?>>
                                    <span><?php esc_html_e('Classic', 'cartt'); ?></span>
                                </label>
                                <label class="cartt-style-option <?php echo ($config['style'] ?? '') === 'bold' ? 'active' : ''; ?>">
                                    <input type="radio" name="style" value="bold" <?php checked($config['style'] ?? '', 'bold'); ?>>
                                    <span><?php esc_html_e('Bold', 'cartt'); ?></span>
                                </label>
                            </div>
                            
                            <div class="cartt-color-controls">
                                <div class="cartt-color-row">
                                    <label>
                                        <span><?php esc_html_e('Button', 'cartt'); ?></span>
                                        <input type="color" name="color_button" id="color-button"
                                               value="<?php echo esc_attr($config['colors']['button'] ?? '#1d1d1f'); ?>">
                                    </label>
                                    <label>
                                        <span><?php esc_html_e('Button Text', 'cartt'); ?></span>
                                        <input type="color" name="color_button_text" id="color-button-text"
                                               value="<?php echo esc_attr($config['colors']['button_text'] ?? '#ffffff'); ?>">
                                    </label>
                                </div>
                                <div class="cartt-color-row">
                                    <label>
                                        <span><?php esc_html_e('Accent', 'cartt'); ?></span>
                                        <input type="color" name="color_accent" id="color-accent"
                                               value="<?php echo esc_attr($config['colors']['accent'] ?? '#1d1d1f'); ?>">
                                    </label>
                                    <label>
                                        <span><?php esc_html_e('Border', 'cartt'); ?></span>
                                        <input type="color" name="color_border" id="color-border"
                                               value="<?php echo esc_attr($config['colors']['border'] ?? '#d2d2d7'); ?>">
                                    </label>
                                </div>
                            </div>
                        </div>
                        
                        <div class="cartt-style-preview-panel" id="style-preview">
                            <div class="cartt-preview-form cartt-preview-<?php echo esc_attr($config['style'] ?? 'modern'); ?>" id="preview-form">
                                <!-- Preview will be dynamically generated from fields -->
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Field Editor -->
                <div class="cartt-builder-section">
                    <h2><?php esc_html_e('Fields', 'cartt'); ?></h2>
                    <p class="cartt-section-desc"><?php esc_html_e('Drag to reorder. Toggle visibility and requirements.', 'cartt'); ?></p>
                    
                    <div class="cartt-fields-list" id="cartt-fields-list">
                        <?php 
                        $currentSection = '';
                        foreach ($fields as $fieldId => $field): 
                            if ($field['section'] !== $currentSection):
                                if ($currentSection !== '') echo '</div>';
                                $currentSection = $field['section'];
                                $sectionLabels = [
                                    'contact' => __('Contact', 'cartt'),
                                    'billing' => __('Billing Address', 'cartt'),
                                    'shipping' => __('Shipping', 'cartt'),
                                    'additional' => __('Additional', 'cartt'),
                                ];
                        ?>
                        <div class="cartt-field-section-header"><?php echo esc_html($sectionLabels[$currentSection] ?? $currentSection); ?></div>
                        <div class="cartt-field-section" data-section="<?php echo esc_attr($currentSection); ?>">
                        <?php endif; ?>
                        
                        <div class="cartt-field-item <?php echo ($field['enabled'] ?? false) ? '' : 'disabled'; ?> <?php echo ($field['locked'] ?? false) ? 'locked' : ''; ?>" 
                             data-field-id="<?php echo esc_attr($fieldId); ?>">
                            <div class="cartt-field-drag">
                                <svg width="14" height="14" viewBox="0 0 24 24" fill="currentColor">
                                    <circle cx="9" cy="6" r="1.5"/><circle cx="15" cy="6" r="1.5"/>
                                    <circle cx="9" cy="12" r="1.5"/><circle cx="15" cy="12" r="1.5"/>
                                    <circle cx="9" cy="18" r="1.5"/><circle cx="15" cy="18" r="1.5"/>
                                </svg>
                            </div>
                            
                            <div class="cartt-field-info">
                                <input type="text" 
                                       name="fields[<?php echo esc_attr($fieldId); ?>][label]" 
                                       value="<?php echo esc_attr($field['label']); ?>"
                                       class="cartt-field-label-input">
                            </div>
                            
                            <?php 
                            // Only email is fully locked (required + visible cannot be changed)
                            $fullyLocked = ($fieldId === 'email');
                            ?>
                            <div class="cartt-field-controls">
                                <select name="fields[<?php echo esc_attr($fieldId); ?>][width]" class="cartt-field-width">
                                    <option value="full" <?php selected($field['width'] ?? 'full', 'full'); ?>>Full</option>
                                    <option value="half" <?php selected($field['width'] ?? '', 'half'); ?>>Half</option>
                                </select>
                                
                                <label class="cartt-field-checkbox">
                                    <input type="checkbox" 
                                           name="fields[<?php echo esc_attr($fieldId); ?>][required]" 
                                           value="1"
                                           <?php checked($field['required'] ?? false); ?>
                                           <?php echo $fullyLocked ? 'disabled' : ''; ?>>
                                    <span>Required</span>
                                </label>
                                
                                <label class="cartt-field-checkbox">
                                    <input type="checkbox" 
                                           name="fields[<?php echo esc_attr($fieldId); ?>][enabled]" 
                                           value="1"
                                           <?php checked($field['enabled'] ?? false); ?>
                                           <?php echo $fullyLocked ? 'disabled' : ''; ?>>
                                    <span>Visible</span>
                                </label>
                            </div>
                        </div>
                        <?php endforeach; ?>
                        </div><!-- Close last section -->
                    </div>
                    
                    <button type="button" class="button" id="add-custom-field">
                        <?php esc_html_e('+ Add Custom Field', 'cartt'); ?>
                    </button>
                </div>

                <!-- Express Payments + Address Autocomplete Row -->
                <div class="cartt-builder-row">
                    <div class="cartt-builder-section cartt-half">
                        <h2><?php esc_html_e('Express Payments', 'cartt'); ?></h2>
                        
                        <label class="cartt-toggle-row">
                            <input type="checkbox" name="express_enabled" id="express-enabled" <?php checked($config['express_payments']['enabled'] ?? false); ?>>
                            <span><?php esc_html_e('Enable', 'cartt'); ?></span>
                        </label>
                        
                        <div class="cartt-express-methods" id="express-methods" style="margin-left: 20px; margin-top: 8px; <?php echo empty($config['express_payments']['enabled']) ? 'display:none;' : ''; ?>">
                            <label class="cartt-toggle-row">
                                <input type="checkbox" name="apple_pay" <?php checked($config['express_payments']['apple_pay'] ?? false); ?>>
                                <span><?php esc_html_e('Apple Pay', 'cartt'); ?></span>
                            </label>
                            <label class="cartt-toggle-row">
                                <input type="checkbox" name="google_pay" <?php checked($config['express_payments']['google_pay'] ?? false); ?>>
                                <span><?php esc_html_e('Google Pay', 'cartt'); ?></span>
                            </label>
                            <label class="cartt-toggle-row">
                                <input type="checkbox" name="stripe_link" <?php checked($config['express_payments']['link'] ?? false); ?>>
                                <span><?php esc_html_e('Link', 'cartt'); ?></span>
                            </label>
                            <p class="cartt-form-hint" style="margin-top:4px;"><?php esc_html_e('Requires Stripe.', 'cartt'); ?></p>
                        </div>
                    </div>

                    <div class="cartt-builder-section cartt-half">
                        <h2><?php esc_html_e('Address Autocomplete', 'cartt'); ?></h2>
                        
                        <label class="cartt-toggle-row">
                            <input type="checkbox" name="address_autocomplete" <?php checked($config['address_autocomplete'] ?? true); ?>>
                            <span><?php esc_html_e('Enable Google Places address suggestions', 'cartt'); ?></span>
                        </label>
                        <p class="cartt-form-hint"><?php esc_html_e('Powered by Cartt - no API key needed.', 'cartt'); ?></p>
                    </div>
                </div>
            </div>

            <!-- Sidebar -->
            <div class="cartt-builder-sidebar">
                <div class="cartt-sidebar-panel">
                    <div class="cartt-sidebar-panel-header"><?php esc_html_e('Settings', 'cartt'); ?></div>
                    <div class="cartt-sidebar-panel-content">
                        <label class="cartt-toggle-row">
                            <input type="checkbox" name="guest_checkout" <?php checked($config['guest_checkout'] ?? true); ?>>
                            <span><?php esc_html_e('Allow guest checkout', 'cartt'); ?></span>
                        </label>
                        <label class="cartt-toggle-row">
                            <input type="checkbox" name="email_first" <?php checked($config['email_first'] ?? true); ?>>
                            <span><?php esc_html_e('Capture email first', 'cartt'); ?></span>
                        </label>
                        <p class="cartt-form-hint"><?php esc_html_e('Enables abandoned cart recovery.', 'cartt'); ?></p>
                    </div>
                </div>

                <div class="cartt-sidebar-panel">
                    <div class="cartt-sidebar-panel-header"><?php esc_html_e('Abandoned Cart Recovery', 'cartt'); ?></div>
                    <div class="cartt-sidebar-panel-content">
                        <div class="cartt-stat-grid">
                            <div class="cartt-mini-stat">
                                <span class="cartt-mini-stat-value"><?php echo esc_html($abandonedStats['total_abandoned']); ?></span>
                                <span class="cartt-mini-stat-label"><?php esc_html_e('Abandoned', 'cartt'); ?></span>
                            </div>
                            <div class="cartt-mini-stat">
                                <span class="cartt-mini-stat-value"><?php echo esc_html($abandonedStats['total_recovered']); ?></span>
                                <span class="cartt-mini-stat-label"><?php esc_html_e('Recovered', 'cartt'); ?></span>
                            </div>
                            <div class="cartt-mini-stat">
                                <span class="cartt-mini-stat-value"><?php echo esc_html($abandonedStats['recovery_rate']); ?>%</span>
                                <span class="cartt-mini-stat-label"><?php esc_html_e('Rate', 'cartt'); ?></span>
                            </div>
                            <div class="cartt-mini-stat">
                                <span class="cartt-mini-stat-value">$<?php echo esc_html(number_format($abandonedStats['total_value_recovered'], 0)); ?></span>
                                <span class="cartt-mini-stat-label"><?php esc_html_e('Recovered $', 'cartt'); ?></span>
                            </div>
                        </div>
                    </div>
                </div>

                <div class="cartt-sidebar-panel">
                    <div class="cartt-sidebar-panel-header"><?php esc_html_e('Order Bumps', 'cartt'); ?></div>
                    <div class="cartt-sidebar-panel-content">
                        <?php 
                        $orderBumps = \Cartt\Services\CheckoutConfig::getOrderBumps();
                        if (!empty($orderBumps)): 
                        ?>
                        <div class="cartt-bumps-list">
                            <?php foreach ($orderBumps as $bumpId => $bump): 
                                $bumpProduct = $wpdb->get_row($wpdb->prepare(
                                    "SELECT name, price FROM {$wpdb->prefix}cartt_products WHERE id = %d",
                                    $bump['product_id']
                                ));
                            ?>
                            <div class="cartt-bump-item">
                                <div class="cartt-bump-info">
                                    <strong><?php echo esc_html($bump['title'] ?: ($bumpProduct->name ?? 'Unknown')); ?></strong>
                                    <?php if ($bump['discount_value'] > 0): ?>
                                    <span class="cartt-bump-discount">
                                        <?php echo $bump['discount_type'] === 'percent' 
                                            ? esc_html($bump['discount_value'] . '% off')
                                            : esc_html('$' . $bump['discount_value'] . ' off'); ?>
                                    </span>
                                    <?php endif; ?>
                                </div>
                                <a href="<?php echo esc_url(wp_nonce_url(admin_url('admin.php?page=cartt-checkout&delete_bump=' . $bumpId), 'delete_bump_' . $bumpId)); ?>" 
                                   class="cartt-bump-delete" title="<?php esc_attr_e('Delete', 'cartt'); ?>">&times;</a>
                            </div>
                            <?php endforeach; ?>
                        </div>
                        <?php endif; ?>
                        
                        <button type="button" class="button" id="add-bump-btn" style="width:100%; margin-top: 8px;">
                            <?php esc_html_e('+ Add Bump', 'cartt'); ?>
                        </button>
                        
                        <div id="bump-form" style="display:none; margin-top: 12px;">
                            <div class="cartt-form-row">
                                <label><?php esc_html_e('Product', 'cartt'); ?></label>
                                <select name="bump_product_id" id="bump-product">
                                    <option value=""><?php esc_html_e('Select...', 'cartt'); ?></option>
                                    <?php foreach ($products as $p): ?>
                                    <option value="<?php echo esc_attr($p->id); ?>">
                                        <?php echo esc_html($p->name); ?> - $<?php echo esc_html(number_format($p->sale_price ?: $p->price, 2)); ?>
                                    </option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                            <div class="cartt-form-row">
                                <label><?php esc_html_e('Title', 'cartt'); ?></label>
                                <input type="text" name="bump_title" placeholder="<?php esc_attr_e('Add this to your order!', 'cartt'); ?>">
                            </div>
                            <div class="cartt-form-row">
                                <label><?php esc_html_e('Description', 'cartt'); ?></label>
                                <input type="text" name="bump_description" placeholder="<?php esc_attr_e('Special one-time offer', 'cartt'); ?>">
                            </div>
                            <div class="cartt-form-row-half">
                                <div class="cartt-form-row">
                                    <label><?php esc_html_e('Discount', 'cartt'); ?></label>
                                    <select name="bump_discount_type">
                                        <option value="percent"><?php esc_html_e('Percent', 'cartt'); ?></option>
                                        <option value="fixed"><?php esc_html_e('Fixed $', 'cartt'); ?></option>
                                    </select>
                                </div>
                                <div class="cartt-form-row">
                                    <label><?php esc_html_e('Amount', 'cartt'); ?></label>
                                    <input type="number" name="bump_discount_value" value="0" min="0" step="1">
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

                <div class="cartt-sidebar-panel">
                    <button type="submit" class="button button-primary" style="width: 100%;">
                        <?php esc_html_e('Save Changes', 'cartt'); ?>
                    </button>
                </div>
            </div>
        </div>
    </form>
</div>

<style>
.cartt-checkout-builder { display: grid; grid-template-columns: 1fr 280px; gap: 20px; margin-top: 20px; }
.cartt-builder-row { display: grid; grid-template-columns: 1fr 1fr; gap: 16px; }
.cartt-builder-section { background: #fff; border: 1px solid #e8e8ed; border-radius: 8px; padding: 20px; margin-bottom: 16px; }
.cartt-builder-section.cartt-half { margin-bottom: 0; }
.cartt-builder-section h2 { font-size: 14px; font-weight: 600; margin: 0 0 12px; }
.cartt-section-desc { color: #86868b; font-size: 12px; margin: 0 0 16px; }

/* Layout Options */
.cartt-layout-options { display: grid; grid-template-columns: repeat(3, 1fr); gap: 12px; }
.cartt-layout-option { cursor: pointer; }
.cartt-layout-option input { display: none; }
.cartt-layout-preview { border: 2px solid #e8e8ed; border-radius: 8px; padding: 20px; text-align: center; transition: all 0.15s; }
.cartt-layout-option.active .cartt-layout-preview,
.cartt-layout-option input:checked + .cartt-layout-preview { border-color: #1d1d1f; background: #fafafa; }
.cartt-layout-preview:hover { border-color: #1d1d1f; }
.cartt-layout-icon { color: #86868b; margin-bottom: 12px; }
.cartt-layout-name { display: block; font-weight: 600; font-size: 13px; }
.cartt-layout-desc { display: block; font-size: 11px; color: #86868b; margin-top: 4px; }

/* Style Container */
.cartt-style-container { display: grid; grid-template-columns: 220px 1fr; gap: 20px; }
.cartt-style-controls { display: flex; flex-direction: column; gap: 16px; }
.cartt-style-presets { display: flex; flex-direction: column; gap: 4px; }
.cartt-style-option { cursor: pointer; display: flex; align-items: center; gap: 8px; padding: 8px 12px; border-radius: 6px; border: 1px solid transparent; font-size: 13px; font-weight: 500; }
.cartt-style-option input { display: none; }
.cartt-style-option:hover { background: #f5f5f7; }
.cartt-style-option.active { background: #f5f5f7; border-color: #1d1d1f; }

/* Color Controls */
.cartt-color-controls { border-top: 1px solid #e8e8ed; padding-top: 12px; }
.cartt-color-row { display: flex; gap: 12px; margin-bottom: 8px; }
.cartt-color-row label { display: flex; align-items: center; gap: 6px; font-size: 11px; color: #666; cursor: pointer; }
.cartt-color-row input[type="color"] { width: 28px; height: 28px; padding: 0; border: 1px solid #d2d2d7; border-radius: 4px; cursor: pointer; }

/* Live Preview Panel */
.cartt-style-preview-panel { background: #f5f5f7; border-radius: 8px; padding: 20px; min-height: 200px; }
.cartt-preview-form { max-width: 320px; }
.preview-input { background: #fff; border: 1px solid #d2d2d7; border-radius: 6px; padding: 10px 12px; margin-bottom: 8px; font-size: 12px; color: #86868b; }
.preview-input.half { display: inline-block; width: calc(50% - 4px); box-sizing: border-box; }
.preview-input.half:nth-child(odd) { margin-right: 8px; }
.preview-row { display: flex; gap: 8px; margin-bottom: 8px; }
.preview-row .preview-input { flex: 1; margin-bottom: 0; }
.preview-btn { background: #1d1d1f; color: #fff; border-radius: 6px; padding: 12px; text-align: center; font-size: 13px; font-weight: 600; margin-top: 12px; }
/* Preview: Minimal */
.cartt-preview-minimal .preview-input { background: transparent; border: none; border-bottom: 1px solid #d2d2d7; border-radius: 0; padding: 10px 0; }
.cartt-preview-minimal .preview-btn { background: transparent; color: #1d1d1f; border: 2px solid #1d1d1f; }
/* Preview: Classic */
.cartt-preview-classic .preview-input { border-radius: 0; }
.cartt-preview-classic .preview-btn { border-radius: 0; }
/* Preview: Bold */
.cartt-preview-bold .preview-input { background: #e8e8ed; border: none; border-radius: 10px; padding: 12px 14px; }
.cartt-preview-bold .preview-btn { border-radius: 10px; padding: 14px; font-size: 14px; }

/* Fields List - Two Column */
.cartt-field-section-header { font-size: 11px; font-weight: 600; text-transform: uppercase; color: #86868b; margin: 16px 0 8px; letter-spacing: 0.5px; }
.cartt-field-section-header:first-child { margin-top: 0; }
.cartt-field-section { display: grid; grid-template-columns: 1fr 1fr; gap: 8px; }
.cartt-field-item { display: flex; align-items: center; gap: 10px; padding: 10px 12px; background: #f5f5f7; border-radius: 6px; cursor: grab; }
.cartt-field-item.disabled { opacity: 0.5; }
.cartt-field-item.locked .cartt-field-drag { opacity: 0.3; cursor: not-allowed; }
.cartt-field-drag { color: #86868b; flex-shrink: 0; cursor: grab; }
.cartt-field-info { flex: 1; min-width: 0; }
.cartt-field-label-input { background: transparent; border: 1px solid transparent; font-size: 13px; font-weight: 500; padding: 4px 8px; width: 100%; border-radius: 4px; cursor: text; }
.cartt-field-label-input:hover { border-color: #d2d2d7; background: #fff; }
.cartt-field-label-input:focus { outline: none; background: #fff; border-color: #1d1d1f; }
.cartt-field-controls { display: flex; align-items: center; gap: 10px; flex-shrink: 0; }
.cartt-field-width { font-size: 12px; padding: 5px 8px; border: 1px solid #d2d2d7; border-radius: 4px; background: #fff; cursor: pointer; }
.cartt-field-checkbox { display: flex; align-items: center; gap: 4px; font-size: 12px; color: #1d1d1f; cursor: pointer; white-space: nowrap; }
.cartt-field-checkbox input[type="checkbox"] { width: 14px; height: 14px; margin: 0; cursor: pointer; }
.cartt-field-checkbox span { cursor: pointer; }

/* Toggle Row */
.cartt-toggle-row { display: flex; align-items: center; gap: 8px; padding: 6px 0; cursor: pointer; font-size: 13px; }
.cartt-toggle-row input { margin: 0; }

/* Stats */
.cartt-stat-grid { display: grid; grid-template-columns: 1fr 1fr; gap: 8px; }
.cartt-mini-stat { background: #f5f5f7; border-radius: 6px; padding: 12px; text-align: center; }
.cartt-mini-stat-value { display: block; font-size: 18px; font-weight: 600; }
.cartt-mini-stat-label { display: block; font-size: 10px; color: #86868b; text-transform: uppercase; margin-top: 2px; }

/* Sortable */
.cartt-field-item.sortable-ghost { opacity: 0.4; }
.cartt-field-item.sortable-drag { background: #fff; box-shadow: 0 4px 12px rgba(0,0,0,0.15); }

/* Bumps List */
.cartt-bumps-list { margin-bottom: 8px; }
.cartt-bump-item { display: flex; align-items: center; gap: 8px; padding: 8px 10px; background: #f5f5f7; border-radius: 6px; margin-bottom: 6px; }
.cartt-bump-info { flex: 1; min-width: 0; }
.cartt-bump-info strong { display: block; font-size: 12px; font-weight: 500; }
.cartt-bump-discount { display: block; font-size: 10px; color: #2e7d32; }
.cartt-bump-delete { color: #c62828; text-decoration: none; font-size: 18px; line-height: 1; padding: 2px 4px; }
.cartt-bump-delete:hover { color: #b71c1c; }

/* Bump Form */
#bump-form .cartt-form-row { margin-bottom: 10px; }
#bump-form .cartt-form-row label { display: block; font-size: 11px; font-weight: 500; margin-bottom: 4px; }
#bump-form .cartt-form-row input,
#bump-form .cartt-form-row select { width: 100%; padding: 6px 8px; font-size: 12px; border: 1px solid #d2d2d7; border-radius: 4px; }
#bump-form .cartt-form-row-half { display: grid; grid-template-columns: 1fr 1fr; gap: 8px; }

@media (max-width: 900px) {
    .cartt-checkout-builder { grid-template-columns: 1fr; }
    .cartt-layout-options { grid-template-columns: 1fr; }
    .cartt-style-container { grid-template-columns: 1fr; }
    .cartt-style-presets { flex-direction: row; flex-wrap: wrap; }
    .cartt-color-row { flex-wrap: wrap; }
    .cartt-builder-row { grid-template-columns: 1fr; }
    .cartt-builder-section.cartt-half { margin-bottom: 16px; }
}
</style>

<script src="https://cdn.jsdelivr.net/npm/sortablejs@1.15.0/Sortable.min.js"></script>
<script>
jQuery(document).ready(function($) {
    // Layout option toggle
    $('.cartt-layout-option input').on('change', function() {
        $('.cartt-layout-option').removeClass('active');
        $(this).closest('.cartt-layout-option').addClass('active');
    });

    // Style option toggle with live preview
    $('.cartt-style-option input').on('change', function() {
        $('.cartt-style-option').removeClass('active');
        $(this).closest('.cartt-style-option').addClass('active');
        updateStylePreview();
    });
    
    // Color controls
    $('#color-button, #color-button-text, #color-accent, #color-border').on('input', function() {
        updatePreviewColors();
    });
    
    // Update preview when fields change
    $(document).on('change', '.cartt-field-checkbox input, .cartt-field-width', function() {
        updateStylePreview();
    });
    
    function updateStylePreview() {
        var style = $('input[name="style"]:checked').val() || 'modern';
        var preview = $('#preview-form');
        
        preview.removeClass('cartt-preview-modern cartt-preview-minimal cartt-preview-classic cartt-preview-bold');
        preview.addClass('cartt-preview-' + style);
        
        // Build preview from visible fields
        var html = '';
        var pendingHalf = null;
        
        $('.cartt-field-item').each(function() {
            var $item = $(this);
            var isVisible = $item.find('input[name*="[enabled]"]').is(':checked');
            if (!isVisible) return;
            
            var label = $item.find('.cartt-field-label-input').val();
            var width = $item.find('.cartt-field-width').val();
            
            if (width === 'half') {
                if (pendingHalf) {
                    html += '<div class="preview-row">';
                    html += '<div class="preview-input"><span>' + escapeHtml(pendingHalf) + '</span></div>';
                    html += '<div class="preview-input"><span>' + escapeHtml(label) + '</span></div>';
                    html += '</div>';
                    pendingHalf = null;
                } else {
                    pendingHalf = label;
                }
            } else {
                if (pendingHalf) {
                    html += '<div class="preview-row">';
                    html += '<div class="preview-input"><span>' + escapeHtml(pendingHalf) + '</span></div>';
                    html += '<div class="preview-input"><span></span></div>';
                    html += '</div>';
                    pendingHalf = null;
                }
                html += '<div class="preview-input"><span>' + escapeHtml(label) + '</span></div>';
            }
        });
        
        // Handle trailing half-width field
        if (pendingHalf) {
            html += '<div class="preview-row">';
            html += '<div class="preview-input"><span>' + escapeHtml(pendingHalf) + '</span></div>';
            html += '<div class="preview-input"><span></span></div>';
            html += '</div>';
        }
        
        html += '<div class="preview-btn" id="preview-btn">Pay $99.00</div>';
        
        preview.html(html);
        updatePreviewColors();
    }
    
    function updatePreviewColors() {
        var btnColor = $('#color-button').val();
        var btnTextColor = $('#color-button-text').val();
        var borderColor = $('#color-border').val();
        
        $('#preview-btn').css({
            'background-color': btnColor,
            'color': btnTextColor
        });
        
        // For minimal style, button is outline
        var style = $('input[name="style"]:checked').val();
        if (style === 'minimal') {
            $('#preview-btn').css({
                'background-color': 'transparent',
                'color': btnColor,
                'border-color': btnColor
            });
        }
        
        $('#preview-form .preview-input').css('border-color', borderColor);
    }
    
    function escapeHtml(text) {
        var div = document.createElement('div');
        div.textContent = text;
        return div.innerHTML;
    }
    
    // Initial preview
    updateStylePreview();

    // Sortable fields
    document.querySelectorAll('.cartt-field-section').forEach(function(section) {
        new Sortable(section, {
            group: 'fields',
            animation: 150,
            handle: '.cartt-field-drag',
            ghostClass: 'sortable-ghost',
            dragClass: 'sortable-drag',
            filter: '.locked',
            onEnd: function() {
                updateFieldOrder();
            }
        });
    });

    function updateFieldOrder() {
        var order = [];
        $('.cartt-field-item').each(function() {
            order.push($(this).data('field-id'));
        });
        $('#field-order').val(order.join(','));
    }

    // Toggle enabled state
    $('.cartt-field-item').on('change', 'input[name$="[enabled]"]', function() {
        $(this).closest('.cartt-field-item').toggleClass('disabled', !this.checked);
    });

    // Add custom field
    $('#add-custom-field').on('click', function() {
        var fieldId = 'custom_' + Date.now();
        var html = `
            <div class="cartt-field-item" data-field-id="${fieldId}">
                <div class="cartt-field-drag">
                    <svg width="16" height="16" viewBox="0 0 24 24" fill="currentColor">
                        <circle cx="9" cy="6" r="1.5"/><circle cx="15" cy="6" r="1.5"/>
                        <circle cx="9" cy="12" r="1.5"/><circle cx="15" cy="12" r="1.5"/>
                        <circle cx="9" cy="18" r="1.5"/><circle cx="15" cy="18" r="1.5"/>
                    </svg>
                </div>
                <div class="cartt-field-info">
                    <input type="text" name="fields[${fieldId}][label]" value="Custom Field" class="cartt-field-label-input">
                    <span class="cartt-field-type">text</span>
                </div>
                <div class="cartt-field-controls">
                    <select name="fields[${fieldId}][width]" class="cartt-field-width">
                        <option value="full">Full</option>
                        <option value="half">Half</option>
                        <option value="third">Third</option>
                    </select>
                    <label class="cartt-field-toggle">
                        <input type="checkbox" name="fields[${fieldId}][required]">
                        <span>Req</span>
                    </label>
                    <label class="cartt-field-toggle">
                        <input type="checkbox" name="fields[${fieldId}][enabled]" checked>
                        <span>Show</span>
                    </label>
                    <button type="button" class="button-link cartt-remove-field" style="color:#c62828;">×</button>
                </div>
            </div>
        `;
        $('[data-section="additional"]').append(html);
        updateFieldOrder();
    });

    // Remove custom field
    $(document).on('click', '.cartt-remove-field', function() {
        $(this).closest('.cartt-field-item').remove();
        updateFieldOrder();
    });

    // Order bump form toggle
    $('#add-bump-btn').on('click', function() {
        var form = $('#bump-form');
        if (form.is(':visible')) {
            form.slideUp(150);
            $(this).text('+ Add Bump');
        } else {
            form.slideDown(150);
            $(this).text('Cancel');
        }
    });

    // Express payments toggle
    $('#express-enabled').on('change', function() {
        if ($(this).is(':checked')) {
            $('#express-methods').slideDown(150);
        } else {
            $('#express-methods').slideUp(150);
        }
    });

    // Initialize order on load
    updateFieldOrder();
});
</script>
