<?php
/**
 * Chat Inbox Admin View
 * @since 1.4.2
 */

if (!defined('ABSPATH')) {
    exit;
}

$includeCartDefault = get_option('cartt_chat_include_cart_default', '0') === '1';
?>

<div class="wrap cartt-admin">
    <h1 class="wp-heading-inline">Chat Inbox</h1>
    <a href="<?php echo admin_url('admin.php?page=cartt-chat-settings'); ?>" class="page-title-action" style="background:#1d2327;color:#fff;border-color:#1d2327;">Settings</a>
    <hr class="wp-header-end">

    <div class="cartt-chat-inbox">
        <!-- Conversations List -->
        <div class="cartt-chat-list">
            <div class="cartt-chat-list-header">
                <span>Conversations</span>
                <button type="button" class="button button-small" id="cartt-refresh-list" title="Refresh">
                    <span class="dashicons dashicons-update"></span>
                </button>
            </div>
            <div class="cartt-chat-conversations" id="cartt-conversations">
                <?php if (empty($conversations)): ?>
                <div class="cartt-chat-no-conversations">
                    <span class="dashicons dashicons-format-chat"></span>
                    <p>No conversations yet</p>
                    <p class="description">When visitors start chatting, they'll appear here.</p>
                </div>
                <?php else: ?>
                    <?php foreach ($conversations as $conv): 
                        $isUrgent = !empty($conv->handoff_status) && in_array($conv->handoff_status, ['pending', 'followup_sent']);
                    ?>
                    <div class="cartt-chat-conv-item <?php echo $conv->unread_count > 0 ? 'unread' : ''; ?> <?php echo $isUrgent ? 'urgent' : ''; ?>" 
                         data-id="<?php echo esc_attr($conv->id); ?>">
                        <div class="cartt-chat-avatar">
                            <?php echo esc_html(strtoupper(substr($conv->visitor_name ?: 'V', 0, 1))); ?>
                        </div>
                        <div class="cartt-chat-conv-info">
                            <div class="cartt-chat-conv-name">
                                <?php if ($isUrgent): ?><span class="cartt-urgent-badge">URGENT</span><?php endif; ?>
                                <?php echo esc_html($conv->visitor_name ?: 'Visitor'); ?>
                                <span class="cartt-chat-conv-time"><?php echo esc_html(human_time_diff(strtotime($conv->updated_at))); ?></span>
                            </div>
                            <div class="cartt-chat-conv-preview"><?php echo esc_html(wp_trim_words($conv->last_message ?? '', 8)); ?></div>
                        </div>
                        <?php if ($conv->unread_count > 0): ?>
                        <span class="cartt-chat-unread-badge"><?php echo esc_html($conv->unread_count); ?></span>
                        <?php endif; ?>
                    </div>
                    <?php endforeach; ?>
                <?php endif; ?>
            </div>
        </div>

        <!-- Chat Panel -->
        <div class="cartt-chat-panel">
            <div class="cartt-chat-empty" id="cartt-chat-empty">
                <span class="dashicons dashicons-format-chat"></span>
                <p>Select a conversation to start chatting</p>
            </div>
            
            <div class="cartt-chat-view" id="cartt-chat-view" style="display:none;">
                <div class="cartt-chat-view-header">
                    <div class="cartt-chat-visitor-info">
                        <h3 id="cartt-visitor-name">Visitor</h3>
                        <span id="cartt-visitor-status" class="cartt-status-badge"></span>
                    </div>
                    <div class="cartt-chat-header-actions">
                        <button class="button" id="cartt-close-conv">Close Chat</button>
                    </div>
                </div>
                
                <div class="cartt-chat-messages" id="cartt-messages"></div>
                
                <div class="cartt-chat-compose">
                    <textarea id="cartt-reply-message" placeholder="Type your reply..." rows="3"></textarea>
                    <div class="cartt-chat-compose-actions">
                        <div class="cartt-chat-send-options">
                            <label class="cartt-checkbox" id="cartt-sms-option" style="display:none;">
                                <input type="checkbox" id="cartt-send-sms"> SMS
                            </label>
                            <label class="cartt-checkbox" id="cartt-whatsapp-option" style="display:none;">
                                <input type="checkbox" id="cartt-send-whatsapp"> WhatsApp
                            </label>
                            <label class="cartt-checkbox" id="cartt-cart-option" style="display:none;">
                                <input type="checkbox" id="cartt-include-cart" <?php echo $includeCartDefault ? 'checked' : ''; ?>> Include Cart (+$0.05)
                            </label>
                        </div>
                        <button class="button button-primary" id="cartt-send-reply">Send Reply</button>
                    </div>
                </div>
            </div>
        </div>

        <!-- Sidebar (Visitor Info & Cart) -->
        <div class="cartt-chat-sidebar" id="cartt-chat-sidebar" style="display:none;">
            <div class="cartt-chat-sidebar-section">
                <h4>Contact Info</h4>
                <div class="cartt-info-row">
                    <span class="dashicons dashicons-email"></span>
                    <span id="cartt-visitor-email">-</span>
                </div>
                <div class="cartt-info-row">
                    <span class="dashicons dashicons-phone"></span>
                    <span id="cartt-visitor-phone">-</span>
                </div>
                <div class="cartt-info-row">
                    <span class="dashicons dashicons-location"></span>
                    <span id="cartt-visitor-ip">-</span>
                </div>
            </div>
            
            <div class="cartt-chat-sidebar-section">
                <h4>Current Cart</h4>
                <div id="cartt-visitor-cart" class="cartt-cart-contents">
                    <p class="cartt-empty-cart">No items</p>
                </div>
            </div>
            
            <div class="cartt-chat-sidebar-section">
                <h4>Browsing</h4>
                <div class="cartt-info-row">
                    <span class="dashicons dashicons-admin-links"></span>
                    <a href="#" id="cartt-visitor-url" target="_blank">-</a>
                </div>
            </div>
            
            <div class="cartt-chat-sidebar-section">
                <h4>Quick Actions</h4>
                <?php
                $quickActions = get_option('cartt_chat_quick_actions', [
                    'Is there anything else I can help you with?',
                    'I\'ll check on that for you right away.',
                    'Thank you for contacting us! Have a great day.'
                ]);
                if (empty($quickActions)) {
                    echo '<p style="color:#666;font-size:12px;">No quick actions configured. <a href="' . admin_url('admin.php?page=cartt-chat-settings') . '">Add some</a></p>';
                } else {
                    foreach ($quickActions as $action) {
                        echo '<button class="button button-small cartt-quick-action" data-message="' . esc_attr($action) . '">' . esc_html(wp_trim_words($action, 3, '...')) . '</button>';
                    }
                }
                ?>
            </div>
        </div>
    </div>
</div>

<script>
jQuery(document).ready(function($) {
    var currentConvId = null;
    var currentConv = null;
    var pollInterval = null;
    var nonce = '<?php echo wp_create_nonce('cartt_chat_admin'); ?>';
    
    // Open conversation
    $(document).on('click', '.cartt-chat-conv-item', function() {
        var id = $(this).data('id');
        openConversation(id);
    });
    
    function openConversation(id) {
        currentConvId = id;
        
        $('.cartt-chat-conv-item').removeClass('active');
        $('.cartt-chat-conv-item[data-id="' + id + '"]').addClass('active').removeClass('unread');
        $('.cartt-chat-conv-item[data-id="' + id + '"] .cartt-chat-unread-badge').remove();
        
        $('#cartt-chat-empty').hide();
        $('#cartt-chat-view').show();
        $('#cartt-chat-sidebar').show();
        
        loadConversation(id);
        startPolling();
    }
    
    function loadConversation(id) {
        $.post(ajaxurl, {
            action: 'cartt_chat_get_conversation',
            conversation_id: id,
            nonce: nonce
        }, function(response) {
            if (response.success) {
                currentConv = response.data.conversation;
                renderConversation(response.data.conversation, response.data.messages);
            }
        });
    }
    
    function renderConversation(conv, messages) {
        // Header
        $('#cartt-visitor-name').text(conv.visitor_name || 'Visitor');
        $('#cartt-visitor-status').text(conv.status).attr('class', 'cartt-status-badge status-' + conv.status);
        
        // Contact info
        $('#cartt-visitor-email').text(conv.visitor_email || '-');
        $('#cartt-visitor-phone').text(conv.visitor_phone || '-');
        $('#cartt-visitor-ip').text(conv.ip_address || '-');
        
        // Show SMS/WhatsApp options if phone available
        if (conv.visitor_phone) {
            $('#cartt-sms-option, #cartt-whatsapp-option, #cartt-cart-option').show();
        } else {
            $('#cartt-sms-option, #cartt-whatsapp-option, #cartt-cart-option').hide();
        }
        
        // Page URL
        if (conv.page_url) {
            $('#cartt-visitor-url').attr('href', conv.page_url).text(conv.page_url.replace(/^https?:\/\//, '').substring(0, 30) + '...');
        } else {
            $('#cartt-visitor-url').attr('href', '#').text('-');
        }
        
        // Cart
        renderCart(conv.cart_data);
        
        // Messages
        renderMessages(messages);
    }
    
    function renderCart(cartJson) {
        var $container = $('#cartt-visitor-cart');
        
        if (!cartJson) {
            $container.html('<p class="cartt-empty-cart">No cart data</p>');
            return;
        }
        
        try {
            var cart = typeof cartJson === 'string' ? JSON.parse(cartJson) : cartJson;
            
            if (!cart.items || cart.items.length === 0) {
                $container.html('<p class="cartt-empty-cart">Cart is empty</p>');
                return;
            }
            
            var html = '<div class="cartt-cart-items">';
            cart.items.forEach(function(item) {
                html += '<div class="cartt-cart-item">' +
                    '<span class="cartt-cart-qty">' + item.qty + 'x</span>' +
                    '<span class="cartt-cart-name">' + escapeHtml(item.name) + '</span>' +
                    '</div>';
            });
            html += '</div>';
            html += '<div class="cartt-cart-total"><strong>Total:</strong> ' + (cart.total_formatted || '$' + parseFloat(cart.total || 0).toFixed(2)) + '</div>';
            
            $container.html(html);
        } catch (e) {
            $container.html('<p class="cartt-empty-cart">Error loading cart</p>');
        }
    }
    
    function renderMessages(messages) {
        var html = '';
        messages.forEach(function(msg) {
            var time = formatTime(msg.created_at);
            var isViaSms = msg.message.indexOf('[Via sms]') === 0 || msg.message.indexOf('[Via whatsapp]') === 0;
            var messageText = msg.message.replace(/^\[Via (sms|whatsapp)\]\s*/i, '');
            
            // Check for image
            var imageMatch = messageText.match(/\[image\](.*?)\[\/image\]/);
            var fileMatch = messageText.match(/\[file\](.*?)\|(.*?)\[\/file\]/);
            var messageContent = '';
            
            if (imageMatch) {
                messageContent = '<img src="' + imageMatch[1] + '" style="max-width:250px;border-radius:8px;cursor:pointer;" onclick="window.open(this.src)">';
            } else if (fileMatch) {
                messageContent = '<a href="' + fileMatch[1] + '" target="_blank" style="display:flex;align-items:center;gap:8px;color:inherit;text-decoration:none;"><span class="dashicons dashicons-media-default"></span>' + escapeHtml(fileMatch[2]) + '</a>';
            } else {
                messageContent = escapeHtml(messageText);
            }
            
            html += '<div class="cartt-chat-message ' + (msg.sender || msg.sender_type) + (isViaSms ? ' via-external' : '') + '">' +
                '<div class="cartt-chat-message-bubble">' +
                messageContent +
                '</div>' +
                '<div class="cartt-chat-message-meta">' +
                (isViaSms ? '<span class="cartt-via-badge">' + (msg.message.match(/\[Via (\w+)\]/i)?.[1] || 'SMS') + '</span>' : '') +
                '<span class="cartt-chat-message-time">' + time + '</span>' +
                '</div>' +
                '</div>';
        });
        
        $('#cartt-messages').html(html);
        scrollToBottom();
    }
    
    function scrollToBottom() {
        var $container = $('#cartt-messages');
        $container.scrollTop($container[0].scrollHeight);
    }
    
    function formatTime(datetime) {
        var date = new Date(datetime);
        return date.toLocaleTimeString([], {hour: '2-digit', minute:'2-digit'});
    }
    
    function escapeHtml(text) {
        var div = document.createElement('div');
        div.textContent = text;
        return div.innerHTML;
    }
    
    function startPolling() {
        stopPolling();
        pollInterval = setInterval(function() {
            if (currentConvId) {
                loadConversation(currentConvId);
            }
        }, 5000);
    }
    
    function stopPolling() {
        if (pollInterval) {
            clearInterval(pollInterval);
            pollInterval = null;
        }
    }
    
    // Send reply
    $('#cartt-send-reply').on('click', function() {
        sendReply();
    });
    
    $('#cartt-reply-message').on('keydown', function(e) {
        if (e.key === 'Enter' && !e.shiftKey) {
            e.preventDefault();
            sendReply();
        }
    });
    
    function sendReply() {
        var message = $('#cartt-reply-message').val().trim();
        if (!message || !currentConvId) return;
        
        var $btn = $('#cartt-send-reply');
        $btn.prop('disabled', true).text('Sending...');
        
        $.post(ajaxurl, {
            action: 'cartt_chat_admin_reply',
            conversation_id: currentConvId,
            message: message,
            send_sms: $('#cartt-send-sms').is(':checked') ? '1' : '0',
            send_whatsapp: $('#cartt-send-whatsapp').is(':checked') ? '1' : '0',
            include_cart: $('#cartt-include-cart').is(':checked') ? '1' : '0',
            nonce: nonce
        }, function(response) {
            $btn.prop('disabled', false).text('Send Reply');
            
            if (response.success) {
                $('#cartt-reply-message').val('');
                loadConversation(currentConvId);
                
                // Show notification if SMS/WhatsApp sent
                var notifications = [];
                if (response.data.sms_sent) notifications.push('SMS sent');
                if (response.data.whatsapp_sent) notifications.push('WhatsApp sent');
                if (notifications.length) {
                    showNotice(notifications.join(', '), 'success');
                }
            } else {
                showNotice(response.data || 'Failed to send', 'error');
            }
        }).fail(function() {
            $btn.prop('disabled', false).text('Send Reply');
            showNotice('Network error', 'error');
        });
    }
    
    // Quick actions
    $('.cartt-quick-action').on('click', function() {
        var msg = $(this).data('message');
        $('#cartt-reply-message').val(msg).focus();
    });
    
    // Close conversation
    $('#cartt-close-conv').on('click', function() {
        if (!currentConvId || !confirm('Close this conversation?')) return;
        
        $.post(ajaxurl, {
            action: 'cartt_chat_close',
            conversation_id: currentConvId,
            nonce: nonce
        }, function(response) {
            if (response.success) {
                location.reload();
            }
        });
    });
    
    // Refresh list
    $('#cartt-refresh-list').on('click', function() {
        location.reload();
    });
    
    // Simple notice
    function showNotice(message, type) {
        var $notice = $('<div class="cartt-notice ' + type + '">' + escapeHtml(message) + '</div>');
        $('body').append($notice);
        setTimeout(function() { $notice.addClass('show'); }, 10);
        setTimeout(function() { $notice.removeClass('show'); }, 3000);
        setTimeout(function() { $notice.remove(); }, 3500);
    }
    
    // Auto-refresh conversations list every 30 seconds
    setInterval(function() {
        $.post(ajaxurl, {
            action: 'cartt_chat_get_conversations',
            nonce: nonce
        }, function(response) {
            if (response.success) {
                updateConversationsList(response.data.conversations);
            }
        });
    }, 30000);
    
    function updateConversationsList(conversations) {
        var $list = $('#cartt-conversations');
        
        if (conversations.length === 0) {
            return;
        }
        
        var html = '';
        conversations.forEach(function(conv) {
            var isActive = currentConvId == conv.id;
            var isUnread = conv.unread_count > 0 && !isActive;
            var isUrgent = conv.handoff_status && (conv.handoff_status === 'pending' || conv.handoff_status === 'followup_sent');
            
            html += '<div class="cartt-chat-conv-item ' + (isActive ? 'active' : '') + ' ' + (isUnread ? 'unread' : '') + ' ' + (isUrgent ? 'urgent' : '') + '" data-id="' + conv.id + '">' +
                '<div class="cartt-chat-avatar">' + (conv.visitor_name || 'V').charAt(0).toUpperCase() + '</div>' +
                '<div class="cartt-chat-conv-info">' +
                '<div class="cartt-chat-conv-name">' + 
                (isUrgent ? '<span class="cartt-urgent-badge">URGENT</span>' : '') +
                escapeHtml(conv.visitor_name || 'Visitor') +
                '<span class="cartt-chat-conv-time">' + conv.time_ago + '</span></div>' +
                '<div class="cartt-chat-conv-preview">' + escapeHtml((conv.last_message || '').substring(0, 40)) + '</div>' +
                '</div>' +
                (isUnread ? '<span class="cartt-chat-unread-badge">' + conv.unread_count + '</span>' : '') +
                '</div>';
        });
        
        $list.html(html);
    }
});
</script>
