<?php
/**
 * Chat Settings Admin View
 * @since 1.4.3
 */

if (!defined('ABSPATH')) {
    exit;
}

// Check if pro features are available
$chatService = \Cartt\Services\ChatService::instance();
$hasPro = $chatService->hasProFeatures();

// Get all settings
$settings = [
    // Basic
    'enabled' => get_option('cartt_chat_enabled', '1'),
    'title' => get_option('cartt_chat_title', 'Chat with us'),
    'greeting' => get_option('cartt_chat_greeting', 'Hi! How can we help you today?'),
    'online_text' => get_option('cartt_chat_online_text', 'Online now'),
    'position' => get_option('cartt_chat_position', 'right'),
    'avatar' => get_option('cartt_chat_avatar', ''),
    
    // Display Options
    'show_online' => get_option('cartt_chat_show_online', '1'),
    'sound_enabled' => get_option('cartt_chat_sound_enabled', '1'),
    'screenshots' => get_option('cartt_chat_screenshots', '1'),
    'attachments' => get_option('cartt_chat_attachments', '1'),
    
    // Colors
    'color_header' => get_option('cartt_chat_color_header', '#000000'),
    'color_header_text' => get_option('cartt_chat_color_header_text', '#ffffff'),
    'color_visitor_bubble' => get_option('cartt_chat_color_visitor_bubble', '#000000'),
    'color_agent_bubble' => get_option('cartt_chat_color_agent_bubble', '#f0f0f0'),
    'color_background' => get_option('cartt_chat_color_background', '#f5f5f5'),
    'color_input_border' => get_option('cartt_chat_color_input_border', '#e0e0e0'),
    'color_input_bg' => get_option('cartt_chat_color_input_bg', '#fafafa'),
    'color_input_field' => get_option('cartt_chat_color_input_field', '#ffffff'),
    'input_border_width' => get_option('cartt_chat_input_border_width', '1'),
    'color_icons' => get_option('cartt_chat_color_icons', '#666666'),
    'color_send_icon' => get_option('cartt_chat_color_send_icon', '#000000'),
    
    // Pre-Chat Form
    'collect_name' => get_option('cartt_chat_collect_name', '0'),
    'collect_email' => get_option('cartt_chat_collect_email', '0'),
    'collect_phone' => get_option('cartt_chat_collect_phone', '0'),
    
    // Email
    'notify_email' => get_option('cartt_chat_notify_email', '1'),
    'email_address' => get_option('cartt_chat_email_address', get_option('admin_email')),
    'email_include_cart' => get_option('cartt_chat_email_include_cart', '1'),
    'email_frequency' => get_option('cartt_chat_email_frequency', 'every'), // 'every' or 'first'
    
    // SMS
    'notify_sms' => get_option('cartt_chat_notify_sms', '0'),
    'sms_number' => get_option('cartt_chat_sms_number', ''),
    'sms_include_cart' => get_option('cartt_chat_sms_include_cart', '0'),
    
    // WhatsApp
    'notify_whatsapp' => get_option('cartt_chat_notify_whatsapp', '0'),
    'whatsapp_number' => get_option('cartt_chat_whatsapp_number', ''),
    'whatsapp_include_cart' => get_option('cartt_chat_whatsapp_include_cart', '0'),
    
    // AI
    'ai_enabled' => get_option('cartt_chat_ai_enabled', '0'),
    'ai_delay' => get_option('cartt_chat_ai_delay', '30'),
    'ai_store_info' => get_option('cartt_chat_ai_store_info', ''),
    
    // Quick Actions
    'quick_actions' => get_option('cartt_chat_quick_actions', [
        'Thanks for reaching out!',
        'Let me check on that for you.',
        'Is there anything else I can help with?'
    ]),
];

$nonce = wp_create_nonce('cartt_chat_admin');
?>

<link href="https://fonts.googleapis.com/icon?family=Material+Icons" rel="stylesheet">

<style>
.cartt-chat-settings {
    max-width: 900px;
    margin: 20px 0;
}

.cartt-settings-section {
    background: #fff;
    border: 1px solid #e0e0e0;
    border-radius: 8px;
    margin-bottom: 20px;
    padding: 24px;
}

.cartt-settings-section h2 {
    margin: 0 0 8px;
    font-size: 16px;
    font-weight: 600;
    display: flex;
    align-items: center;
    gap: 10px;
}

.cartt-settings-section > p {
    margin: 0 0 20px;
    color: #666;
    font-size: 13px;
}

.cartt-settings-section hr {
    border: 0;
    border-top: 1px solid #f0f0f0;
    margin: 0 0 20px;
}

.cartt-two-col {
    display: grid;
    grid-template-columns: 1fr 1fr;
    gap: 40px;
}

@media (max-width: 782px) {
    .cartt-two-col {
        grid-template-columns: 1fr;
        gap: 20px;
    }
}

.cartt-form-row {
    margin-bottom: 20px;
}

.cartt-form-row:last-child {
    margin-bottom: 0;
}

.cartt-form-row > label {
    display: block;
    font-weight: 500;
    margin-bottom: 8px;
    font-size: 13px;
}

.cartt-form-row input[type="text"],
.cartt-form-row input[type="email"],
.cartt-form-row input[type="tel"],
.cartt-form-row input[type="number"],
.cartt-form-row textarea,
.cartt-form-row select {
    width: 100%;
    max-width: 400px;
    border-radius: 4px;
    border: 1px solid #8c8f94;
    padding: 6px 10px;
    font-size: 14px;
}

.cartt-form-row textarea {
    min-height: 120px;
    max-width: 100%;
}

.cartt-form-row .description {
    margin-top: 6px;
    font-size: 12px;
    color: #666;
}

.cartt-checkbox-row {
    display: flex;
    align-items: center;
    gap: 8px;
    margin-bottom: 16px;
    font-size: 13px;
}

.cartt-checkbox-row input[type="checkbox"] {
    margin: 0;
}

.cartt-inline-row {
    display: flex;
    align-items: center;
    gap: 10px;
}

.cartt-inline-row input[type="text"],
.cartt-inline-row input[type="tel"],
.cartt-inline-row input[type="email"] {
    max-width: 250px;
}

.cartt-inline-row .button {
    white-space: nowrap;
}

.cartt-color-row {
    display: flex;
    align-items: center;
    justify-content: space-between;
    margin-bottom: 16px;
    max-width: 300px;
}

.cartt-color-row label {
    font-size: 13px;
}

.cartt-color-row input[type="color"] {
    width: 50px;
    height: 30px;
    padding: 0;
    border: 1px solid #ddd;
    border-radius: 4px;
    cursor: pointer;
}

.cartt-color-row select {
    width: 80px;
    padding: 4px 8px;
}

.cartt-color-row input[type="number"] {
    width: 60px;
    padding: 4px 8px;
}

.cartt-avatar-preview {
    display: flex;
    align-items: center;
    gap: 12px;
    margin-top: 8px;
}

.cartt-avatar-preview img {
    width: 40px;
    height: 40px;
    border-radius: 50%;
    object-fit: cover;
}

.cartt-avatar-preview .cartt-default-icon {
    width: 40px;
    height: 40px;
    background: #f0f0f0;
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
}

.cartt-avatar-preview .cartt-default-icon svg {
    width: 24px;
    height: 24px;
    fill: #666;
}

.cartt-badge {
    display: inline-block;
    padding: 2px 8px;
    border-radius: 3px;
    font-size: 10px;
    font-weight: 600;
    text-transform: uppercase;
}

.cartt-badge-pro {
    background: #1d2327;
    color: #fff;
}

.cartt-badge-business {
    background: #1d2327;
    color: #fff;
}

.cartt-locked-section {
    opacity: 0.6;
    pointer-events: none;
}

.cartt-locked-section.unlocked {
    opacity: 1;
    pointer-events: auto;
}

.cartt-upgrade-notice {
    color: #666;
    font-size: 13px;
    margin-top: 8px;
}

.cartt-save-row {
    margin-top: 20px;
    padding-top: 20px;
    border-top: 1px solid #f0f0f0;
}

.cartt-save-row .button-primary {
    padding: 6px 24px;
}

.cartt-notice {
    padding: 10px 14px;
    border-radius: 4px;
    margin-bottom: 16px;
    font-size: 13px;
}

.cartt-notice.success {
    background: #d4edda;
    color: #155724;
    border: 1px solid #c3e6cb;
}

.cartt-notice.error {
    background: #f8d7da;
    color: #721c24;
    border: 1px solid #f5c6cb;
}

.cartt-help-icon {
    display: inline-flex;
    align-items: center;
    justify-content: center;
    width: 16px;
    height: 16px;
    background: #ddd;
    border-radius: 50%;
    font-size: 11px;
    color: #666;
    cursor: help;
    margin-left: 4px;
}
</style>

<div class="wrap cartt-chat-settings">
    <h1 style="display:flex;align-items:center;gap:20px;">Chat Settings <button type="button" class="button button-primary" id="cartt-save-settings-top">Save Settings</button></h1>
    <p style="margin-bottom:20px;"><a href="<?php echo admin_url('admin.php?page=cartt-chat'); ?>">&larr; Back to Inbox</a></p>
    
    <div id="cartt-settings-notice"></div>
    
    <div class="cartt-two-col">
        <!-- Left Column -->
        <div>
            <!-- Basic Settings -->
            <div class="cartt-settings-section <?php echo !$hasPro ? 'cartt-locked-section' : ''; ?>">
                <h2>Widget Customization <?php if (!$hasPro): ?><span class="cartt-badge cartt-badge-pro">PRO</span><?php endif; ?></h2>
                <?php if (!$hasPro): ?>
                <p class="cartt-upgrade-notice"><a href="<?php echo admin_url('admin.php?page=cartt-addons'); ?>">Activate Chat addon</a> to customize the widget appearance.</p>
                <?php endif; ?>
                <hr>
                
                <div class="cartt-form-row">
                    <label for="cartt-chat-title">Chat Title</label>
                    <input type="text" id="cartt-chat-title" value="<?php echo esc_attr($settings['title']); ?>" <?php echo !$hasPro ? 'disabled' : ''; ?>>
                </div>
                
                <div class="cartt-form-row">
                    <label for="cartt-chat-greeting">Greeting Message</label>
                    <textarea id="cartt-chat-greeting" <?php echo !$hasPro ? 'disabled' : ''; ?>><?php echo esc_textarea($settings['greeting']); ?></textarea>
                </div>
                
                <div class="cartt-form-row">
                    <label for="cartt-online-text">Online Status Text</label>
                    <input type="text" id="cartt-online-text" value="<?php echo esc_attr($settings['online_text']); ?>" <?php echo !$hasPro ? 'disabled' : ''; ?>>
                </div>
                
                <div class="cartt-form-row">
                    <label for="cartt-position">Widget Position</label>
                    <select id="cartt-position" <?php echo !$hasPro ? 'disabled' : ''; ?>>
                        <option value="right" <?php selected($settings['position'], 'right'); ?>>Bottom Right</option>
                        <option value="left" <?php selected($settings['position'], 'left'); ?>>Bottom Left</option>
                    </select>
                </div>
                
                <div class="cartt-form-row">
                    <label>Avatar Image</label>
                    <button type="button" class="button" id="cartt-select-avatar" <?php echo !$hasPro ? 'disabled' : ''; ?>>Select Image</button>
                    <div class="cartt-avatar-preview" id="cartt-avatar-preview">
                        <?php if ($settings['avatar']): ?>
                            <img src="<?php echo esc_url($settings['avatar']); ?>">
                            <button type="button" class="button-link" id="cartt-remove-avatar" style="color:#b32d2e;" <?php echo !$hasPro ? 'disabled' : ''; ?>>Remove</button>
                        <?php else: ?>
                            <div class="cartt-default-icon">
                                <span class="material-icons" style="font-size:32px;color:#666;">support_agent</span>
                            </div>
                            <span style="color:#888;font-size:12px;">Default icon</span>
                        <?php endif; ?>
                    </div>
                    <input type="hidden" id="cartt-avatar" value="<?php echo esc_attr($settings['avatar']); ?>">
                </div>
            </div>
            
            <!-- Display Options -->
            <div class="cartt-settings-section <?php echo !$hasPro ? 'cartt-locked-section' : ''; ?>">
                <h2>Display Options <?php if (!$hasPro): ?><span class="cartt-badge cartt-badge-pro">PRO</span><?php endif; ?></h2>
                <hr>
                
                <label class="cartt-checkbox-row">
                    <input type="checkbox" id="cartt-show-online" <?php checked($settings['show_online'], '1'); ?> <?php echo !$hasPro ? 'disabled' : ''; ?>>
                    Show Online Indicator
                </label>
                
                <label class="cartt-checkbox-row">
                    <input type="checkbox" id="cartt-sound-enabled" <?php checked($settings['sound_enabled'], '1'); ?> <?php echo !$hasPro ? 'disabled' : ''; ?>>
                    Sound Notifications
                </label>
                
                <label class="cartt-checkbox-row">
                    <input type="checkbox" id="cartt-screenshots" <?php checked($settings['screenshots'], '1'); ?> <?php echo !$hasPro ? 'disabled' : ''; ?>>
                    Enable screenshots in chat
                </label>
                
                <label class="cartt-checkbox-row">
                    <input type="checkbox" id="cartt-attachments" <?php checked($settings['attachments'], '1'); ?> <?php echo !$hasPro ? 'disabled' : ''; ?>>
                    Enable file attachments
                </label>
            </div>
            
            <!-- Pre-Chat Form -->
            <div class="cartt-settings-section <?php echo !$hasPro ? 'cartt-locked-section' : ''; ?>">
                <h2>Pre-Chat Form <?php if (!$hasPro): ?><span class="cartt-badge cartt-badge-pro">PRO</span><?php endif; ?></h2>
                <p>Collect visitor info before chat.</p>
                <hr>
                
                <div style="display:flex;gap:24px;flex-wrap:wrap;">
                    <label class="cartt-checkbox-row" style="margin-bottom:0;">
                        <input type="checkbox" id="cartt-collect-name" <?php checked($settings['collect_name'], '1'); ?> <?php echo !$hasPro ? 'disabled' : ''; ?>>
                        Name
                    </label>
                    
                    <label class="cartt-checkbox-row" style="margin-bottom:0;">
                        <input type="checkbox" id="cartt-collect-email" <?php checked($settings['collect_email'], '1'); ?> <?php echo !$hasPro ? 'disabled' : ''; ?>>
                        Email
                    </label>
                    
                    <label class="cartt-checkbox-row" style="margin-bottom:0;">
                        <input type="checkbox" id="cartt-collect-phone" <?php checked($settings['collect_phone'], '1'); ?> <?php echo !$hasPro ? 'disabled' : ''; ?>>
                        Phone
                    </label>
                </div>
            </div>
        </div>
        
        <!-- Right Column -->
        <div>
            <!-- Colors -->
            <div class="cartt-settings-section <?php echo !$hasPro ? 'cartt-locked-section' : ''; ?>">
                <h2>Colors <?php if (!$hasPro): ?><span class="cartt-badge cartt-badge-pro">PRO</span><?php endif; ?></h2>
                <hr>
                
                <div class="cartt-color-row">
                    <label>Header / Button</label>
                    <input type="color" id="cartt-color-header" value="<?php echo esc_attr($settings['color_header']); ?>" <?php echo !$hasPro ? 'disabled' : ''; ?>>
                </div>
                
                <div class="cartt-color-row">
                    <label>Header Text</label>
                    <input type="color" id="cartt-color-header-text" value="<?php echo esc_attr($settings['color_header_text']); ?>" <?php echo !$hasPro ? 'disabled' : ''; ?>>
                </div>
                
                <div class="cartt-color-row">
                    <label>Visitor Bubble</label>
                    <input type="color" id="cartt-color-visitor-bubble" value="<?php echo esc_attr($settings['color_visitor_bubble']); ?>" <?php echo !$hasPro ? 'disabled' : ''; ?>>
                </div>
                
                <div class="cartt-color-row">
                    <label>Agent Bubble</label>
                    <input type="color" id="cartt-color-agent-bubble" value="<?php echo esc_attr($settings['color_agent_bubble']); ?>" <?php echo !$hasPro ? 'disabled' : ''; ?>>
                </div>
                
                <div class="cartt-color-row">
                    <label>Chat Background</label>
                    <input type="color" id="cartt-color-background" value="<?php echo esc_attr($settings['color_background']); ?>" <?php echo !$hasPro ? 'disabled' : ''; ?>>
                </div>
                
                <div class="cartt-color-row">
                    <label>Input Border Color</label>
                    <input type="color" id="cartt-color-input-border" value="<?php echo esc_attr($settings['color_input_border']); ?>" <?php echo !$hasPro ? 'disabled' : ''; ?>>
                </div>
                
                <div class="cartt-color-row">
                    <label>Input Border Width, px</label>
                    <input type="number" id="cartt-input-border-width" value="<?php echo esc_attr($settings['input_border_width']); ?>" min="0" max="5" style="width:50px;text-align:center;" <?php echo !$hasPro ? 'disabled' : ''; ?>>
                </div>
                <p class="description" style="margin-top:-12px;margin-bottom:16px;">Set to 0 to remove border</p>
                
                <div class="cartt-color-row">
                    <label>Input Area Background</label>
                    <input type="color" id="cartt-color-input-bg" value="<?php echo esc_attr($settings['color_input_bg'] ?? '#fafafa'); ?>" <?php echo !$hasPro ? 'disabled' : ''; ?>>
                </div>
                
                <div class="cartt-color-row">
                    <label>Input Field Background</label>
                    <input type="color" id="cartt-color-input-field" value="<?php echo esc_attr($settings['color_input_field'] ?? '#ffffff'); ?>" <?php echo !$hasPro ? 'disabled' : ''; ?>>
                </div>
                
                <div class="cartt-color-row">
                    <label>Icon Color<br><small>(Screenshot/Attach)</small></label>
                    <input type="color" id="cartt-color-icons" value="<?php echo esc_attr($settings['color_icons']); ?>" <?php echo !$hasPro ? 'disabled' : ''; ?>>
                </div>
                
                <div class="cartt-color-row">
                    <label>Send Icon Color</label>
                    <input type="color" id="cartt-color-send-icon" value="<?php echo esc_attr($settings['color_send_icon']); ?>" <?php echo !$hasPro ? 'disabled' : ''; ?>>
                </div>
                
                <!-- Live Preview -->
                <div style="margin-top:20px;padding-top:20px;border-top:1px solid #e0e0e0;display:flex;flex-direction:column;align-items:center;">
                    <h3 style="margin:0 0 12px;font-size:13px;color:#666;align-self:flex-start;">Preview</h3>
                    <div id="cartt-chat-preview" style="width:320px;border-radius:8px;overflow:hidden;box-shadow:0 4px 20px rgba(0,0,0,0.15);font-family:-apple-system,BlinkMacSystemFont,'Segoe UI',Roboto,sans-serif;">
                        <div id="preview-header" style="padding:16px 18px;display:flex;align-items:center;gap:12px;">
                            <div id="preview-avatar" style="width:40px;height:40px;border-radius:50%;background:rgba(255,255,255,0.15);display:flex;align-items:center;justify-content:center;">
                                <span class="material-icons" style="color:#fff;font-size:22px;">support_agent</span>
                            </div>
                            <div>
                                <div id="preview-title" style="font-weight:600;font-size:15px;">Chat with us</div>
                                <div id="preview-status" style="font-size:12px;opacity:0.8;display:flex;align-items:center;gap:4px;"><span style="width:7px;height:7px;background:#4ade80;border-radius:50%;"></span> Online now</div>
                            </div>
                        </div>
                        <div id="preview-messages" style="padding:14px;min-height:200px;background:#fff;">
                            <div id="preview-greeting" style="padding:12px 14px;border-radius:6px;border-bottom-left-radius:0;font-size:14px;max-width:85%;margin-bottom:10px;">Hi! How can we help you today?</div>
                            <div id="preview-visitor-msg" style="padding:12px 14px;border-radius:6px;border-bottom-right-radius:0;font-size:14px;max-width:85%;margin-left:auto;margin-bottom:10px;text-align:right;">I have a question about shipping</div>
                        </div>
                        <div id="preview-input-area" style="padding:12px 14px;border-top:1px solid #e0e0e0;">
                            <div style="display:flex;gap:10px;align-items:center;">
                                <div id="preview-input" style="flex:1;padding:10px 14px;border-radius:6px;font-size:13px;color:#999;display:flex;align-items:center;justify-content:space-between;">
                                    <span>Type a message...</span>
                                    <div style="display:flex;gap:8px;">
                                        <span class="material-icons" id="preview-screenshot-icon" style="font-size:18px;color:#888;cursor:pointer;">screenshot_monitor</span>
                                        <span class="material-icons" id="preview-attach-icon" style="font-size:18px;color:#888;cursor:pointer;">attach_file</span>
                                    </div>
                                </div>
                                <span class="material-icons" id="preview-send-icon" style="font-size:22px;">send</span>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Email Alerts -->
    <div class="cartt-settings-section">
        <h2>Email Alerts</h2>
        <hr>
        
        <label class="cartt-checkbox-row">
            <input type="checkbox" id="cartt-notify-email" <?php checked($settings['notify_email'], '1'); ?>>
            Get email alerts for new messages
        </label>
        
        <div class="cartt-form-row">
            <label>Email Address</label>
            <div class="cartt-inline-row">
                <input type="email" id="cartt-email-address" value="<?php echo esc_attr($settings['email_address']); ?>">
                <button type="button" class="button" id="cartt-test-email">Send Test</button>
            </div>
        </div>
        
        <label class="cartt-checkbox-row">
            <input type="checkbox" id="cartt-email-include-cart" <?php checked($settings['email_include_cart'], '1'); ?>>
            Include cart items in email
            <span class="cartt-help-icon" title="Shows what items the visitor has in their cart">?</span>
        </label>
        
        <div class="cartt-form-row" style="margin-top:12px;">
            <label>Email Frequency</label>
            <select id="cartt-email-frequency" style="max-width:300px;">
                <option value="every" <?php selected($settings['email_frequency'], 'every'); ?>>Every message</option>
                <option value="first" <?php selected($settings['email_frequency'], 'first'); ?>>First message per conversation</option>
            </select>
            <p class="description" style="margin-top:6px;">Choose when to receive email notifications</p>
        </div>
    </div>
    
    <!-- SMS Chat -->
    <div class="cartt-settings-section <?php echo !$hasPro ? 'cartt-locked-section' : ''; ?>">
        <h2>SMS Chat <?php if (!$hasPro): ?><span class="cartt-badge cartt-badge-pro">PRO</span><?php endif; ?></h2>
        <?php if (!$hasPro): ?>
        <p class="cartt-upgrade-notice"><a href="<?php echo admin_url('admin.php?page=cartt-addons'); ?>">Activate Chat addon</a> to chat with visitors via SMS.</p>
        <?php else: ?>
        <p>Receive visitor messages and reply directly from your phone via SMS.</p>
        <?php endif; ?>
        <hr>
        
        <label class="cartt-checkbox-row">
            <input type="checkbox" id="cartt-notify-sms" <?php checked($settings['notify_sms'], '1'); ?> <?php echo !$hasPro ? 'disabled' : ''; ?>>
            Chat with visitors via SMS
            <span class="cartt-help-icon" title="$0.03 per message, $0.05 with cart/screenshot">?</span>
        </label>
        
        <div class="cartt-form-row">
            <label>Phone Number</label>
            <div class="cartt-inline-row">
                <input type="tel" id="cartt-sms-number" value="<?php echo esc_attr($settings['sms_number']); ?>" placeholder="+12345678900" <?php echo !$hasPro ? 'disabled' : ''; ?>>
                <button type="button" class="button" id="cartt-test-sms" <?php echo !$hasPro ? 'disabled' : ''; ?>>Send Test</button>
            </div>
        </div>
        
        <label class="cartt-checkbox-row">
            <input type="checkbox" id="cartt-sms-include-cart" <?php checked($settings['sms_include_cart'], '1'); ?> <?php echo !$hasPro ? 'disabled' : ''; ?>>
            Include cart items in SMS
            <span class="cartt-help-icon" title="Adds cart summary (+$0.02 per message)">?</span>
        </label>
    </div>
    
    <!-- WhatsApp Chat -->
    <div class="cartt-settings-section <?php echo !$hasPro ? 'cartt-locked-section' : ''; ?>">
        <h2>WhatsApp Chat <?php if (!$hasPro): ?><span class="cartt-badge cartt-badge-pro">PRO</span><?php endif; ?></h2>
        <?php if (!$hasPro): ?>
        <p class="cartt-upgrade-notice"><a href="<?php echo admin_url('admin.php?page=cartt-addons'); ?>">Activate Chat addon</a> to chat with visitors via WhatsApp.</p>
        <?php else: ?>
        <p>Receive visitor messages and reply directly from WhatsApp on your phone.</p>
        <?php endif; ?>
        <hr>
        
        <label class="cartt-checkbox-row">
            <input type="checkbox" id="cartt-notify-whatsapp" <?php checked($settings['notify_whatsapp'], '1'); ?> <?php echo !$hasPro ? 'disabled' : ''; ?>>
            Chat with visitors via WhatsApp
            <span class="cartt-help-icon" title="$0.03 per message, $0.05 with cart/screenshot">?</span>
        </label>
        
        <div class="cartt-form-row">
            <label>WhatsApp Number</label>
            <div class="cartt-inline-row">
                <input type="tel" id="cartt-whatsapp-number" value="<?php echo esc_attr($settings['whatsapp_number']); ?>" placeholder="+12345678900" <?php echo !$hasPro ? 'disabled' : ''; ?>>
                <button type="button" class="button" id="cartt-test-whatsapp" <?php echo !$hasPro ? 'disabled' : ''; ?>>Send Test</button>
            </div>
        </div>
        
        <label class="cartt-checkbox-row">
            <input type="checkbox" id="cartt-whatsapp-include-cart" <?php checked($settings['whatsapp_include_cart'], '1'); ?> <?php echo !$hasPro ? 'disabled' : ''; ?>>
            Include cart items in WhatsApp
            <span class="cartt-help-icon" title="Adds cart summary (+$0.02 per message)">?</span>
        </label>
    </div>
    
    <!-- AI Auto-Reply -->
    <div class="cartt-settings-section <?php echo !$hasPro ? 'cartt-locked-section' : ''; ?>">
        <h2>AI Auto-Reply <?php if (!$hasPro): ?><span class="cartt-badge cartt-badge-pro">PRO</span><?php endif; ?></h2>
        <?php if (!$hasPro): ?>
        <p class="cartt-upgrade-notice"><a href="<?php echo admin_url('admin.php?page=cartt-addons'); ?>">Activate Chat addon</a> to enable AI auto-replies.</p>
        <?php else: ?>
        <p>AI will automatically respond to visitors when no agent replies within the delay period.</p>
        <?php endif; ?>
        <hr>
        
        <label class="cartt-checkbox-row">
            <input type="checkbox" id="cartt-ai-enabled" <?php checked($settings['ai_enabled'], '1'); ?> <?php echo !$hasPro ? 'disabled' : ''; ?>>
            Enable AI auto-replies
            <span class="cartt-help-icon" title="$0.01 per AI reply">?</span>
        </label>
        
        <div class="cartt-form-row">
            <label for="cartt-ai-delay">Reply Delay (seconds)</label>
            <input type="number" id="cartt-ai-delay" value="<?php echo esc_attr($settings['ai_delay']); ?>" min="0" max="300" style="width:80px;" <?php echo !$hasPro ? 'disabled' : ''; ?>>
            <p class="description">Wait this many seconds for a human agent before AI responds. Set to 0 for instant AI reply.</p>
        </div>
        
        <div class="cartt-form-row">
            <label for="cartt-ai-store-info">Store Description</label>
            <textarea id="cartt-ai-store-info" placeholder="We sell handmade jewelry and accessories.
Shipping: Free over $50, otherwise $5.99. Ships in 2-3 days. US only.
Returns: 30 days for unused items. Customer pays return shipping.
Hours: Mon-Fri 9am-5pm EST. Weekend messages answered Monday.
Custom orders: Tell them to email hello@example.com" <?php echo !$hasPro ? 'disabled' : ''; ?>><?php echo esc_textarea($settings['ai_store_info']); ?></textarea>
            <p class="description">Describe your store, products, policies, shipping, and FAQs. This helps AI answer accurately.</p>
        </div>
        
        <hr style="margin:20px 0;">
        <h3 style="margin:0 0 16px;font-size:14px;">Handoff Settings</h3>
        
        <div class="cartt-form-row">
            <label for="cartt-handoff-timeout">Follow-up Timeout (seconds)</label>
            <input type="number" id="cartt-handoff-timeout" value="<?php echo esc_attr(get_option('cartt_chat_handoff_timeout', '60')); ?>" min="30" max="300" style="width:80px;" <?php echo !$hasPro ? 'disabled' : ''; ?>>
            <p class="description">If no human responds within this time after AI handoff, ask customer for contact info.</p>
        </div>
        
        <div class="cartt-form-row">
            <label for="cartt-handoff-message">Follow-up Message</label>
            <textarea id="cartt-handoff-message" rows="2" style="width:100%;max-width:600px;" <?php echo !$hasPro ? 'disabled' : ''; ?>><?php echo esc_textarea(get_option('cartt_chat_handoff_message', "Our team members are currently assisting other customers. Please leave your email or phone number and we'll get back to you as soon as possible!")); ?></textarea>
            <p class="description">Message shown to customer when no agent responds in time.</p>
        </div>
    </div>
    
    <!-- Agents -->
    <div class="cartt-settings-section <?php echo !$hasPro ? 'cartt-locked-section' : ''; ?>">
        <h2>Chat Agents <?php if (!$hasPro): ?><span class="cartt-badge cartt-badge-pro">PRO</span><?php endif; ?></h2>
        <?php if (!$hasPro): ?>
        <p class="cartt-upgrade-notice"><a href="<?php echo admin_url('admin.php?page=cartt-addons'); ?>">Activate Chat addon</a> to add multiple agents.</p>
        <?php else: ?>
        <p>Add team members who can receive chat notifications and respond to customers.</p>
        <?php endif; ?>
        <hr>
        
        <div id="cartt-agents-list">
            <!-- Will be populated by JS -->
        </div>
        
        <div style="margin-top:16px;padding-top:16px;border-top:1px solid #e0e0e0;">
            <h4 style="margin:0 0 12px;font-size:13px;">Add Agent</h4>
            <div style="display:grid;grid-template-columns:1fr 1fr;gap:12px;max-width:500px;">
                <div>
                    <label style="display:block;font-size:12px;margin-bottom:4px;">Name</label>
                    <input type="text" id="cartt-new-agent-name" placeholder="Agent name" style="width:100%;" <?php echo !$hasPro ? 'disabled' : ''; ?>>
                </div>
                <div>
                    <label style="display:block;font-size:12px;margin-bottom:4px;">Email</label>
                    <input type="email" id="cartt-new-agent-email" placeholder="agent@example.com" style="width:100%;" <?php echo !$hasPro ? 'disabled' : ''; ?>>
                </div>
                <div>
                    <label style="display:block;font-size:12px;margin-bottom:4px;">SMS Number</label>
                    <input type="text" id="cartt-new-agent-sms" placeholder="+1234567890" style="width:100%;" <?php echo !$hasPro ? 'disabled' : ''; ?>>
                </div>
                <div>
                    <label style="display:block;font-size:12px;margin-bottom:4px;">WhatsApp Number</label>
                    <input type="text" id="cartt-new-agent-whatsapp" placeholder="+1234567890" style="width:100%;" <?php echo !$hasPro ? 'disabled' : ''; ?>>
                </div>
            </div>
            <div style="margin-top:12px;display:flex;align-items:center;gap:12px;">
                <button type="button" class="button" id="cartt-select-agent-avatar" <?php echo !$hasPro ? 'disabled' : ''; ?>>Select Avatar</button>
                <img id="cartt-new-agent-avatar-preview" src="" style="width:32px;height:32px;border-radius:50%;display:none;object-fit:cover;">
                <input type="hidden" id="cartt-new-agent-avatar" value="">
                <button type="button" class="button button-primary" id="cartt-add-agent" <?php echo !$hasPro ? 'disabled' : ''; ?>>Add Agent</button>
            </div>
        </div>
    </div>
    
    <!-- Quick Actions -->
    <div class="cartt-settings-section <?php echo !$hasPro ? 'cartt-locked-section' : ''; ?>">
        <h2>Quick Actions <?php if (!$hasPro): ?><span class="cartt-badge cartt-badge-pro">PRO</span><?php endif; ?></h2>
        <p>Pre-written responses you can send with one click from the chat dashboard.</p>
        <hr>
        
        <div id="cartt-quick-actions-list">
            <!-- Will be populated by JS -->
        </div>
        
        <div class="cartt-add-quick-action" style="display:flex;align-items:center;gap:10px;margin-top:16px;">
            <input type="text" id="cartt-new-quick-action" placeholder="Type a quick action message..." style="flex:1;max-width:500px;" <?php echo !$hasPro ? 'disabled' : ''; ?>>
            <button type="button" class="button" id="cartt-add-quick-action" <?php echo !$hasPro ? 'disabled' : ''; ?>>Add Quick Action</button>
        </div>
    </div>
    
    <!-- Balance & Billing -->
    <?php if ($hasPro): ?>
    <h2 style="margin:30px 0 20px;">Balance & Billing</h2>
    
    <div class="cartt-settings-section">
        <h2>Chat Balance</h2>
        <p>SMS, WhatsApp, and AI messages are charged per use. Add funds to enable these features.</p>
        <hr>
        
        <div class="cartt-balance-display" id="cartt-balance-display">
            <div class="cartt-balance-amount">
                <span class="cartt-balance-label">Current Balance</span>
                <span class="cartt-balance-value" id="cartt-balance-value">Loading...</span>
            </div>
            <div class="cartt-balance-actions">
                <button type="button" class="button button-primary" id="cartt-add-funds">Add $20</button>
            </div>
        </div>
        
        <div class="cartt-pricing-table" style="margin-top:20px;">
            <h4 style="margin:0 0 12px;font-size:13px;font-weight:600;">Pricing</h4>
            <table class="widefat" style="max-width:400px;">
                <tr><td>SMS / WhatsApp message</td><td style="text-align:right;">$0.03</td></tr>
                <tr><td>With cart content or screenshot</td><td style="text-align:right;">$0.05</td></tr>
                <tr><td>AI auto-reply</td><td style="text-align:right;">$0.01</td></tr>
            </table>
        </div>
        
        <div class="cartt-auto-refill" style="margin-top:20px;">
            <label class="cartt-checkbox-row">
                <input type="checkbox" id="cartt-auto-refill" <?php checked(get_option('cartt_chat_auto_refill', '1'), '1'); ?>>
                Auto-refill $20 when balance drops to $5
            </label>
            <p class="description">We'll automatically charge your card to keep your chat features running.</p>
        </div>
    </div>
    
    <!-- Payment Method -->
    <div class="cartt-settings-section">
        <h2>Payment Method</h2>
        <hr>
        
        <div id="cartt-payment-method">
            <p style="color:#666;">Loading payment info...</p>
        </div>
        
        <div style="margin-top:16px;">
            <button type="button" class="button" id="cartt-update-card">Update Card</button>
        </div>
    </div>
    
    <!-- Transaction History -->
    <div class="cartt-settings-section">
        <h2>Transaction History</h2>
        <hr>
        
        <div id="cartt-transactions">
            <p style="color:#666;">Loading transactions...</p>
        </div>
    </div>
    <?php endif; ?>
    
    <!-- Save Button -->
    <div class="cartt-save-row">
        <button type="button" class="button button-primary" id="cartt-save-settings">Save Settings</button>
    </div>
</div>

<style>
.cartt-balance-display {
    display: flex;
    align-items: center;
    justify-content: space-between;
    padding: 20px;
    background: #f8f9fa;
    border-radius: 8px;
    max-width: 400px;
}

.cartt-balance-amount {
    display: flex;
    flex-direction: column;
    gap: 4px;
}

.cartt-balance-label {
    font-size: 12px;
    color: #666;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

.cartt-balance-value {
    font-size: 28px;
    font-weight: 600;
    color: #1e1e1e;
}

.cartt-balance-value.low {
    color: #dc3545;
}

.cartt-balance-value.ok {
    color: #28a745;
}

#cartt-transactions table {
    max-width: 600px;
}

#cartt-transactions td {
    padding: 8px 12px;
}

.cartt-tx-credit {
    color: #28a745;
    font-weight: 500;
}

.cartt-tx-debit {
    color: #666;
}
</style>

<script>
jQuery(document).ready(function($) {
    var nonce = '<?php echo $nonce; ?>';
    var ajaxurl = '<?php echo admin_url('admin-ajax.php'); ?>';
    
    function showNotice(message, type) {
        var html = '<div class="cartt-notice ' + type + '">' + message + '</div>';
        $('#cartt-settings-notice').html(html);
        setTimeout(function() {
            $('#cartt-settings-notice').html('');
        }, 4000);
    }
    
    // Preview update function
    function updatePreview() {
        var headerColor = $('#cartt-color-header').val();
        var headerText = $('#cartt-color-header-text').val();
        var visitorBubble = $('#cartt-color-visitor-bubble').val();
        var agentBubble = $('#cartt-color-agent-bubble').val();
        var background = $('#cartt-color-background').val();
        var inputBorder = $('#cartt-color-input-border').val();
        var inputBg = $('#cartt-color-input-bg').val();
        var inputField = $('#cartt-color-input-field').val();
        var borderWidth = $('#cartt-input-border-width').val();
        var sendIcon = $('#cartt-color-send-icon').val();
        var title = $('#cartt-chat-title').val() || 'Chat with us';
        var greeting = $('#cartt-chat-greeting').val() || 'Hi! How can we help?';
        var onlineText = $('#cartt-online-text').val() || 'Online now';
        
        // Use selected header text color
        var textColor = headerText;
        
        // Calculate visitor bubble text
        var vHex = visitorBubble.replace('#', '');
        var vLum = (0.299 * parseInt(vHex.substr(0, 2), 16) + 0.587 * parseInt(vHex.substr(2, 2), 16) + 0.114 * parseInt(vHex.substr(4, 2), 16)) / 255;
        var visitorText = vLum > 0.5 ? '#000000' : '#ffffff';
        
        // Calculate agent bubble text
        var aHex = agentBubble.replace('#', '');
        var aLum = (0.299 * parseInt(aHex.substr(0, 2), 16) + 0.587 * parseInt(aHex.substr(2, 2), 16) + 0.114 * parseInt(aHex.substr(4, 2), 16)) / 255;
        var agentText = aLum > 0.5 ? '#333333' : '#ffffff';
        
        $('#preview-header').css({ background: headerColor, color: textColor });
        $('#preview-avatar').css('background', 'rgba(255,255,255,0.15)').find('.material-icons').css('color', textColor);
        $('#preview-title').text(title).css('color', textColor);
        $('#preview-status').css('color', textColor);
        $('#preview-messages').css('background', background);
        $('#preview-greeting').css({ background: agentBubble, color: agentText }).text(greeting.substring(0, 30) + (greeting.length > 30 ? '...' : ''));
        $('#preview-visitor-msg').css({ background: visitorBubble, color: visitorText });
        $('#preview-input-area').css({ background: inputBg, 'border-top-color': inputBorder });
        $('#preview-input').css({ border: borderWidth + 'px solid ' + inputBorder, background: inputField });
        $('#preview-send-icon').css('color', sendIcon);
    }
    
    // Bind preview updates to color inputs
    $('#cartt-color-header, #cartt-color-header-text, #cartt-color-visitor-bubble, #cartt-color-agent-bubble, #cartt-color-background, #cartt-color-input-border, #cartt-color-input-bg, #cartt-color-input-field, #cartt-input-border-width, #cartt-color-send-icon, #cartt-chat-title, #cartt-chat-greeting, #cartt-online-text').on('input change', updatePreview);
    
    // Initial preview
    updatePreview();
    
    // Avatar selection
    $('#cartt-select-avatar').on('click', function() {
        var frame = wp.media({
            title: 'Select Avatar Image',
            button: { text: 'Use this image' },
            multiple: false
        });
        
        frame.on('select', function() {
            var attachment = frame.state().get('selection').first().toJSON();
            $('#cartt-avatar').val(attachment.url);
            $('#cartt-avatar-preview').html(
                '<img src="' + attachment.url + '">' +
                '<button type="button" class="button-link" id="cartt-remove-avatar" style="color:#b32d2e;">Remove</button>'
            );
        });
        
        frame.open();
    });
    
    $(document).on('click', '#cartt-remove-avatar', function() {
        $('#cartt-avatar').val('');
        $('#cartt-avatar-preview').html(
            '<div class="cartt-default-icon"><span class="material-icons" style="font-size:32px;color:#666;">support_agent</span></div>' +
            '<span style="color:#888;font-size:12px;">Default icon</span>'
        );
    });
    
    // Save settings
    $('#cartt-save-settings, #cartt-save-settings-top').on('click', function() {
        var $btn = $('#cartt-save-settings, #cartt-save-settings-top');
        $btn.prop('disabled', true).text('Saving...');
        
        $.post(ajaxurl, {
            action: 'cartt_chat_save_settings',
            nonce: nonce,
            // Basic
            enabled: '1',
            title: $('#cartt-chat-title').val(),
            greeting: $('#cartt-chat-greeting').val(),
            online_text: $('#cartt-online-text').val(),
            position: $('#cartt-position').val(),
            avatar: $('#cartt-avatar').val(),
            // Display
            show_online: $('#cartt-show-online').is(':checked') ? '1' : '0',
            sound_enabled: $('#cartt-sound-enabled').is(':checked') ? '1' : '0',
            screenshots: $('#cartt-screenshots').is(':checked') ? '1' : '0',
            attachments: $('#cartt-attachments').is(':checked') ? '1' : '0',
            // Colors
            color_header: $('#cartt-color-header').val(),
            color_header_text: $('#cartt-color-header-text').val(),
            color_visitor_bubble: $('#cartt-color-visitor-bubble').val(),
            color_agent_bubble: $('#cartt-color-agent-bubble').val(),
            color_background: $('#cartt-color-background').val(),
            color_input_border: $('#cartt-color-input-border').val(),
            color_input_bg: $('#cartt-color-input-bg').val(),
            color_input_field: $('#cartt-color-input-field').val(),
            input_border_width: $('#cartt-input-border-width').val(),
            color_icons: $('#cartt-color-icons').val(),
            color_send_icon: $('#cartt-color-send-icon').val(),
            // Pre-chat
            collect_name: $('#cartt-collect-name').is(':checked') ? '1' : '0',
            collect_email: $('#cartt-collect-email').is(':checked') ? '1' : '0',
            collect_phone: $('#cartt-collect-phone').is(':checked') ? '1' : '0',
            // Email
            notify_email: $('#cartt-notify-email').is(':checked') ? '1' : '0',
            email_address: $('#cartt-email-address').val(),
            email_include_cart: $('#cartt-email-include-cart').is(':checked') ? '1' : '0',
            email_frequency: $('#cartt-email-frequency').val(),
            // SMS
            notify_sms: $('#cartt-notify-sms').is(':checked') ? '1' : '0',
            sms_number: $('#cartt-sms-number').val(),
            sms_include_cart: $('#cartt-sms-include-cart').is(':checked') ? '1' : '0',
            // WhatsApp
            notify_whatsapp: $('#cartt-notify-whatsapp').is(':checked') ? '1' : '0',
            whatsapp_number: $('#cartt-whatsapp-number').val(),
            whatsapp_include_cart: $('#cartt-whatsapp-include-cart').is(':checked') ? '1' : '0',
            // AI
            ai_enabled: $('#cartt-ai-enabled').is(':checked') ? '1' : '0',
            ai_delay: $('#cartt-ai-delay').val(),
            ai_store_info: $('#cartt-ai-store-info').val(),
            // Handoff
            handoff_timeout: $('#cartt-handoff-timeout').val(),
            handoff_message: $('#cartt-handoff-message').val()
        }, function(response) {
            $btn.prop('disabled', false).text('Save Settings');
            showNotice(response.success ? 'Settings saved!' : 'Error saving settings', response.success ? 'success' : 'error');
        });
    });
    
    // Test email
    $('#cartt-test-email').on('click', function() {
        var $btn = $(this);
        var email = $('#cartt-email-address').val();
        if (!email) {
            showNotice('Please enter an email address first', 'error');
            return;
        }
        $btn.prop('disabled', true).text('Sending...');
        
        $.post(ajaxurl, {
            action: 'cartt_chat_test_notification',
            nonce: nonce,
            channel: 'email',
            to: email
        }, function(response) {
            $btn.prop('disabled', false).text('Send Test');
            if (response.success) {
                showNotice(response.data.message || 'Test email sent to ' + email, 'success');
            } else {
                showNotice(response.data || 'Failed to send email', 'error');
            }
        }).fail(function() {
            $btn.prop('disabled', false).text('Send Test');
            showNotice('Request failed. Check your connection.', 'error');
        });
    });
    
    // Test SMS
    $('#cartt-test-sms').on('click', function() {
        var $btn = $(this);
        var phone = $('#cartt-sms-number').val();
        if (!phone) {
            showNotice('Please enter a phone number first', 'error');
            return;
        }
        $btn.prop('disabled', true).text('Sending...');
        
        $.post(ajaxurl, {
            action: 'cartt_chat_test_notification',
            nonce: nonce,
            channel: 'sms',
            to: phone
        }, function(response) {
            $btn.prop('disabled', false).text('Send Test');
            if (response.success) {
                showNotice(response.data.message || 'Test SMS sent to ' + phone, 'success');
            } else {
                showNotice(response.data || 'Failed to send SMS', 'error');
            }
        }).fail(function() {
            $btn.prop('disabled', false).text('Send Test');
            showNotice('Request failed. Check your connection.', 'error');
        });
    });
    
    // Test WhatsApp
    $('#cartt-test-whatsapp').on('click', function() {
        var $btn = $(this);
        var phone = $('#cartt-whatsapp-number').val();
        if (!phone) {
            showNotice('Please enter a WhatsApp number first', 'error');
            return;
        }
        $btn.prop('disabled', true).text('Sending...');
        
        $.post(ajaxurl, {
            action: 'cartt_chat_test_notification',
            nonce: nonce,
            channel: 'whatsapp',
            to: phone
        }, function(response) {
            $btn.prop('disabled', false).text('Send Test');
            if (response.success) {
                showNotice(response.data.message || 'Test WhatsApp sent to ' + phone, 'success');
            } else {
                showNotice(response.data || 'Failed to send WhatsApp', 'error');
            }
        }).fail(function() {
            $btn.prop('disabled', false).text('Send Test');
            showNotice('Request failed. Check your connection.', 'error');
        });
    });
    
    // Load balance on page load
    function loadBalance() {
        $.post(ajaxurl, {
            action: 'cartt_chat_get_balance',
            nonce: nonce
        }, function(response) {
            if (response.success && response.data) {
                var balance = parseFloat(response.data.balance) || 0;
                var $val = $('#cartt-balance-value');
                $val.text('$' + balance.toFixed(2));
                $val.removeClass('low ok');
                if (balance < 5) {
                    $val.addClass('low');
                } else {
                    $val.addClass('ok');
                }
            } else {
                $('#cartt-balance-value').text('$0.00').addClass('low');
            }
        });
    }
    
    // Load payment method
    function loadPaymentMethod() {
        $.post(ajaxurl, {
            action: 'cartt_chat_get_payment_method',
            nonce: nonce
        }, function(response) {
            var $pm = $('#cartt-payment-method');
            if (response.success && response.data && response.data.card) {
                var card = response.data.card;
                $pm.html(
                    '<div style="display:flex;align-items:center;gap:12px;">' +
                    '<span style="font-size:24px;">' + getCardIcon(card.brand) + '</span>' +
                    '<span style="font-weight:500;">' + card.brand.toUpperCase() + ' ending in ' + card.last4 + '</span>' +
                    '<span style="color:#666;">Expires ' + card.exp_month + '/' + card.exp_year + '</span>' +
                    '</div>'
                );
            } else {
                $pm.html('<p style="color:#666;">No payment method on file. Add funds to set up automatic billing.</p>');
            }
        });
    }
    
    function getCardIcon(brand) {
        var icons = {
            'visa': '💳',
            'mastercard': '💳',
            'amex': '💳',
            'discover': '💳'
        };
        return icons[brand.toLowerCase()] || '💳';
    }
    
    // Load transactions
    function loadTransactions() {
        $.post(ajaxurl, {
            action: 'cartt_chat_get_transactions',
            nonce: nonce
        }, function(response) {
            var $tx = $('#cartt-transactions');
            if (response.success && response.data && response.data.transactions && response.data.transactions.length > 0) {
                var html = '<table class="widefat striped">';
                html += '<thead><tr><th>Date</th><th>Description</th><th style="text-align:right;">Amount</th></tr></thead>';
                html += '<tbody>';
                response.data.transactions.forEach(function(tx) {
                    var cls = tx.amount > 0 ? 'cartt-tx-credit' : 'cartt-tx-debit';
                    var amt = tx.amount > 0 ? '+$' + tx.amount.toFixed(2) : '-$' + Math.abs(tx.amount).toFixed(2);
                    html += '<tr>';
                    html += '<td>' + tx.date + '</td>';
                    html += '<td>' + tx.description + '</td>';
                    html += '<td style="text-align:right;" class="' + cls + '">' + amt + '</td>';
                    html += '</tr>';
                });
                html += '</tbody></table>';
                $tx.html(html);
            } else {
                $tx.html('<p style="color:#666;">No transactions yet.</p>');
            }
        });
    }
    
    // Add funds
    $('#cartt-add-funds').on('click', function() {
        var $btn = $(this);
        $btn.prop('disabled', true).text('Processing...');
        
        $.post(ajaxurl, {
            action: 'cartt_chat_add_funds',
            nonce: nonce,
            amount: 20
        }, function(response) {
            $btn.prop('disabled', false).text('Add $20');
            if (response.success) {
                if (response.data && response.data.checkout_url) {
                    // Redirect to Stripe checkout
                    window.location.href = response.data.checkout_url;
                } else {
                    showNotice('Funds added successfully!', 'success');
                    loadBalance();
                    loadTransactions();
                }
            } else {
                showNotice(response.data || 'Failed to add funds', 'error');
            }
        });
    });
    
    // Update card
    $('#cartt-update-card').on('click', function() {
        var $btn = $(this);
        $btn.prop('disabled', true).text('Loading...');
        
        $.post(ajaxurl, {
            action: 'cartt_chat_get_card_update_url',
            nonce: nonce
        }, function(response) {
            $btn.prop('disabled', false).text('Update Card');
            if (response.success && response.data && response.data.url) {
                window.location.href = response.data.url;
            } else {
                showNotice(response.data || 'Could not get update link', 'error');
            }
        });
    });
    
    // Auto-refill toggle
    $('#cartt-auto-refill').on('change', function() {
        var enabled = $(this).is(':checked') ? '1' : '0';
        $.post(ajaxurl, {
            action: 'cartt_chat_set_auto_refill',
            nonce: nonce,
            enabled: enabled
        }, function(response) {
            if (response.success) {
                showNotice('Auto-refill ' + (enabled === '1' ? 'enabled' : 'disabled'), 'success');
            }
        });
    });
    
    // ==================== QUICK ACTIONS ====================
    
    var quickActions = <?php echo wp_json_encode($settings['quick_actions']); ?>;
    
    function renderQuickActions() {
        var html = '';
        if (quickActions.length === 0) {
            html = '<p style="color:#666;font-style:italic;">No quick actions yet. Add some below.</p>';
        } else {
            quickActions.forEach(function(action, index) {
                html += '<div class="cartt-quick-action-item" style="display:flex;align-items:center;gap:10px;padding:8px 0;border-bottom:1px solid #eee;">';
                html += '<span style="flex:1;">' + escapeHtml(action) + '</span>';
                html += '<button type="button" class="button-link cartt-remove-quick-action" data-index="' + index + '" style="color:#b32d2e;">Remove</button>';
                html += '</div>';
            });
        }
        $('#cartt-quick-actions-list').html(html);
    }
    
    function escapeHtml(text) {
        var div = document.createElement('div');
        div.textContent = text;
        return div.innerHTML;
    }
    
    function saveQuickActions() {
        $.post(ajaxurl, {
            action: 'cartt_chat_save_quick_actions',
            nonce: nonce,
            quick_actions: quickActions
        }, function(response) {
            if (response.success) {
                showNotice('Quick actions saved', 'success');
            }
        });
    }
    
    // Add quick action
    $('#cartt-add-quick-action').on('click', function() {
        var newAction = $('#cartt-new-quick-action').val().trim();
        if (newAction) {
            quickActions.push(newAction);
            $('#cartt-new-quick-action').val('');
            renderQuickActions();
            saveQuickActions();
        }
    });
    
    // Enter key to add
    $('#cartt-new-quick-action').on('keypress', function(e) {
        if (e.which === 13) {
            e.preventDefault();
            $('#cartt-add-quick-action').click();
        }
    });
    
    // Remove quick action
    $(document).on('click', '.cartt-remove-quick-action', function() {
        var index = $(this).data('index');
        quickActions.splice(index, 1);
        renderQuickActions();
        saveQuickActions();
    });
    
    renderQuickActions();
    
    // === AGENTS ===
    var agents = [];
    
    function loadAgents() {
        $.post(ajaxurl, {
            action: 'cartt_chat_get_agents',
            nonce: nonce
        }, function(response) {
            if (response.success) {
                agents = response.data.agents || [];
                renderAgents();
            }
        });
    }
    
    function renderAgents() {
        var html = '';
        if (agents.length === 0) {
            html = '<p style="color:#666;font-size:13px;">No agents added yet. Add team members to receive notifications.</p>';
        } else {
            agents.forEach(function(agent, index) {
                html += '<div class="cartt-agent-item" style="display:flex;align-items:center;gap:12px;padding:12px;background:#f9f9f9;border-radius:6px;margin-bottom:8px;">';
                html += '<img src="' + (agent.avatar || 'data:image/svg+xml,<svg xmlns=%22http://www.w3.org/2000/svg%22 viewBox=%220 0 24 24%22><circle cx=%2212%22 cy=%228%22 r=%224%22 fill=%22%23999%22/><path d=%22M12 14c-4 0-8 2-8 4v2h16v-2c0-2-4-4-8-4z%22 fill=%22%23999%22/></svg>') + '" style="width:40px;height:40px;border-radius:50%;object-fit:cover;background:#e0e0e0;">';
                html += '<div style="flex:1;">';
                html += '<div style="font-weight:600;">' + escapeHtml(agent.display_name || 'Agent') + '</div>';
                html += '<div style="font-size:12px;color:#666;">';
                if (agent.email) html += agent.email;
                if (agent.sms_number) html += (agent.email ? ' | ' : '') + 'SMS: ' + agent.sms_number;
                if (agent.whatsapp_number) html += ' | WA: ' + agent.whatsapp_number;
                html += '</div></div>';
                html += '<label style="display:flex;align-items:center;gap:6px;font-size:12px;"><input type="checkbox" class="cartt-agent-active" data-id="' + agent.id + '" ' + (agent.is_active == 1 ? 'checked' : '') + '> Active</label>';
                html += '<button type="button" class="button-link cartt-remove-agent" data-id="' + agent.id + '" style="color:#b32d2e;">Remove</button>';
                html += '</div>';
            });
        }
        $('#cartt-agents-list').html(html);
    }
    
    // Agent avatar selection
    $('#cartt-select-agent-avatar').on('click', function() {
        var frame = wp.media({
            title: 'Select Agent Avatar',
            button: { text: 'Use this image' },
            multiple: false
        });
        frame.on('select', function() {
            var attachment = frame.state().get('selection').first().toJSON();
            $('#cartt-new-agent-avatar').val(attachment.url);
            $('#cartt-new-agent-avatar-preview').attr('src', attachment.url).show();
        });
        frame.open();
    });
    
    // Add agent
    $('#cartt-add-agent').on('click', function() {
        var name = $('#cartt-new-agent-name').val().trim();
        var email = $('#cartt-new-agent-email').val().trim();
        var sms = $('#cartt-new-agent-sms').val().trim();
        var whatsapp = $('#cartt-new-agent-whatsapp').val().trim();
        var avatar = $('#cartt-new-agent-avatar').val();
        
        if (!name) {
            showNotice('Please enter agent name', 'error');
            return;
        }
        
        $.post(ajaxurl, {
            action: 'cartt_chat_add_agent',
            nonce: nonce,
            display_name: name,
            email: email,
            sms_number: sms,
            whatsapp_number: whatsapp,
            avatar: avatar
        }, function(response) {
            if (response.success) {
                $('#cartt-new-agent-name, #cartt-new-agent-email, #cartt-new-agent-sms, #cartt-new-agent-whatsapp, #cartt-new-agent-avatar').val('');
                $('#cartt-new-agent-avatar-preview').hide();
                loadAgents();
                showNotice('Agent added', 'success');
            } else {
                showNotice(response.data || 'Error adding agent', 'error');
            }
        });
    });
    
    // Toggle agent active
    $(document).on('change', '.cartt-agent-active', function() {
        var id = $(this).data('id');
        var active = $(this).is(':checked') ? 1 : 0;
        $.post(ajaxurl, {
            action: 'cartt_chat_update_agent',
            nonce: nonce,
            id: id,
            is_active: active
        });
    });
    
    // Remove agent
    $(document).on('click', '.cartt-remove-agent', function() {
        if (!confirm('Remove this agent?')) return;
        var id = $(this).data('id');
        $.post(ajaxurl, {
            action: 'cartt_chat_remove_agent',
            nonce: nonce,
            id: id
        }, function(response) {
            if (response.success) {
                loadAgents();
                showNotice('Agent removed', 'success');
            }
        });
    });
    
    loadAgents();
    
    // Initialize
    loadBalance();
    loadPaymentMethod();
    loadTransactions();
});
</script>
