<?php
/**
 * Categories Admin Page
 */

if (!defined('ABSPATH')) {
    exit;
}

$categoryService = \Cartt\Services\CategoryService::class;
$action = isset($_GET['action']) ? sanitize_text_field($_GET['action']) : 'list';
$categoryId = isset($_GET['id']) ? intval($_GET['id']) : 0;

// Handle save
$message = '';
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['cartt_cat_nonce'])) {
    if (wp_verify_nonce($_POST['cartt_cat_nonce'], 'cartt_save_category')) {
        $data = [
            'name' => $_POST['name'] ?? '',
            'description' => $_POST['description'] ?? '',
            'parent_id' => $_POST['parent_id'] ?? null,
            'image' => $_POST['image'] ?? null,
        ];
        
        if ($categoryId) {
            $categoryService::updateCategory($categoryId, $data);
            $message = __('Category updated.', 'cartt');
        } else {
            $categoryId = $categoryService::createCategory($data);
            $message = __('Category created.', 'cartt');
        }
    }
}

// Handle delete
if (isset($_GET['delete']) && isset($_GET['_wpnonce'])) {
    if (wp_verify_nonce($_GET['_wpnonce'], 'delete_cat_' . $_GET['delete'])) {
        $categoryService::deleteCategory((int)$_GET['delete']);
        $message = __('Category deleted.', 'cartt');
    }
}

$categories = $categoryService::getAllCategories();
$editCategory = ($action === 'edit' && $categoryId) ? $categoryService::getCategory($categoryId) : null;
?>

<?php
$pendingQA = \Cartt\Services\ProductQAService::getPendingCount();
?>
<div class="wrap cartt-wrap">
    <!-- Tabs for Products section -->
    <nav class="nav-tab-wrapper" style="margin-bottom: 20px;">
        <a href="?page=cartt-products&tab=products" class="nav-tab">
            All Products
        </a>
        <a href="?page=cartt-products&tab=categories" class="nav-tab nav-tab-active">
            Categories
        </a>
        <a href="?page=cartt-products&tab=inventory" class="nav-tab">
            Inventory
        </a>
        <a href="?page=cartt-products&tab=qa" class="nav-tab">
            Q&A
            <?php if ($pendingQA > 0): ?>
            <span class="awaiting-mod"><?php echo $pendingQA; ?></span>
            <?php endif; ?>
        </a>
    </nav>

    <div class="cartt-header">
        <h1><?php esc_html_e('Categories', 'cartt'); ?></h1>
    </div>

    <?php if ($message): ?>
        <div class="notice notice-success is-dismissible"><p><?php echo esc_html($message); ?></p></div>
    <?php endif; ?>

    <div class="cartt-categories-layout">
        <!-- Category Form -->
        <div class="cartt-cat-form-panel">
            <h2><?php echo $editCategory ? esc_html__('Edit Category', 'cartt') : esc_html__('Add Category', 'cartt'); ?></h2>
            
            <form method="post">
                <?php wp_nonce_field('cartt_save_category', 'cartt_cat_nonce'); ?>
                
                <div class="cartt-form-row">
                    <label><?php esc_html_e('Name', 'cartt'); ?></label>
                    <input type="text" name="name" value="<?php echo esc_attr($editCategory->name ?? ''); ?>" required>
                </div>
                
                <div class="cartt-form-row">
                    <label><?php esc_html_e('Parent Category', 'cartt'); ?></label>
                    <select name="parent_id">
                        <option value=""><?php esc_html_e('None (Top Level)', 'cartt'); ?></option>
                        <?php foreach ($categories as $cat): 
                            if ($editCategory && $cat->id === $editCategory->id) continue;
                        ?>
                        <option value="<?php echo esc_attr($cat->id); ?>" <?php selected($editCategory->parent_id ?? '', $cat->id); ?>>
                            <?php echo esc_html($cat->name); ?>
                        </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                
                <div class="cartt-form-row">
                    <label><?php esc_html_e('Description', 'cartt'); ?></label>
                    <textarea name="description" rows="4"><?php echo esc_textarea($editCategory->description ?? ''); ?></textarea>
                </div>
                
                <div class="cartt-form-row">
                    <label><?php esc_html_e('Image', 'cartt'); ?></label>
                    <div class="cartt-image-upload" id="cat-image-upload">
                        <?php if (!empty($editCategory->image)): 
                            $imgUrl = wp_get_attachment_image_url($editCategory->image, 'thumbnail');
                        ?>
                            <img src="<?php echo esc_url($imgUrl); ?>" class="cartt-image-preview" id="cat-image-preview">
                            <input type="hidden" name="image" id="cat-image" value="<?php echo esc_attr($editCategory->image); ?>">
                            <button type="button" class="button button-small" id="remove-cat-image"><?php esc_html_e('Remove', 'cartt'); ?></button>
                        <?php else: ?>
                            <div class="cartt-image-placeholder"><?php esc_html_e('Click to add image', 'cartt'); ?></div>
                            <input type="hidden" name="image" id="cat-image" value="">
                        <?php endif; ?>
                    </div>
                </div>
                
                <div class="cartt-form-actions">
                    <button type="submit" class="button button-primary">
                        <?php echo $editCategory ? esc_html__('Update Category', 'cartt') : esc_html__('Add Category', 'cartt'); ?>
                    </button>
                    <?php if ($editCategory): ?>
                    <a href="<?php echo esc_url(admin_url('admin.php?page=cartt-categories')); ?>" class="button">
                        <?php esc_html_e('Cancel', 'cartt'); ?>
                    </a>
                    <?php endif; ?>
                </div>
            </form>
        </div>

        <!-- Categories List -->
        <div class="cartt-cat-list-panel">
            <?php if (empty($categories)): ?>
            <div class="cartt-empty-state-small">
                <p><?php esc_html_e('No categories yet. Create your first category to organize your products.', 'cartt'); ?></p>
            </div>
            <?php else: ?>
            <table class="cartt-table">
                <thead>
                    <tr>
                        <th style="width:30px;"></th>
                        <th><?php esc_html_e('Name', 'cartt'); ?></th>
                        <th><?php esc_html_e('Slug', 'cartt'); ?></th>
                        <th><?php esc_html_e('Products', 'cartt'); ?></th>
                        <th></th>
                    </tr>
                </thead>
                <tbody id="categories-sortable">
                    <?php 
                    $hierarchy = $categoryService::getCategoriesHierarchy();
                    renderCategoryRows($hierarchy, $categoryService);
                    ?>
                </tbody>
            </table>
            <?php endif; ?>
        </div>
    </div>
</div>

<?php
function renderCategoryRows($categories, $service, $depth = 0) {
    foreach ($categories as $cat) {
        $indent = str_repeat('— ', $depth);
        $count = $service::getCategoryProductCount($cat->id);
        ?>
        <tr data-cat-id="<?php echo esc_attr($cat->id); ?>">
            <td style="width:30px;cursor:grab;" class="cartt-drag-handle">⋮⋮</td>
            <td>
                <strong><?php echo $indent . esc_html($cat->name); ?></strong>
            </td>
            <td><?php echo esc_html($cat->slug); ?></td>
            <td><?php echo number_format($count); ?></td>
            <td class="cartt-actions">
                <a href="<?php echo esc_url(admin_url('admin.php?page=cartt-products&tab=categories&action=edit&id=' . $cat->id)); ?>">
                    <?php esc_html_e('Edit', 'cartt'); ?>
                </a>
                <a href="<?php echo esc_url(wp_nonce_url(admin_url('admin.php?page=cartt-products&tab=categories&delete=' . $cat->id), 'delete_cat_' . $cat->id)); ?>"
                   onclick="return confirm('<?php esc_attr_e('Delete this category?', 'cartt'); ?>');"
                   class="cartt-action-delete">
                    <?php esc_html_e('Delete', 'cartt'); ?>
                </a>
            </td>
        </tr>
        <?php
        if (!empty($cat->children)) {
            renderCategoryRows($cat->children, $service, $depth + 1);
        }
    }
}
?>

<style>
.cartt-categories-layout { display: grid; grid-template-columns: 300px 1fr; gap: 20px; margin-top: 20px; }
.cartt-cat-form-panel { background: #fff; border: 1px solid #e8e8ed; border-radius: 8px; padding: 20px; height: fit-content; }
.cartt-cat-form-panel h2 { font-size: 14px; font-weight: 600; margin: 0 0 16px; }
.cartt-cat-list-panel { background: #fff; border: 1px solid #e8e8ed; border-radius: 8px; padding: 20px; }

.cartt-form-row { margin-bottom: 16px; }
.cartt-form-row label { display: block; font-size: 12px; font-weight: 500; margin-bottom: 6px; }
.cartt-form-row input, .cartt-form-row select, .cartt-form-row textarea { width: 100%; padding: 8px 10px; border: 1px solid #d2d2d7; border-radius: 6px; font-size: 13px; }
.cartt-form-row textarea { resize: vertical; }

.cartt-image-upload { border: 2px dashed #d2d2d7; border-radius: 6px; padding: 20px; text-align: center; cursor: pointer; }
.cartt-image-upload:hover { border-color: #1d1d1f; }
.cartt-image-placeholder { color: #86868b; font-size: 13px; }
.cartt-image-preview { max-width: 100px; max-height: 100px; border-radius: 4px; margin-bottom: 8px; }

.cartt-form-actions { display: flex; gap: 8px; margin-top: 20px; }

.cartt-empty-state-small { text-align: center; padding: 40px; color: #86868b; }

@media (max-width: 900px) {
    .cartt-categories-layout { grid-template-columns: 1fr; }
}
</style>

<script>
jQuery(document).ready(function($) {
    // Image upload
    $('#cat-image-upload').on('click', function(e) {
        if ($(e.target).is('button')) return;
        
        var frame = wp.media({
            title: '<?php esc_attr_e('Select Image', 'cartt'); ?>',
            button: { text: '<?php esc_attr_e('Use Image', 'cartt'); ?>' },
            multiple: false
        });
        
        frame.on('select', function() {
            var attachment = frame.state().get('selection').first().toJSON();
            $('#cat-image').val(attachment.id);
            $('#cat-image-upload').html(
                '<img src="' + attachment.sizes.thumbnail.url + '" class="cartt-image-preview" id="cat-image-preview">' +
                '<input type="hidden" name="image" id="cat-image" value="' + attachment.id + '">' +
                '<button type="button" class="button button-small" id="remove-cat-image"><?php esc_attr_e('Remove', 'cartt'); ?></button>'
            );
        });
        
        frame.open();
    });
    
    $(document).on('click', '#remove-cat-image', function(e) {
        e.stopPropagation();
        $('#cat-image-upload').html(
            '<div class="cartt-image-placeholder"><?php esc_attr_e('Click to add image', 'cartt'); ?></div>' +
            '<input type="hidden" name="image" id="cat-image" value="">'
        );
    });
    
    // Sortable categories
    if ($('#categories-sortable').length && typeof $.fn.sortable !== 'undefined') {
        $('#categories-sortable').sortable({
            handle: '.cartt-drag-handle',
            placeholder: 'cartt-sortable-placeholder',
            update: function(event, ui) {
                var order = $(this).sortable('toArray', { attribute: 'data-cat-id' });
                $.post(ajaxurl, {
                    action: 'cartt_reorder_categories',
                    order: order,
                    nonce: '<?php echo wp_create_nonce('cartt_reorder'); ?>'
                }, function(response) {
                    if (response.success) {
                        if (typeof carttToast !== 'undefined') carttToast('Order saved', 'success');
                    }
                });
            }
        });
    }
});
</script>
