<?php
/**
 * Visual Card Designer
 * @since 1.6.0
 */

if (!defined('ABSPATH')) {
    exit;
}

// Get saved design or defaults
$design = get_option('cartt_card_design', []);
$defaults = [
    'preset' => 'modern',
    'elements' => ['image', 'title', 'price', 'rating', 'add_to_cart'],
    'card' => [
        'background' => '#ffffff',
        'border_color' => '#e0e0e0',
        'border_width' => 1,
        'border_radius' => 8,
        'shadow' => 'small',
        'padding' => 0,
        'hover_effect' => 'lift',
        'hover_shadow_size' => 15
    ],
    'image' => [
        'aspect_ratio' => 'square',
        'border_radius' => 8,
        'hover_effect' => 'zoom'
    ],
    'title' => [
        'font_size' => 16,
        'font_weight' => '600',
        'color' => '#1d2327',
        'margin_top' => 12,
        'lines' => 2
    ],
    'price' => [
        'font_size' => 18,
        'font_weight' => '700',
        'color' => '#1d2327',
        'sale_color' => '#dc2626',
        'margin_top' => 8
    ],
    'rating' => [
        'star_color' => '#fbbf24',
        'show_count' => true,
        'margin_top' => 8
    ],
    'add_to_cart' => [
        'style' => 'button',
        'bg_color' => '#1d2327',
        'text_color' => '#ffffff',
        'border_radius' => 6,
        'full_width' => true,
        'margin_top' => 12,
        'button_text' => 'Add to Cart'
    ],
    'quick_view' => [
        'enabled' => false,
        'style' => 'icon',
        'bg_color' => '#ffffff',
        'icon_color' => '#1d2327'
    ],
    'wishlist' => [
        'enabled' => false,
        'position' => 'top-right',
        'bg_color' => '#ffffff',
        'icon_color' => '#1d2327',
        'active_color' => '#dc2626'
    ],
    'sale_badge' => [
        'enabled' => true,
        'bg_color' => '#dc2626',
        'text_color' => '#ffffff',
        'text' => 'Sale'
    ],
    'description' => [
        'font_size' => 13,
        'color' => '#666666',
        'lines' => 2,
        'margin_top' => 8
    ],
    'sku' => [
        'font_size' => 12,
        'color' => '#999999',
        'prefix' => 'SKU:',
        'margin_top' => 8
    ],
    'stock' => [
        'font_size' => 13,
        'in_stock_color' => '#16a34a',
        'out_of_stock_color' => '#dc2626',
        'margin_top' => 8
    ],
    'grid' => [
        'columns' => 4,
        'gap' => 24,
        'columns_tablet' => 3,
        'columns_mobile' => 2
    ]
];

$design = wp_parse_args($design, $defaults);

// Presets
$presets = [
    'minimal' => 'Minimal',
    'modern' => 'Modern',
    'classic' => 'Classic',
    'bold' => 'Bold',
    'elegant' => 'Elegant'
];

$nonce = wp_create_nonce('cartt_card_designer');
?>

<div class="wrap cartt-admin cartt-card-designer">
    <h1>Product Card Designer</h1>
    <p class="description">Design how your products appear in grids and listings.</p>
    
    <div id="cartt-designer-notice"></div>
    
    <div class="cartt-designer-layout">
        <!-- Left Panel: Elements & Settings -->
        <div class="cartt-designer-panel">
            <!-- Presets -->
            <div class="cartt-designer-section">
                <h3>Presets</h3>
                <div class="cartt-preset-grid">
                    <?php foreach ($presets as $key => $label): ?>
                    <button type="button" class="cartt-preset-btn <?php echo $design['preset'] === $key ? 'active' : ''; ?>" data-preset="<?php echo esc_attr($key); ?>">
                        <?php echo esc_html($label); ?>
                    </button>
                    <?php endforeach; ?>
                </div>
            </div>
            
            <!-- Elements -->
            <div class="cartt-designer-section">
                <h3>Elements</h3>
                <p class="description">Drag to reorder. Click to edit.</p>
                <ul id="cartt-elements-list" class="cartt-elements-list">
                    <?php 
                    $all_elements = [
                        'image' => ['label' => 'Product Image', 'icon' => 'format-image'],
                        'title' => ['label' => 'Title', 'icon' => 'editor-textcolor'],
                        'price' => ['label' => 'Price', 'icon' => 'tag'],
                        'rating' => ['label' => 'Rating', 'icon' => 'star-filled'],
                        'add_to_cart' => ['label' => 'Add to Cart', 'icon' => 'cart'],
                        'quick_view' => ['label' => 'Quick View', 'icon' => 'visibility'],
                        'wishlist' => ['label' => 'Wishlist', 'icon' => 'heart'],
                        'sale_badge' => ['label' => 'Sale Badge', 'icon' => 'megaphone'],
                        'description' => ['label' => 'Short Description', 'icon' => 'editor-paragraph'],
                        'sku' => ['label' => 'SKU', 'icon' => 'editor-code'],
                        'stock' => ['label' => 'Stock Status', 'icon' => 'archive']
                    ];
                    
                    // First show enabled elements in order
                    foreach ($design['elements'] as $el):
                        if (isset($all_elements[$el])):
                    ?>
                    <li class="cartt-element-item" data-element="<?php echo esc_attr($el); ?>">
                        <span class="cartt-drag-handle dashicons dashicons-menu"></span>
                        <label>
                            <input type="checkbox" checked data-element="<?php echo esc_attr($el); ?>">
                            <span class="dashicons dashicons-<?php echo esc_attr($all_elements[$el]['icon']); ?>"></span>
                            <?php echo esc_html($all_elements[$el]['label']); ?>
                        </label>
                        <button type="button" class="cartt-element-edit" data-element="<?php echo esc_attr($el); ?>">
                            <span class="dashicons dashicons-admin-generic"></span>
                        </button>
                    </li>
                    <?php 
                        endif;
                    endforeach;
                    
                    // Then show disabled elements
                    foreach ($all_elements as $el => $info):
                        if (!in_array($el, $design['elements'])):
                    ?>
                    <li class="cartt-element-item" data-element="<?php echo esc_attr($el); ?>">
                        <span class="cartt-drag-handle dashicons dashicons-menu"></span>
                        <label>
                            <input type="checkbox" data-element="<?php echo esc_attr($el); ?>">
                            <span class="dashicons dashicons-<?php echo esc_attr($info['icon']); ?>"></span>
                            <?php echo esc_html($info['label']); ?>
                        </label>
                        <button type="button" class="cartt-element-edit" data-element="<?php echo esc_attr($el); ?>">
                            <span class="dashicons dashicons-admin-generic"></span>
                        </button>
                    </li>
                    <?php 
                        endif;
                    endforeach; 
                    ?>
                </ul>
            </div>
            
            <!-- Card Settings -->
            <div class="cartt-designer-section">
                <h3>Card Style</h3>
                
                <div class="cartt-setting-row">
                    <label>Background</label>
                    <input type="color" id="card-background" value="<?php echo esc_attr($design['card']['background']); ?>">
                </div>
                
                <div class="cartt-setting-row">
                    <label>Border</label>
                    <div class="cartt-inline-controls">
                        <input type="number" id="card-border-width" value="<?php echo esc_attr($design['card']['border_width']); ?>" min="0" max="5" style="width:60px;">
                        <input type="color" id="card-border-color" value="<?php echo esc_attr($design['card']['border_color']); ?>">
                    </div>
                </div>
                
                <div class="cartt-setting-row">
                    <label>Corners</label>
                    <input type="range" id="card-border-radius" value="<?php echo esc_attr($design['card']['border_radius']); ?>" min="0" max="24" step="2">
                    <span class="cartt-range-value"><?php echo esc_attr($design['card']['border_radius']); ?>px</span>
                </div>
                
                <div class="cartt-setting-row">
                    <label>Shadow</label>
                    <select id="card-shadow">
                        <option value="none" <?php selected($design['card']['shadow'], 'none'); ?>>None</option>
                        <option value="small" <?php selected($design['card']['shadow'], 'small'); ?>>Small</option>
                        <option value="medium" <?php selected($design['card']['shadow'], 'medium'); ?>>Medium</option>
                        <option value="large" <?php selected($design['card']['shadow'], 'large'); ?>>Large</option>
                    </select>
                </div>
                
                <div class="cartt-setting-row">
                    <label>Hover Effect</label>
                    <select id="card-hover">
                        <option value="none" <?php selected($design['card']['hover_effect'], 'none'); ?>>None</option>
                        <option value="lift" <?php selected($design['card']['hover_effect'], 'lift'); ?>>Lift</option>
                        <option value="shadow" <?php selected($design['card']['hover_effect'], 'shadow'); ?>>Shadow</option>
                        <option value="border" <?php selected($design['card']['hover_effect'], 'border'); ?>>Border</option>
                    </select>
                </div>
                
                <div class="cartt-setting-row" id="hover-shadow-row" style="<?php echo $design['card']['hover_effect'] === 'none' || $design['card']['hover_effect'] === 'border' ? 'display:none;' : ''; ?>">
                    <label>Hover Shadow</label>
                    <input type="range" id="card-hover-shadow" value="<?php echo esc_attr($design['card']['hover_shadow_size'] ?? 15); ?>" min="5" max="40" step="5">
                    <span class="cartt-range-value"><?php echo esc_attr($design['card']['hover_shadow_size'] ?? 15); ?>px</span>
                </div>
            </div>
            
            <!-- Grid Settings -->
            <div class="cartt-designer-section">
                <h3>Grid Layout</h3>
                
                <div class="cartt-setting-row">
                    <label>Columns (Desktop)</label>
                    <select id="grid-columns">
                        <?php for ($i = 2; $i <= 6; $i++): ?>
                        <option value="<?php echo $i; ?>" <?php selected($design['grid']['columns'], $i); ?>><?php echo $i; ?></option>
                        <?php endfor; ?>
                    </select>
                </div>
                
                <div class="cartt-setting-row">
                    <label>Columns (Tablet)</label>
                    <select id="grid-columns-tablet">
                        <?php for ($i = 1; $i <= 4; $i++): ?>
                        <option value="<?php echo $i; ?>" <?php selected($design['grid']['columns_tablet'], $i); ?>><?php echo $i; ?></option>
                        <?php endfor; ?>
                    </select>
                </div>
                
                <div class="cartt-setting-row">
                    <label>Columns (Mobile)</label>
                    <select id="grid-columns-mobile">
                        <?php for ($i = 1; $i <= 3; $i++): ?>
                        <option value="<?php echo $i; ?>" <?php selected($design['grid']['columns_mobile'], $i); ?>><?php echo $i; ?></option>
                        <?php endfor; ?>
                    </select>
                </div>
                
                <div class="cartt-setting-row">
                    <label>Gap</label>
                    <input type="range" id="grid-gap" value="<?php echo esc_attr($design['grid']['gap']); ?>" min="8" max="48" step="4">
                    <span class="cartt-range-value"><?php echo esc_attr($design['grid']['gap']); ?>px</span>
                </div>
            </div>
        </div>
        
        <!-- Right Panel: Live Preview -->
        <div class="cartt-designer-preview">
            <div class="cartt-preview-header">
                <h3>Live Preview</h3>
                <div class="cartt-preview-controls">
                    <button type="button" class="cartt-preview-device active" data-device="desktop" title="Desktop">
                        <span class="dashicons dashicons-desktop"></span>
                    </button>
                    <button type="button" class="cartt-preview-device" data-device="tablet" title="Tablet">
                        <span class="dashicons dashicons-tablet"></span>
                    </button>
                    <button type="button" class="cartt-preview-device" data-device="mobile" title="Mobile">
                        <span class="dashicons dashicons-smartphone"></span>
                    </button>
                </div>
            </div>
            
            <div class="cartt-preview-container" data-device="desktop">
                <div class="cartt-preview-grid" id="cartt-preview-grid">
                    <!-- Preview cards rendered by JS -->
                </div>
            </div>
            
            <div class="cartt-preview-actions">
                <button type="button" class="button" id="cartt-reset-design">Reset to Default</button>
                <button type="button" class="button button-primary" id="cartt-save-design">Save Design</button>
            </div>
        </div>
    </div>
    
    <!-- Element Settings Modal -->
    <div id="cartt-element-modal" class="cartt-modal" style="display:none;">
        <div class="cartt-modal-content">
            <div class="cartt-modal-header">
                <h3 id="cartt-modal-title">Element Settings</h3>
                <button type="button" class="cartt-modal-close">&times;</button>
            </div>
            <div class="cartt-modal-body" id="cartt-modal-body">
                <!-- Dynamic content -->
            </div>
            <div class="cartt-modal-footer">
                <button type="button" class="button cartt-modal-close">Cancel</button>
                <button type="button" class="button button-primary" id="cartt-modal-save">Apply</button>
            </div>
        </div>
    </div>
</div>

<style>
.cartt-card-designer {
    max-width: 1400px;
}

.cartt-designer-layout {
    display: grid;
    grid-template-columns: 320px 1fr;
    gap: 24px;
    margin-top: 20px;
}

.cartt-designer-panel {
    background: #fff;
    border: 1px solid #e0e0e0;
    border-radius: 8px;
    padding: 0;
    max-height: calc(100vh - 180px);
    overflow-y: auto;
}

.cartt-designer-section {
    padding: 16px 20px;
    border-bottom: 1px solid #f0f0f0;
}

.cartt-designer-section:last-child {
    border-bottom: none;
}

.cartt-designer-section h3 {
    margin: 0 0 12px;
    font-size: 13px;
    font-weight: 600;
    color: #1d2327;
}

.cartt-designer-section .description {
    margin: -8px 0 12px;
    font-size: 12px;
    color: #666;
}

/* Presets */
.cartt-preset-grid {
    display: grid;
    grid-template-columns: repeat(3, 1fr);
    gap: 8px;
}

.cartt-preset-btn {
    padding: 8px 12px;
    border: 1px solid #e0e0e0;
    border-radius: 4px;
    background: #fff;
    font-size: 12px;
    cursor: pointer;
    transition: all 0.2s;
}

.cartt-preset-btn:hover {
    border-color: #1d2327;
}

.cartt-preset-btn.active {
    background: #1d2327;
    color: #fff;
    border-color: #1d2327;
}

/* Elements List */
.cartt-elements-list {
    margin: 0;
    padding: 0;
    list-style: none;
}

.cartt-element-item {
    display: flex;
    align-items: center;
    gap: 8px;
    padding: 10px 12px;
    background: #f9f9f9;
    border: 1px solid #e0e0e0;
    border-radius: 4px;
    margin-bottom: 6px;
    cursor: grab;
}

.cartt-element-item:active {
    cursor: grabbing;
}

.cartt-element-item.sortable-ghost {
    opacity: 0.4;
}

.cartt-drag-handle {
    color: #999;
    cursor: grab;
}

.cartt-element-item label {
    flex: 1;
    display: flex;
    align-items: center;
    gap: 8px;
    cursor: pointer;
    font-size: 13px;
}

.cartt-element-item input[type="checkbox"] {
    margin: 0;
}

.cartt-element-item .dashicons {
    font-size: 16px;
    width: 16px;
    height: 16px;
    color: #666;
}

.cartt-element-edit {
    background: none;
    border: none;
    padding: 4px;
    cursor: pointer;
    opacity: 0.5;
    transition: opacity 0.2s;
}

.cartt-element-edit:hover {
    opacity: 1;
}

/* Settings Rows */
.cartt-setting-row {
    display: flex;
    align-items: center;
    justify-content: space-between;
    margin-bottom: 12px;
}

.cartt-setting-row:last-child {
    margin-bottom: 0;
}

.cartt-setting-row label {
    font-size: 13px;
    color: #333;
}

.cartt-setting-row input[type="color"] {
    width: 36px;
    height: 28px;
    padding: 0;
    border: 1px solid #e0e0e0;
    border-radius: 4px;
    cursor: pointer;
}

.cartt-setting-row input[type="range"] {
    width: 100px;
    -webkit-appearance: none;
    appearance: none;
    height: 6px;
    background: #e0e0e0;
    border-radius: 3px;
    outline: none;
}

.cartt-setting-row input[type="range"]::-webkit-slider-thumb {
    -webkit-appearance: none;
    appearance: none;
    width: 16px;
    height: 16px;
    background: #1d2327;
    border-radius: 50%;
    cursor: pointer;
}

.cartt-setting-row input[type="range"]::-moz-range-thumb {
    width: 16px;
    height: 16px;
    background: #1d2327;
    border-radius: 50%;
    cursor: pointer;
    border: none;
}

.cartt-setting-row select {
    min-width: 100px;
}

.cartt-range-value {
    font-size: 12px;
    color: #666;
    min-width: 40px;
    text-align: right;
}

.cartt-inline-controls {
    display: flex;
    gap: 8px;
    align-items: center;
}

/* Preview Panel */
.cartt-designer-preview {
    background: #fff;
    border: 1px solid #e0e0e0;
    border-radius: 8px;
    padding: 20px;
    display: flex;
    flex-direction: column;
}

.cartt-preview-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 16px;
}

.cartt-preview-header h3 {
    margin: 0;
    font-size: 14px;
}

.cartt-preview-controls {
    display: flex;
    gap: 4px;
}

.cartt-preview-device {
    padding: 6px 10px;
    border: 1px solid #e0e0e0;
    border-radius: 4px;
    background: #fff;
    cursor: pointer;
}

.cartt-preview-device.active {
    background: #1d2327;
    color: #fff;
    border-color: #1d2327;
}

.cartt-preview-device .dashicons {
    font-size: 18px;
    width: 18px;
    height: 18px;
}

.cartt-preview-container {
    flex: 1;
    background: #f5f5f5;
    border-radius: 8px;
    padding: 24px;
    overflow: auto;
    min-height: 400px;
    transition: all 0.3s;
}

.cartt-preview-container[data-device="tablet"] {
    max-width: 768px;
    margin: 0 auto;
}

.cartt-preview-container[data-device="mobile"] {
    max-width: 375px;
    margin: 0 auto;
}

.cartt-preview-grid {
    display: grid;
    gap: 24px;
    grid-auto-rows: 1fr;
}

.cartt-preview-actions {
    display: flex;
    justify-content: flex-end;
    gap: 12px;
    margin-top: 16px;
    padding-top: 16px;
    border-top: 1px solid #e0e0e0;
}

/* Preview Card */
.cartt-preview-card {
    background: #fff;
    border-radius: 8px;
    overflow: hidden;
    transition: all 0.3s;
    position: relative;
}

/* Hover Effects - base classes, shadow values set dynamically */
.cartt-hover-lift:hover {
    transform: translateY(-4px);
}

.cartt-hover-border:hover {
    border-color: #1d2327 !important;
}

.cartt-preview-card:hover img {
    transform: scale(1.05);
}

.cartt-preview-card:hover .cartt-preview-quickview {
    opacity: 1;
}

.cartt-preview-card-inner {
    padding: 0;
}

.cartt-preview-image {
    position: relative;
    background: #f0f0f0;
    overflow: hidden;
}

.cartt-preview-image img {
    width: 100%;
    height: 100%;
    object-fit: cover;
    display: block;
}

.cartt-preview-image-placeholder {
    width: 100%;
    padding-bottom: 100%;
    background: linear-gradient(135deg, #e0e0e0 25%, #f0f0f0 50%, #e0e0e0 75%);
    display: flex;
    align-items: center;
    justify-content: center;
}

.cartt-preview-content {
    padding: 16px;
}

.cartt-preview-title {
    font-size: 16px;
    font-weight: 600;
    color: #1d2327;
    margin: 0;
    overflow: hidden;
    display: -webkit-box;
    -webkit-line-clamp: 2;
    -webkit-box-orient: vertical;
}

.cartt-preview-price {
    font-size: 18px;
    font-weight: 700;
    color: #1d2327;
    margin-top: 8px;
}

.cartt-preview-price .sale {
    color: #dc2626;
}

.cartt-preview-price .original {
    text-decoration: line-through;
    color: #999;
    font-size: 14px;
    font-weight: 400;
    margin-left: 8px;
}

.cartt-preview-rating {
    display: flex;
    align-items: center;
    gap: 4px;
    margin-top: 8px;
}

.cartt-preview-stars {
    color: #fbbf24;
    font-size: 14px;
}

.cartt-preview-rating-count {
    font-size: 12px;
    color: #666;
}

.cartt-preview-button {
    margin-top: 12px;
}

.cartt-preview-button button {
    width: 100%;
    padding: 10px 16px;
    background: #1d2327;
    color: #fff;
    border: none;
    border-radius: 6px;
    font-size: 14px;
    font-weight: 500;
    cursor: pointer;
}

.cartt-preview-badge {
    position: absolute;
    top: 12px;
    left: 12px;
    background: #dc2626;
    color: #fff;
    font-size: 11px;
    font-weight: 600;
    padding: 4px 8px;
    border-radius: 4px;
}

.cartt-preview-wishlist {
    position: absolute;
    top: 12px;
    right: 12px;
    width: 32px;
    height: 32px;
    background: #fff;
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    box-shadow: 0 2px 4px rgba(0,0,0,0.1);
    cursor: pointer;
}

.cartt-preview-wishlist .dashicons {
    font-size: 16px;
    color: #666;
}

/* Modal */
.cartt-modal {
    position: fixed;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: rgba(0,0,0,0.5);
    z-index: 100000;
    display: flex;
    align-items: center;
    justify-content: center;
}

.cartt-modal-content {
    background: #fff;
    border-radius: 8px;
    width: 100%;
    max-width: 480px;
    max-height: 80vh;
    overflow: hidden;
    display: flex;
    flex-direction: column;
}

.cartt-modal-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding: 16px 20px;
    border-bottom: 1px solid #e0e0e0;
}

.cartt-modal-header h3 {
    margin: 0;
    font-size: 16px;
}

.cartt-modal-close {
    background: none;
    border: none;
    font-size: 24px;
    cursor: pointer;
    color: #666;
    line-height: 1;
}

.cartt-modal-body {
    padding: 20px;
    overflow-y: auto;
}

.cartt-modal-footer {
    padding: 16px 20px;
    border-top: 1px solid #e0e0e0;
    display: flex;
    justify-content: flex-end;
    gap: 12px;
}

/* Notice */
#cartt-designer-notice .cartt-notice {
    padding: 12px 16px;
    border-radius: 4px;
    margin-bottom: 16px;
}

#cartt-designer-notice .cartt-notice.success {
    background: #d4edda;
    color: #155724;
}

#cartt-designer-notice .cartt-notice.error {
    background: #f8d7da;
    color: #721c24;
}
</style>

<script src="https://cdn.jsdelivr.net/npm/sortablejs@1.15.0/Sortable.min.js"></script>
<script>
jQuery(document).ready(function($) {
    var nonce = '<?php echo $nonce; ?>';
    var design = <?php echo wp_json_encode($design); ?>;
    
    // Sample products for preview
    var sampleProducts = [
        { title: 'Premium Wireless Headphones', price: 149.99, sale_price: 99.99, rating: 4.5, reviews: 128, image: 'https://images.unsplash.com/photo-1505740420928-5e560c06d30e?w=400&h=400&fit=crop' },
        { title: 'Minimalist Watch', price: 299.99, rating: 5, reviews: 64, image: 'https://images.unsplash.com/photo-1523275335684-37898b6baf30?w=400&h=400&fit=crop' },
        { title: 'Leather Backpack', price: 189.99, rating: 4, reviews: 42, image: 'https://images.unsplash.com/photo-1553062407-98eeb64c6a62?w=400&h=400&fit=crop' },
        { title: 'Smart Fitness Tracker', price: 79.99, sale_price: 59.99, rating: 4.5, reviews: 256, image: 'https://images.unsplash.com/photo-1575311373937-040b8e1fd5b6?w=400&h=400&fit=crop' }
    ];
    
    // Initialize Sortable
    var el = document.getElementById('cartt-elements-list');
    if (el) {
        new Sortable(el, {
            handle: '.cartt-drag-handle',
            animation: 150,
            ghostClass: 'sortable-ghost',
            onEnd: function() {
                updateElementsOrder();
                renderPreview();
            }
        });
    }
    
    // Get enabled elements in order
    function getEnabledElements() {
        var elements = [];
        $('#cartt-elements-list .cartt-element-item').each(function() {
            var $item = $(this);
            if ($item.find('input[type="checkbox"]').is(':checked')) {
                elements.push($item.data('element'));
            }
        });
        return elements;
    }
    
    function updateElementsOrder() {
        design.elements = getEnabledElements();
    }
    
    // Render preview
    function renderPreview() {
        var $grid = $('#cartt-preview-grid');
        var columns = design.grid.columns;
        var device = $('.cartt-preview-container').data('device');
        
        if (device === 'tablet') columns = design.grid.columns_tablet;
        if (device === 'mobile') columns = design.grid.columns_mobile;
        
        $grid.css({
            'grid-template-columns': 'repeat(' + columns + ', 1fr)',
            'gap': design.grid.gap + 'px'
        });
        
        // Update dynamic hover styles
        var shadowSize = design.card.hover_shadow_size || 15;
        var hoverCSS = '.cartt-hover-lift:hover, .cartt-hover-shadow:hover { box-shadow: 0 ' + Math.round(shadowSize * 0.5) + 'px ' + shadowSize + 'px rgba(0,0,0,0.15) !important; }';
        
        if ($('#cartt-dynamic-hover-css').length === 0) {
            $('head').append('<style id="cartt-dynamic-hover-css"></style>');
        }
        $('#cartt-dynamic-hover-css').text(hoverCSS);
        
        var html = '';
        sampleProducts.forEach(function(product, i) {
            html += renderCard(product, i === 0 || i === 3);
        });
        
        $grid.html(html);
    }
    
    function renderCard(product, hasSale) {
        var card = design.card;
        var shadow = {
            'none': 'none',
            'small': '0 1px 3px rgba(0,0,0,0.1)',
            'medium': '0 4px 6px rgba(0,0,0,0.1)',
            'large': '0 10px 25px rgba(0,0,0,0.15)'
        }[card.shadow];
        
        // Hover effect classes
        var hoverClass = 'cartt-hover-' + (card.hover_effect || 'none');
        
        var cardStyle = 'background:' + card.background + ';';
        cardStyle += 'border:' + card.border_width + 'px solid ' + card.border_color + ';';
        cardStyle += 'border-radius:' + card.border_radius + 'px;';
        cardStyle += 'box-shadow:' + shadow + ';';
        cardStyle += 'overflow:hidden;';
        cardStyle += 'display:flex;flex-direction:column;height:100%;'; // Flexbox for button alignment
        
        var imgStyle = '';
        var aspectRatio = design.image.aspect_ratio;
        if (aspectRatio === 'square') imgStyle = 'padding-bottom:100%;';
        else if (aspectRatio === 'portrait') imgStyle = 'padding-bottom:133%;';
        else if (aspectRatio === 'landscape') imgStyle = 'padding-bottom:75%;';
        
        var html = '<div class="cartt-preview-card ' + hoverClass + '" style="' + cardStyle + '">';
        
        // Track if we've output the button yet
        var hasButton = design.elements.includes('add_to_cart');
        var buttonHtml = '';
        
        design.elements.forEach(function(el) {
            switch(el) {
                case 'image':
                    html += '<div class="cartt-preview-image" style="overflow:hidden;position:relative;flex-shrink:0;">';
                    html += '<div style="' + imgStyle + 'position:relative;">';
                    html += '<img src="' + product.image + '" style="position:absolute;top:0;left:0;width:100%;height:100%;object-fit:cover;transition:transform 0.3s;">';
                    html += '</div></div>';
                    break;
                
                case 'sale_badge':
                    if (hasSale) {
                        html += '<span class="cartt-preview-badge" style="position:absolute;top:12px;left:12px;z-index:2;background:' + (design.sale_badge.bg_color || '#dc2626') + ';color:' + (design.sale_badge.text_color || '#fff') + ';font-size:11px;font-weight:600;padding:4px 10px;border-radius:4px;">' + (design.sale_badge.text || 'Sale') + '</span>';
                    }
                    break;
                    
                case 'wishlist':
                    var wishPos = (design.wishlist?.position || 'top-right') === 'top-left' ? 'left:12px;' : 'right:12px;';
                    var wishBg = design.wishlist?.bg_color || '#ffffff';
                    var wishIcon = design.wishlist?.icon_color || '#1d2327';
                    html += '<button class="cartt-preview-wishlist" style="position:absolute;top:12px;' + wishPos + 'z-index:2;width:32px;height:32px;background:' + wishBg + ';border:none;border-radius:50%;padding:0;cursor:pointer;display:flex;align-items:center;justify-content:center;box-shadow:0 2px 4px rgba(0,0,0,0.1);"><svg width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="' + wishIcon + '" stroke-width="2"><path d="M20.84 4.61a5.5 5.5 0 0 0-7.78 0L12 5.67l-1.06-1.06a5.5 5.5 0 0 0-7.78 7.78l1.06 1.06L12 21.23l7.78-7.78 1.06-1.06a5.5 5.5 0 0 0 0-7.78z"></path></svg></button>';
                    break;
                    
                case 'quick_view':
                    var qvBg = design.quick_view?.bg_color || '#ffffff';
                    var qvIcon = design.quick_view?.icon_color || '#1d2327';
                    var qvStyle = design.quick_view?.style || 'icon';
                    if (qvStyle === 'button') {
                        html += '<button class="cartt-preview-quickview" style="position:absolute;top:50%;left:50%;transform:translate(-50%,-50%);z-index:2;padding:10px 20px;background:' + qvBg + ';color:' + qvIcon + ';border:none;border-radius:4px;font-size:13px;font-weight:500;cursor:pointer;opacity:0;transition:opacity 0.3s;box-shadow:0 2px 8px rgba(0,0,0,0.15);">Quick View</button>';
                    } else {
                        html += '<button class="cartt-preview-quickview" style="position:absolute;top:50%;left:50%;transform:translate(-50%,-50%);z-index:2;width:40px;height:40px;background:' + qvBg + ';border:none;border-radius:50%;cursor:pointer;opacity:0;transition:opacity 0.3s;display:flex;align-items:center;justify-content:center;box-shadow:0 2px 8px rgba(0,0,0,0.15);"><svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="' + qvIcon + '" stroke-width="2"><circle cx="12" cy="12" r="3"></circle><path d="M12 5c-7 0-10 7-10 7s3 7 10 7 10-7 10-7-3-7-10-7z"></path></svg></button>';
                    }
                    break;
                    
                case 'title':
                    html += '<div style="padding:' + (design.title.margin_top || 12) + 'px 16px 0;">';
                    html += '<h3 class="cartt-preview-title" style="margin:0;font-size:' + design.title.font_size + 'px;font-weight:' + design.title.font_weight + ';color:' + design.title.color + ';overflow:hidden;display:-webkit-box;-webkit-line-clamp:' + (design.title.lines || 2) + ';-webkit-box-orient:vertical;">' + product.title + '</h3>';
                    html += '</div>';
                    break;
                    
                case 'price':
                    var priceHtml = '';
                    if (hasSale && product.sale_price) {
                        priceHtml = '<span style="color:' + design.price.sale_color + '">$' + product.sale_price.toFixed(2) + '</span>';
                        priceHtml += '<span style="text-decoration:line-through;color:#999;font-size:0.75em;font-weight:400;margin-left:8px;">$' + product.price.toFixed(2) + '</span>';
                    } else {
                        priceHtml = '$' + product.price.toFixed(2);
                    }
                    html += '<div style="padding:0 16px;margin-top:' + design.price.margin_top + 'px;font-size:' + design.price.font_size + 'px;font-weight:' + design.price.font_weight + ';color:' + design.price.color + ';">' + priceHtml + '</div>';
                    break;
                    
                case 'rating':
                    var stars = '';
                    for (var i = 0; i < 5; i++) {
                        stars += i < Math.floor(product.rating) ? '★' : '☆';
                    }
                    html += '<div style="padding:0 16px;margin-top:' + design.rating.margin_top + 'px;display:flex;align-items:center;gap:4px;">';
                    html += '<span style="color:' + design.rating.star_color + ';font-size:14px;">' + stars + '</span>';
                    if (design.rating.show_count) {
                        html += '<span style="color:#666;font-size:12px;">(' + product.reviews + ')</span>';
                    }
                    html += '</div>';
                    break;
                    
                case 'description':
                    var descStyle = 'padding:0 16px;margin-top:' + (design.description?.margin_top || 8) + 'px;font-size:' + (design.description?.font_size || 13) + 'px;color:' + (design.description?.color || '#666666') + ';overflow:hidden;display:-webkit-box;-webkit-line-clamp:' + (design.description?.lines || 2) + ';-webkit-box-orient:vertical;';
                    html += '<div style="' + descStyle + '">Premium quality product with excellent features and craftsmanship.</div>';
                    break;
                    
                case 'sku':
                    var skuPrefix = design.sku?.prefix || 'SKU:';
                    var skuStyle = 'padding:0 16px;margin-top:' + (design.sku?.margin_top || 8) + 'px;font-size:' + (design.sku?.font_size || 12) + 'px;color:' + (design.sku?.color || '#999999') + ';';
                    html += '<div style="' + skuStyle + '">' + skuPrefix + ' PRD-' + Math.floor(Math.random() * 9000 + 1000) + '</div>';
                    break;
                    
                case 'stock':
                    var inStock = Math.random() > 0.3;
                    var stockColor = inStock ? (design.stock?.in_stock_color || '#16a34a') : (design.stock?.out_of_stock_color || '#dc2626');
                    var stockStyle = 'padding:0 16px;margin-top:' + (design.stock?.margin_top || 8) + 'px;font-size:' + (design.stock?.font_size || 13) + 'px;font-weight:500;color:' + stockColor + ';';
                    html += '<div style="' + stockStyle + '">' + (inStock ? 'In Stock' : 'Out of Stock') + '</div>';
                    break;
                    
                case 'add_to_cart':
                    // Store button HTML to add at the end with margin-top:auto
                    var btnStyle = 'width:100%;padding:10px 16px;background:' + design.add_to_cart.bg_color + ';color:' + design.add_to_cart.text_color + ';border:none;border-radius:' + design.add_to_cart.border_radius + 'px;font-size:14px;font-weight:500;cursor:pointer;transition:opacity 0.2s;';
                    if (!design.add_to_cart.full_width) btnStyle = btnStyle.replace('width:100%;', '');
                    var btnText = design.add_to_cart.button_text || 'Add to Cart';
                    buttonHtml = '<div style="padding:16px;margin-top:auto;">';
                    buttonHtml += '<button style="' + btnStyle + '">' + btnText + '</button>';
                    buttonHtml += '</div>';
                    break;
            }
        });
        
        // Add button at the end with margin-top:auto to push to bottom
        html += buttonHtml;
        
        html += '</div>';
        return html;
    }
    
    // Element checkbox change
    $(document).on('change', '.cartt-element-item input[type="checkbox"]', function() {
        updateElementsOrder();
        renderPreview();
    });
    
    // Card settings change
    $('#card-background, #card-border-color, #card-border-width, #card-border-radius, #card-shadow, #card-hover, #card-hover-shadow').on('input change', function() {
        var hoverEffect = $('#card-hover').val();
        design.card = {
            background: $('#card-background').val(),
            border_color: $('#card-border-color').val(),
            border_width: parseInt($('#card-border-width').val()),
            border_radius: parseInt($('#card-border-radius').val()),
            shadow: $('#card-shadow').val(),
            hover_effect: hoverEffect,
            hover_shadow_size: parseInt($('#card-hover-shadow').val())
        };
        $('#card-border-radius').next('.cartt-range-value').text($('#card-border-radius').val() + 'px');
        $('#card-hover-shadow').next('.cartt-range-value').text($('#card-hover-shadow').val() + 'px');
        
        // Show/hide hover shadow control
        if (hoverEffect === 'none' || hoverEffect === 'border') {
            $('#hover-shadow-row').hide();
        } else {
            $('#hover-shadow-row').show();
        }
        
        renderPreview();
    });
    
    // Grid settings change
    $('#grid-columns, #grid-columns-tablet, #grid-columns-mobile, #grid-gap').on('input change', function() {
        design.grid = {
            columns: parseInt($('#grid-columns').val()),
            columns_tablet: parseInt($('#grid-columns-tablet').val()),
            columns_mobile: parseInt($('#grid-columns-mobile').val()),
            gap: parseInt($('#grid-gap').val())
        };
        $('#grid-gap').next('.cartt-range-value').text($('#grid-gap').val() + 'px');
        renderPreview();
    });
    
    // Device preview
    $('.cartt-preview-device').on('click', function() {
        var device = $(this).data('device');
        $('.cartt-preview-device').removeClass('active');
        $(this).addClass('active');
        $('.cartt-preview-container').attr('data-device', device);
        renderPreview();
    });
    
    // Presets
    $('.cartt-preset-btn').on('click', function() {
        var preset = $(this).data('preset');
        applyPreset(preset);
        $('.cartt-preset-btn').removeClass('active');
        $(this).addClass('active');
        design.preset = preset;
        renderPreview();
    });
    
    function applyPreset(preset) {
        var presets = {
            minimal: {
                card: { background: '#ffffff', border_color: '#f0f0f0', border_width: 0, border_radius: 0, shadow: 'none', hover_effect: 'none', hover_shadow_size: 15 },
                title: { font_size: 14, font_weight: '400', color: '#333333' },
                price: { font_size: 16, font_weight: '600', color: '#000000', sale_color: '#dc2626' },
                add_to_cart: { bg_color: '#000000', text_color: '#ffffff', border_radius: 0, full_width: true }
            },
            modern: {
                card: { background: '#ffffff', border_color: '#e0e0e0', border_width: 1, border_radius: 8, shadow: 'small', hover_effect: 'lift', hover_shadow_size: 15 },
                title: { font_size: 16, font_weight: '600', color: '#1d2327' },
                price: { font_size: 18, font_weight: '700', color: '#1d2327', sale_color: '#dc2626' },
                add_to_cart: { bg_color: '#1d2327', text_color: '#ffffff', border_radius: 6, full_width: true }
            },
            classic: {
                card: { background: '#ffffff', border_color: '#d0d0d0', border_width: 1, border_radius: 4, shadow: 'none', hover_effect: 'shadow', hover_shadow_size: 12 },
                title: { font_size: 15, font_weight: '500', color: '#333333' },
                price: { font_size: 17, font_weight: '600', color: '#333333', sale_color: '#b91c1c' },
                add_to_cart: { bg_color: '#2563eb', text_color: '#ffffff', border_radius: 4, full_width: true }
            },
            bold: {
                card: { background: '#ffffff', border_color: '#000000', border_width: 2, border_radius: 0, shadow: 'none', hover_effect: 'border', hover_shadow_size: 15 },
                title: { font_size: 18, font_weight: '700', color: '#000000' },
                price: { font_size: 20, font_weight: '800', color: '#000000', sale_color: '#dc2626' },
                add_to_cart: { bg_color: '#000000', text_color: '#ffffff', border_radius: 0, full_width: true }
            },
            elegant: {
                card: { background: '#fafafa', border_color: '#e5e5e5', border_width: 1, border_radius: 12, shadow: 'medium', hover_effect: 'lift', hover_shadow_size: 18 },
                title: { font_size: 15, font_weight: '500', color: '#374151' },
                price: { font_size: 17, font_weight: '600', color: '#374151', sale_color: '#9333ea' },
                add_to_cart: { bg_color: '#374151', text_color: '#ffffff', border_radius: 24, full_width: true }
            }
        };
        
        if (presets[preset]) {
            $.extend(true, design, presets[preset]);
            updateFormFromDesign();
        }
    }
    
    function updateFormFromDesign() {
        $('#card-background').val(design.card.background);
        $('#card-border-color').val(design.card.border_color);
        $('#card-border-width').val(design.card.border_width);
        $('#card-border-radius').val(design.card.border_radius);
        $('#card-shadow').val(design.card.shadow);
        $('#card-hover').val(design.card.hover_effect);
        $('#card-border-radius').next('.cartt-range-value').text(design.card.border_radius + 'px');
        
        // Hover shadow
        var hoverShadow = design.card.hover_shadow_size || 15;
        $('#card-hover-shadow').val(hoverShadow);
        $('#card-hover-shadow').next('.cartt-range-value').text(hoverShadow + 'px');
        
        // Show/hide hover shadow based on effect
        if (design.card.hover_effect === 'none' || design.card.hover_effect === 'border') {
            $('#hover-shadow-row').hide();
        } else {
            $('#hover-shadow-row').show();
        }
    }
    
    // Element edit modal
    $(document).on('click', '.cartt-element-edit', function() {
        var element = $(this).data('element');
        showElementModal(element);
    });
    
    function showElementModal(element) {
        var titles = {
            image: 'Image Settings',
            title: 'Title Settings',
            price: 'Price Settings',
            rating: 'Rating Settings',
            add_to_cart: 'Button Settings',
            sale_badge: 'Sale Badge Settings',
            wishlist: 'Wishlist Settings',
            quick_view: 'Quick View Settings',
            description: 'Description Settings',
            sku: 'SKU Settings',
            stock: 'Stock Status Settings'
        };
        
        $('#cartt-modal-title').text(titles[element] || 'Element Settings');
        
        var html = '';
        switch(element) {
            case 'image':
                html = '<div class="cartt-setting-row"><label>Aspect Ratio</label><select id="modal-image-ratio">';
                html += '<option value="square"' + (design.image.aspect_ratio === 'square' ? ' selected' : '') + '>Square (1:1)</option>';
                html += '<option value="portrait"' + (design.image.aspect_ratio === 'portrait' ? ' selected' : '') + '>Portrait (3:4)</option>';
                html += '<option value="landscape"' + (design.image.aspect_ratio === 'landscape' ? ' selected' : '') + '>Landscape (4:3)</option>';
                html += '</select></div>';
                html += '<div class="cartt-setting-row"><label>Border Radius</label><input type="range" id="modal-image-radius" value="' + design.image.border_radius + '" min="0" max="24"><span>' + design.image.border_radius + 'px</span></div>';
                break;
                
            case 'title':
                html = '<div class="cartt-setting-row"><label>Font Size</label><input type="number" id="modal-title-size" value="' + design.title.font_size + '" min="12" max="24">px</div>';
                html += '<div class="cartt-setting-row"><label>Font Weight</label><select id="modal-title-weight">';
                html += '<option value="400"' + (design.title.font_weight === '400' ? ' selected' : '') + '>Normal</option>';
                html += '<option value="500"' + (design.title.font_weight === '500' ? ' selected' : '') + '>Medium</option>';
                html += '<option value="600"' + (design.title.font_weight === '600' ? ' selected' : '') + '>Semi Bold</option>';
                html += '<option value="700"' + (design.title.font_weight === '700' ? ' selected' : '') + '>Bold</option>';
                html += '</select></div>';
                html += '<div class="cartt-setting-row"><label>Color</label><input type="color" id="modal-title-color" value="' + design.title.color + '"></div>';
                html += '<div class="cartt-setting-row"><label>Max Lines</label><input type="number" id="modal-title-lines" value="' + (design.title.lines || 2) + '" min="1" max="4"></div>';
                break;
                
            case 'price':
                html = '<div class="cartt-setting-row"><label>Font Size</label><input type="number" id="modal-price-size" value="' + design.price.font_size + '" min="12" max="28">px</div>';
                html += '<div class="cartt-setting-row"><label>Color</label><input type="color" id="modal-price-color" value="' + design.price.color + '"></div>';
                html += '<div class="cartt-setting-row"><label>Sale Color</label><input type="color" id="modal-price-sale" value="' + design.price.sale_color + '"></div>';
                break;
                
            case 'rating':
                html = '<div class="cartt-setting-row"><label>Star Color</label><input type="color" id="modal-rating-color" value="' + design.rating.star_color + '"></div>';
                html += '<div class="cartt-setting-row"><label>Show Count</label><input type="checkbox" id="modal-rating-count"' + (design.rating.show_count ? ' checked' : '') + '></div>';
                break;
                
            case 'add_to_cart':
                html = '<div class="cartt-setting-row"><label>Button Text</label><input type="text" id="modal-btn-label" value="' + (design.add_to_cart.button_text || 'Add to Cart') + '" style="width:140px;"></div>';
                html += '<div class="cartt-setting-row"><label>Background</label><input type="color" id="modal-btn-bg" value="' + design.add_to_cart.bg_color + '"></div>';
                html += '<div class="cartt-setting-row"><label>Text Color</label><input type="color" id="modal-btn-text" value="' + design.add_to_cart.text_color + '"></div>';
                html += '<div class="cartt-setting-row"><label>Border Radius</label><input type="range" id="modal-btn-radius" value="' + design.add_to_cart.border_radius + '" min="0" max="24"><span>' + design.add_to_cart.border_radius + 'px</span></div>';
                html += '<div class="cartt-setting-row"><label>Full Width</label><input type="checkbox" id="modal-btn-full"' + (design.add_to_cart.full_width ? ' checked' : '') + '></div>';
                break;
                
            case 'sale_badge':
                html = '<div class="cartt-setting-row"><label>Background</label><input type="color" id="modal-badge-bg" value="' + (design.sale_badge.bg_color || '#dc2626') + '"></div>';
                html += '<div class="cartt-setting-row"><label>Text Color</label><input type="color" id="modal-badge-text" value="' + (design.sale_badge.text_color || '#ffffff') + '"></div>';
                html += '<div class="cartt-setting-row"><label>Text</label><input type="text" id="modal-badge-label" value="' + (design.sale_badge.text || 'Sale') + '"></div>';
                break;
                
            case 'description':
                html = '<div class="cartt-setting-row"><label>Font Size</label><input type="number" id="modal-desc-size" value="' + (design.description?.font_size || 13) + '" min="10" max="18">px</div>';
                html += '<div class="cartt-setting-row"><label>Color</label><input type="color" id="modal-desc-color" value="' + (design.description?.color || '#666666') + '"></div>';
                html += '<div class="cartt-setting-row"><label>Max Lines</label><input type="number" id="modal-desc-lines" value="' + (design.description?.lines || 2) + '" min="1" max="5"></div>';
                break;
                
            case 'sku':
                html = '<div class="cartt-setting-row"><label>Font Size</label><input type="number" id="modal-sku-size" value="' + (design.sku?.font_size || 12) + '" min="10" max="16">px</div>';
                html += '<div class="cartt-setting-row"><label>Color</label><input type="color" id="modal-sku-color" value="' + (design.sku?.color || '#999999') + '"></div>';
                html += '<div class="cartt-setting-row"><label>Prefix</label><input type="text" id="modal-sku-prefix" value="' + (design.sku?.prefix || 'SKU:') + '"></div>';
                break;
                
            case 'stock':
                html = '<div class="cartt-setting-row"><label>Font Size</label><input type="number" id="modal-stock-size" value="' + (design.stock?.font_size || 13) + '" min="10" max="16">px</div>';
                html += '<div class="cartt-setting-row"><label>In Stock Color</label><input type="color" id="modal-stock-in" value="' + (design.stock?.in_stock_color || '#16a34a') + '"></div>';
                html += '<div class="cartt-setting-row"><label>Out of Stock Color</label><input type="color" id="modal-stock-out" value="' + (design.stock?.out_of_stock_color || '#dc2626') + '"></div>';
                break;
                
            case 'quick_view':
                html = '<div class="cartt-setting-row"><label>Style</label><select id="modal-qv-style">';
                html += '<option value="icon"' + ((design.quick_view?.style || 'icon') === 'icon' ? ' selected' : '') + '>Icon Only</option>';
                html += '<option value="button"' + ((design.quick_view?.style) === 'button' ? ' selected' : '') + '>Button</option>';
                html += '</select></div>';
                html += '<div class="cartt-setting-row"><label>Background</label><input type="color" id="modal-qv-bg" value="' + (design.quick_view?.bg_color || '#ffffff') + '"></div>';
                html += '<div class="cartt-setting-row"><label>Icon Color</label><input type="color" id="modal-qv-icon" value="' + (design.quick_view?.icon_color || '#1d2327') + '"></div>';
                break;
                
            case 'wishlist':
                html = '<div class="cartt-setting-row"><label>Position</label><select id="modal-wish-pos">';
                html += '<option value="top-right"' + ((design.wishlist?.position || 'top-right') === 'top-right' ? ' selected' : '') + '>Top Right</option>';
                html += '<option value="top-left"' + ((design.wishlist?.position) === 'top-left' ? ' selected' : '') + '>Top Left</option>';
                html += '</select></div>';
                html += '<div class="cartt-setting-row"><label>Background</label><input type="color" id="modal-wish-bg" value="' + (design.wishlist?.bg_color || '#ffffff') + '"></div>';
                html += '<div class="cartt-setting-row"><label>Icon Color</label><input type="color" id="modal-wish-icon" value="' + (design.wishlist?.icon_color || '#1d2327') + '"></div>';
                html += '<div class="cartt-setting-row"><label>Active Color</label><input type="color" id="modal-wish-active" value="' + (design.wishlist?.active_color || '#dc2626') + '"></div>';
                break;
        }
        
        $('#cartt-modal-body').html(html);
        $('#cartt-element-modal').data('element', element).show();
    }
    
    // Modal close
    $('.cartt-modal-close').on('click', function() {
        $('#cartt-element-modal').hide();
    });
    
    // Modal save
    $('#cartt-modal-save').on('click', function() {
        var element = $('#cartt-element-modal').data('element');
        
        switch(element) {
            case 'image':
                design.image.aspect_ratio = $('#modal-image-ratio').val();
                design.image.border_radius = parseInt($('#modal-image-radius').val());
                break;
            case 'title':
                design.title.font_size = parseInt($('#modal-title-size').val());
                design.title.font_weight = $('#modal-title-weight').val();
                design.title.color = $('#modal-title-color').val();
                design.title.lines = parseInt($('#modal-title-lines').val());
                break;
            case 'price':
                design.price.font_size = parseInt($('#modal-price-size').val());
                design.price.color = $('#modal-price-color').val();
                design.price.sale_color = $('#modal-price-sale').val();
                break;
            case 'rating':
                design.rating.star_color = $('#modal-rating-color').val();
                design.rating.show_count = $('#modal-rating-count').is(':checked');
                break;
            case 'add_to_cart':
                design.add_to_cart.button_text = $('#modal-btn-label').val() || 'Add to Cart';
                design.add_to_cart.bg_color = $('#modal-btn-bg').val();
                design.add_to_cart.text_color = $('#modal-btn-text').val();
                design.add_to_cart.border_radius = parseInt($('#modal-btn-radius').val());
                design.add_to_cart.full_width = $('#modal-btn-full').is(':checked');
                break;
            case 'sale_badge':
                design.sale_badge.bg_color = $('#modal-badge-bg').val();
                design.sale_badge.text_color = $('#modal-badge-text').val();
                design.sale_badge.text = $('#modal-badge-label').val();
                break;
            case 'description':
                if (!design.description) design.description = {};
                design.description.font_size = parseInt($('#modal-desc-size').val());
                design.description.color = $('#modal-desc-color').val();
                design.description.lines = parseInt($('#modal-desc-lines').val());
                break;
            case 'sku':
                if (!design.sku) design.sku = {};
                design.sku.font_size = parseInt($('#modal-sku-size').val());
                design.sku.color = $('#modal-sku-color').val();
                design.sku.prefix = $('#modal-sku-prefix').val();
                break;
            case 'stock':
                if (!design.stock) design.stock = {};
                design.stock.font_size = parseInt($('#modal-stock-size').val());
                design.stock.in_stock_color = $('#modal-stock-in').val();
                design.stock.out_of_stock_color = $('#modal-stock-out').val();
                break;
            case 'quick_view':
                if (!design.quick_view) design.quick_view = {};
                design.quick_view.style = $('#modal-qv-style').val();
                design.quick_view.bg_color = $('#modal-qv-bg').val();
                design.quick_view.icon_color = $('#modal-qv-icon').val();
                break;
            case 'wishlist':
                if (!design.wishlist) design.wishlist = {};
                design.wishlist.position = $('#modal-wish-pos').val();
                design.wishlist.bg_color = $('#modal-wish-bg').val();
                design.wishlist.icon_color = $('#modal-wish-icon').val();
                design.wishlist.active_color = $('#modal-wish-active').val();
                break;
        }
        
        $('#cartt-element-modal').hide();
        renderPreview();
    });
    
    // Save design
    $('#cartt-save-design').on('click', function() {
        var $btn = $(this);
        $btn.prop('disabled', true).text('Saving...');
        
        $.post(ajaxurl, {
            action: 'cartt_save_card_design',
            nonce: nonce,
            design: JSON.stringify(design)
        }, function(response) {
            $btn.prop('disabled', false).text('Save Design');
            if (response.success) {
                $('#cartt-designer-notice').html('<div class="cartt-notice success">Design saved successfully!</div>');
                setTimeout(function() { $('#cartt-designer-notice').html(''); }, 3000);
            } else {
                $('#cartt-designer-notice').html('<div class="cartt-notice error">' + (response.data || 'Error saving design') + '</div>');
            }
        });
    });
    
    // Reset design
    $('#cartt-reset-design').on('click', function() {
        if (confirm('Reset to default design? This cannot be undone.')) {
            design = <?php echo wp_json_encode($defaults); ?>;
            updateFormFromDesign();
            
            // Reset checkboxes
            $('#cartt-elements-list input[type="checkbox"]').each(function() {
                var el = $(this).data('element');
                $(this).prop('checked', design.elements.includes(el));
            });
            
            // Reset preset
            $('.cartt-preset-btn').removeClass('active');
            $('.cartt-preset-btn[data-preset="modern"]').addClass('active');
            
            renderPreview();
        }
    });
    
    // Initial render
    renderPreview();
});
</script>
