<?php
/**
 * Advanced Analytics Admin View
 * @since 1.4.0
 */

if (!defined('ABSPATH')) {
    exit;
}

global $wpdb;

$subtab = isset($_GET['subtab']) ? sanitize_text_field($_GET['subtab']) : 'overview';
$period = isset($_GET['period']) ? sanitize_text_field($_GET['period']) : '30days';

// Date range calculation
$dateFilter = '';
switch ($period) {
    case '7days':
        $dateFilter = "AND created_at >= DATE_SUB(NOW(), INTERVAL 7 DAY)";
        $periodLabel = 'Last 7 Days';
        break;
    case '30days':
        $dateFilter = "AND created_at >= DATE_SUB(NOW(), INTERVAL 30 DAY)";
        $periodLabel = 'Last 30 Days';
        break;
    case '90days':
        $dateFilter = "AND created_at >= DATE_SUB(NOW(), INTERVAL 90 DAY)";
        $periodLabel = 'Last 90 Days';
        break;
    case 'year':
        $dateFilter = "AND created_at >= DATE_SUB(NOW(), INTERVAL 1 YEAR)";
        $periodLabel = 'Last Year';
        break;
    default:
        $periodLabel = 'All Time';
}

// Real-time stats
$activeVisitors = $wpdb->get_var(
    "SELECT COUNT(DISTINCT session_id) FROM {$wpdb->prefix}cartt_visitor_sessions 
    WHERE last_activity >= DATE_SUB(NOW(), INTERVAL 5 MINUTE)"
);

$todayOrders = $wpdb->get_var(
    "SELECT COUNT(*) FROM {$wpdb->prefix}cartt_orders WHERE DATE(created_at) = CURDATE()"
);

$todayRevenue = $wpdb->get_var(
    "SELECT COALESCE(SUM(total), 0) FROM {$wpdb->prefix}cartt_orders 
    WHERE DATE(created_at) = CURDATE() AND status IN ('completed', 'processing')"
);

// Conversion funnel data
$funnelData = [];
$funnelData['visitors'] = $wpdb->get_var(
    "SELECT COUNT(DISTINCT session_id) FROM {$wpdb->prefix}cartt_analytics_events 
    WHERE event_type = 'visitor' {$dateFilter}"
) ?: 0;
$funnelData['product_views'] = $wpdb->get_var(
    "SELECT COUNT(DISTINCT session_id) FROM {$wpdb->prefix}cartt_analytics_events 
    WHERE event_type = 'product_view' {$dateFilter}"
) ?: 0;
$funnelData['add_to_cart'] = $wpdb->get_var(
    "SELECT COUNT(DISTINCT session_id) FROM {$wpdb->prefix}cartt_analytics_events 
    WHERE event_type = 'add_to_cart' {$dateFilter}"
) ?: 0;
$funnelData['checkout'] = $wpdb->get_var(
    "SELECT COUNT(DISTINCT session_id) FROM {$wpdb->prefix}cartt_analytics_events 
    WHERE event_type = 'checkout_start' {$dateFilter}"
) ?: 0;
$funnelData['purchase'] = $wpdb->get_var(
    "SELECT COUNT(DISTINCT session_id) FROM {$wpdb->prefix}cartt_analytics_events 
    WHERE event_type = 'purchase' {$dateFilter}"
) ?: 0;

// Calculate conversion rates
$conversionRate = $funnelData['visitors'] > 0 
    ? round(($funnelData['purchase'] / $funnelData['visitors']) * 100, 2) 
    : 0;

// Top products
$topProducts = $wpdb->get_results(
    "SELECT p.id, p.name, 
        COUNT(DISTINCT e.session_id) as views,
        (SELECT COUNT(*) FROM {$wpdb->prefix}cartt_order_items WHERE product_id = p.id) as purchases,
        (SELECT SUM(total) FROM {$wpdb->prefix}cartt_order_items WHERE product_id = p.id) as revenue
    FROM {$wpdb->prefix}cartt_products p
    LEFT JOIN {$wpdb->prefix}cartt_analytics_events e ON p.id = e.product_id AND e.event_type = 'product_view'
    WHERE p.status = 'publish'
    GROUP BY p.id
    ORDER BY revenue DESC
    LIMIT 10"
);

// Revenue by source
$revenueBySources = $wpdb->get_results(
    "SELECT 
        COALESCE(utm_source, 'Direct') as source,
        COUNT(DISTINCT e.session_id) as sessions,
        COUNT(DISTINCT CASE WHEN e.event_type = 'purchase' THEN e.order_id END) as orders,
        COALESCE(SUM(CASE WHEN e.event_type = 'purchase' THEN o.total END), 0) as revenue
    FROM {$wpdb->prefix}cartt_analytics_events e
    LEFT JOIN {$wpdb->prefix}cartt_orders o ON e.order_id = o.id
    WHERE 1=1 {$dateFilter}
    GROUP BY utm_source
    ORDER BY revenue DESC
    LIMIT 10"
);
?>

<div class="wrap cartt-admin">
    <h1 class="wp-heading-inline">Analytics Dashboard</h1>
    <hr class="wp-header-end">

    <!-- Period Selector -->
    <div style="margin-bottom: 20px;">
        <label>Time Period:</label>
        <select onchange="location.href='<?php echo admin_url('admin.php?page=cartt-finance&tab=analytics&subtab=' . $subtab); ?>&period=' + this.value">
            <option value="7days" <?php selected($period, '7days'); ?>>Last 7 Days</option>
            <option value="30days" <?php selected($period, '30days'); ?>>Last 30 Days</option>
            <option value="90days" <?php selected($period, '90days'); ?>>Last 90 Days</option>
            <option value="year" <?php selected($period, 'year'); ?>>Last Year</option>
            <option value="all" <?php selected($period, 'all'); ?>>All Time</option>
        </select>
    </div>

    <!-- Real-Time Stats -->
    <div class="cartt-stats-row" style="display: grid; grid-template-columns: repeat(4, 1fr); gap: 20px; margin-bottom: 30px;">
        <div class="cartt-stat-card" style="border-left: 4px solid #4CAF50;">
            <div class="stat-value" style="color: #4CAF50;"><?php echo intval($activeVisitors); ?></div>
            <div class="stat-label">Active Visitors (5 min)</div>
        </div>
        <div class="cartt-stat-card">
            <div class="stat-value"><?php echo intval($todayOrders); ?></div>
            <div class="stat-label">Orders Today</div>
        </div>
        <div class="cartt-stat-card">
            <div class="stat-value"><?php echo \Cartt\Services\CurrencyService::formatPrice($todayRevenue ?? 0); ?></div>
            <div class="stat-label">Revenue Today</div>
        </div>
        <div class="cartt-stat-card" style="border-left: 4px solid #2196F3;">
            <div class="stat-value"><?php echo $conversionRate; ?>%</div>
            <div class="stat-label">Conversion Rate (<?php echo $periodLabel; ?>)</div>
        </div>
    </div>

    <!-- Tabs -->
    <h2 class="nav-tab-wrapper">
        <a href="<?php echo admin_url('admin.php?page=cartt-finance&tab=analytics&subtab=overview&period=' . $period); ?>" class="nav-tab <?php echo $subtab === 'overview' ? 'nav-tab-active' : ''; ?>">Overview</a>
        <a href="<?php echo admin_url('admin.php?page=cartt-finance&tab=analytics&subtab=funnel&period=' . $period); ?>" class="nav-tab <?php echo $subtab === 'funnel' ? 'nav-tab-active' : ''; ?>">Conversion Funnel</a>
        <a href="<?php echo admin_url('admin.php?page=cartt-finance&tab=analytics&subtab=products&period=' . $period); ?>" class="nav-tab <?php echo $subtab === 'products' ? 'nav-tab-active' : ''; ?>">Product Performance</a>
        <a href="<?php echo admin_url('admin.php?page=cartt-finance&tab=analytics&subtab=customers&period=' . $period); ?>" class="nav-tab <?php echo $subtab === 'customers' ? 'nav-tab-active' : ''; ?>">Customer Insights</a>
        <a href="<?php echo admin_url('admin.php?page=cartt-finance&tab=analytics&subtab=sources&period=' . $period); ?>" class="nav-tab <?php echo $subtab === 'sources' ? 'nav-tab-active' : ''; ?>">Traffic Sources</a>
    </h2>

    <div style="background: #fff; padding: 20px; border: 1px solid #c3c4c7; border-top: none;">
        <?php if ($subtab === 'overview'): ?>
        
        <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 20px;">
            <!-- Conversion Funnel Summary -->
            <div class="cartt-card">
                <h2>Conversion Funnel</h2>
                <div style="padding: 20px 0;">
                    <?php 
                    $funnelSteps = [
                        'visitors' => ['label' => 'Visitors', 'color' => '#e3f2fd'],
                        'product_views' => ['label' => 'Product Views', 'color' => '#bbdefb'],
                        'add_to_cart' => ['label' => 'Add to Cart', 'color' => '#90caf9'],
                        'checkout' => ['label' => 'Checkout Started', 'color' => '#64b5f6'],
                        'purchase' => ['label' => 'Purchases', 'color' => '#4CAF50']
                    ];
                    $maxValue = max(array_values($funnelData)) ?: 1;
                    
                    foreach ($funnelSteps as $key => $step):
                        $value = $funnelData[$key];
                        $width = ($value / $maxValue) * 100;
                        $prevKey = array_search($key, array_keys($funnelSteps));
                        $prevValue = $prevKey > 0 ? array_values($funnelData)[$prevKey - 1] : $value;
                        $dropRate = $prevValue > 0 ? round((1 - $value / $prevValue) * 100, 1) : 0;
                    ?>
                    <div style="margin-bottom: 15px;">
                        <div style="display: flex; justify-content: space-between; margin-bottom: 5px;">
                            <span><?php echo $step['label']; ?></span>
                            <span>
                                <strong><?php echo number_format($value); ?></strong>
                                <?php if ($key !== 'visitors' && $dropRate > 0): ?>
                                <span style="color: #f44336; font-size: 11px; margin-left: 5px;">-<?php echo $dropRate; ?>%</span>
                                <?php endif; ?>
                            </span>
                        </div>
                        <div style="background: #eee; border-radius: 4px; overflow: hidden;">
                            <div style="background: <?php echo $step['color']; ?>; height: 24px; width: <?php echo $width; ?>%;"></div>
                        </div>
                    </div>
                    <?php endforeach; ?>
                </div>
            </div>
            
            <!-- Top Products -->
            <div class="cartt-card">
                <h2>Top Products by Revenue</h2>
                <?php if (empty($topProducts)): ?>
                <p style="color: #666; padding: 20px 0; text-align: center;">No product data yet.</p>
                <?php else: ?>
                <table class="wp-list-table widefat fixed striped">
                    <thead>
                        <tr>
                            <th>Product</th>
                            <th>Views</th>
                            <th>Sales</th>
                            <th>Revenue</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($topProducts as $product): ?>
                        <tr>
                            <td><?php echo esc_html($product->name); ?></td>
                            <td><?php echo number_format($product->views); ?></td>
                            <td><?php echo number_format($product->purchases); ?></td>
                            <td><?php echo \Cartt\Services\CurrencyService::formatPrice($product->revenue ?? 0); ?></td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
                <?php endif; ?>
            </div>
        </div>
        
        <!-- Traffic Sources -->
        <div class="cartt-card" style="margin-top: 20px;">
            <h2>Revenue by Traffic Source</h2>
            <?php if (empty($revenueBySources)): ?>
            <p style="color: #666; padding: 20px 0; text-align: center;">No attribution data yet. Make sure UTM tracking is enabled.</p>
            <?php else: ?>
            <table class="wp-list-table widefat fixed striped">
                <thead>
                    <tr>
                        <th>Source</th>
                        <th>Sessions</th>
                        <th>Orders</th>
                        <th>Revenue</th>
                        <th>Conv. Rate</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($revenueBySources as $source): 
                        $sourceConv = $source->sessions > 0 ? round(($source->orders / $source->sessions) * 100, 2) : 0;
                    ?>
                    <tr>
                        <td><strong><?php echo esc_html($source->source); ?></strong></td>
                        <td><?php echo number_format($source->sessions); ?></td>
                        <td><?php echo number_format($source->orders); ?></td>
                        <td><?php echo \Cartt\Services\CurrencyService::formatPrice($source->revenue ?? 0); ?></td>
                        <td><?php echo $sourceConv; ?>%</td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
            <?php endif; ?>
        </div>
        
        <?php elseif ($subtab === 'funnel'): ?>
        <!-- Detailed Funnel Analysis -->
        <div class="cartt-card">
            <h2>Conversion Funnel Analysis - <?php echo $periodLabel; ?></h2>
            
            <div style="display: flex; justify-content: space-between; margin: 30px 0;">
                <?php 
                $funnelSteps = [
                    'visitors' => ['label' => 'Visitors', 'icon' => 'visibility'],
                    'product_views' => ['label' => 'Product Views', 'icon' => 'products'],
                    'add_to_cart' => ['label' => 'Add to Cart', 'icon' => 'cart'],
                    'checkout' => ['label' => 'Checkout', 'icon' => 'clipboard'],
                    'purchase' => ['label' => 'Purchase', 'icon' => 'yes-alt']
                ];
                
                $stepKeys = array_keys($funnelSteps);
                foreach ($funnelSteps as $key => $step):
                    $value = $funnelData[$key];
                    $index = array_search($key, $stepKeys);
                    $prevValue = $index > 0 ? $funnelData[$stepKeys[$index - 1]] : $value;
                    $convRate = $prevValue > 0 ? round(($value / $prevValue) * 100, 1) : 0;
                ?>
                <div style="text-align: center; flex: 1;">
                    <div style="font-size: 36px; font-weight: 700; color: #333;"><?php echo number_format($value); ?></div>
                    <div style="font-size: 14px; color: #666; margin: 5px 0;"><?php echo $step['label']; ?></div>
                    <?php if ($key !== 'visitors'): ?>
                    <div style="font-size: 12px; color: <?php echo $convRate >= 50 ? '#4CAF50' : ($convRate >= 25 ? '#f0b849' : '#f44336'); ?>;">
                        <?php echo $convRate; ?>% from previous
                    </div>
                    <?php endif; ?>
                </div>
                <?php if ($key !== 'purchase'): ?>
                <div style="display: flex; align-items: center; color: #ccc; font-size: 24px;">→</div>
                <?php endif; ?>
                <?php endforeach; ?>
            </div>
            
            <div style="background: #f5f5f5; padding: 20px; border-radius: 4px; margin-top: 20px;">
                <h4 style="margin-top: 0;">Key Metrics</h4>
                <div style="display: grid; grid-template-columns: repeat(3, 1fr); gap: 20px;">
                    <div>
                        <div style="font-size: 24px; font-weight: 600; margin-bottom: 6px;"><?php echo $conversionRate; ?>%</div>
                        <div style="color: #666; font-size: 13px;">Overall Conversion Rate</div>
                    </div>
                    <div>
                        <div style="font-size: 24px; font-weight: 600; margin-bottom: 6px;">
                            <?php 
                            $cartToCheckout = $funnelData['add_to_cart'] > 0 
                                ? round(($funnelData['checkout'] / $funnelData['add_to_cart']) * 100, 1) 
                                : 0;
                            echo $cartToCheckout;
                            ?>%
                        </div>
                        <div style="color: #666; font-size: 13px;">Cart to Checkout Rate</div>
                    </div>
                    <div>
                        <div style="font-size: 24px; font-weight: 600; margin-bottom: 6px;">
                            <?php 
                            $checkoutCompletion = $funnelData['checkout'] > 0 
                                ? round(($funnelData['purchase'] / $funnelData['checkout']) * 100, 1) 
                                : 0;
                            echo $checkoutCompletion;
                            ?>%
                        </div>
                        <div style="color: #666; font-size: 13px;">Checkout Completion Rate</div>
                    </div>
                </div>
            </div>
        </div>
        
        <?php elseif ($subtab === 'products'): ?>
        <!-- Product Performance -->
        <div class="cartt-card">
            <h2>Product Performance - <?php echo $periodLabel; ?></h2>
            
            <?php
            $productPerformance = $wpdb->get_results(
                "SELECT 
                    p.id, p.name, p.price,
                    COALESCE(v.views, 0) as views,
                    COALESCE(s.sales, 0) as sales,
                    COALESCE(s.revenue, 0) as revenue,
                    CASE WHEN v.views > 0 THEN ROUND((s.sales / v.views) * 100, 2) ELSE 0 END as conversion_rate
                FROM {$wpdb->prefix}cartt_products p
                LEFT JOIN (
                    SELECT product_id, COUNT(*) as views 
                    FROM {$wpdb->prefix}cartt_product_views 
                    WHERE 1=1 {$dateFilter}
                    GROUP BY product_id
                ) v ON p.id = v.product_id
                LEFT JOIN (
                    SELECT product_id, SUM(quantity) as sales, SUM(total) as revenue
                    FROM {$wpdb->prefix}cartt_order_items oi
                    JOIN {$wpdb->prefix}cartt_orders o ON oi.order_id = o.id
                    WHERE o.status IN ('completed', 'processing') {$dateFilter}
                    GROUP BY product_id
                ) s ON p.id = s.product_id
                WHERE p.status = 'publish'
                ORDER BY revenue DESC
                LIMIT 50"
            );
            ?>
            
            <table class="wp-list-table widefat fixed striped">
                <thead>
                    <tr>
                        <th>Product</th>
                        <th>Price</th>
                        <th>Views</th>
                        <th>Sales</th>
                        <th>Conversion</th>
                        <th>Revenue</th>
                        <th>Rev/View</th>
                    </tr>
                </thead>
                <tbody>
                    <?php if (empty($productPerformance)): ?>
                    <tr>
                        <td colspan="7" style="text-align: center; padding: 40px;">No product data available.</td>
                    </tr>
                    <?php else: ?>
                    <?php foreach ($productPerformance as $product): 
                        $revPerView = $product->views > 0 ? $product->revenue / $product->views : 0;
                    ?>
                    <tr>
                        <td><strong><?php echo esc_html($product->name); ?></strong></td>
                        <td><?php echo \Cartt\Services\CurrencyService::formatPrice($product->price); ?></td>
                        <td><?php echo number_format($product->views); ?></td>
                        <td><?php echo number_format($product->sales); ?></td>
                        <td>
                            <span style="color: <?php echo $product->conversion_rate >= 3 ? '#4CAF50' : ($product->conversion_rate >= 1 ? '#f0b849' : '#f44336'); ?>;">
                                <?php echo $product->conversion_rate; ?>%
                            </span>
                        </td>
                        <td><strong><?php echo \Cartt\Services\CurrencyService::formatPrice($product->revenue); ?></strong></td>
                        <td><?php echo \Cartt\Services\CurrencyService::formatPrice($revPerView); ?></td>
                    </tr>
                    <?php endforeach; ?>
                    <?php endif; ?>
                </tbody>
            </table>
        </div>
        
        <?php elseif ($subtab === 'customers'): ?>
        <!-- Customer Insights / RFM -->
        <div class="cartt-card">
            <h2>Customer Segments (RFM Analysis)</h2>
            <p>Customers segmented by Recency, Frequency, and Monetary value.</p>
            
            <?php
            // Simplified RFM calculation
            $customers = $wpdb->get_results(
                "SELECT 
                    c.id, c.email, c.first_name, c.last_name,
                    c.order_count, c.total_spent,
                    MAX(o.created_at) as last_order,
                    DATEDIFF(NOW(), MAX(o.created_at)) as days_since_order
                FROM {$wpdb->prefix}cartt_customers c
                LEFT JOIN {$wpdb->prefix}cartt_orders o ON c.id = o.customer_id
                WHERE c.order_count > 0
                GROUP BY c.id
                ORDER BY c.total_spent DESC
                LIMIT 100"
            );
            
            // Segment customers
            $segments = [
                'champions' => ['label' => 'Champions', 'color' => '#4CAF50', 'customers' => []],
                'loyal' => ['label' => 'Loyal Customers', 'color' => '#2196F3', 'customers' => []],
                'promising' => ['label' => 'Promising', 'color' => '#9C27B0', 'customers' => []],
                'new' => ['label' => 'New Customers', 'color' => '#00BCD4', 'customers' => []],
                'at_risk' => ['label' => 'At Risk', 'color' => '#f0b849', 'customers' => []],
                'lost' => ['label' => 'Lost', 'color' => '#f44336', 'customers' => []]
            ];
            
            foreach ($customers as $customer) {
                $daysSince = $customer->days_since_order ?? 999;
                $orderCount = $customer->order_count;
                $totalSpent = $customer->total_spent;
                
                if ($daysSince <= 30 && $orderCount >= 5 && $totalSpent >= 500) {
                    $segments['champions']['customers'][] = $customer;
                } elseif ($orderCount >= 3 && $totalSpent >= 200) {
                    $segments['loyal']['customers'][] = $customer;
                } elseif ($daysSince <= 60 && $orderCount >= 2) {
                    $segments['promising']['customers'][] = $customer;
                } elseif ($orderCount === 1 && $daysSince <= 30) {
                    $segments['new']['customers'][] = $customer;
                } elseif ($daysSince > 60 && $daysSince <= 120) {
                    $segments['at_risk']['customers'][] = $customer;
                } else {
                    $segments['lost']['customers'][] = $customer;
                }
            }
            ?>
            
            <div style="display: grid; grid-template-columns: repeat(3, 1fr); gap: 20px; margin-bottom: 30px;">
                <?php foreach ($segments as $key => $segment): ?>
                <div style="background: <?php echo $segment['color']; ?>15; border-left: 4px solid <?php echo $segment['color']; ?>; padding: 15px; border-radius: 4px;">
                    <div style="font-size: 24px; font-weight: 600; color: <?php echo $segment['color']; ?>;">
                        <?php echo count($segment['customers']); ?>
                    </div>
                    <div style="color: #666;"><?php echo $segment['label']; ?></div>
                </div>
                <?php endforeach; ?>
            </div>
            
            <!-- Customer Lifetime Value Distribution -->
            <h3>Top Customers by Lifetime Value</h3>
            <table class="wp-list-table widefat fixed striped" style="margin-top: 15px;">
                <thead>
                    <tr>
                        <th>Customer</th>
                        <th>Orders</th>
                        <th>Total Spent</th>
                        <th>Avg Order Value</th>
                        <th>Last Order</th>
                        <th>Segment</th>
                    </tr>
                </thead>
                <tbody>
                    <?php 
                    $topCustomers = array_slice($customers, 0, 20);
                    foreach ($topCustomers as $customer): 
                        $avgOrder = $customer->order_count > 0 ? $customer->total_spent / $customer->order_count : 0;
                        
                        // Determine segment
                        $customerSegment = 'lost';
                        $daysSince = $customer->days_since_order ?? 999;
                        if ($daysSince <= 30 && $customer->order_count >= 5) $customerSegment = 'champions';
                        elseif ($customer->order_count >= 3) $customerSegment = 'loyal';
                        elseif ($daysSince <= 60 && $customer->order_count >= 2) $customerSegment = 'promising';
                        elseif ($customer->order_count === 1 && $daysSince <= 30) $customerSegment = 'new';
                        elseif ($daysSince > 60 && $daysSince <= 120) $customerSegment = 'at_risk';
                    ?>
                    <tr>
                        <td>
                            <strong><?php echo esc_html($customer->first_name . ' ' . $customer->last_name); ?></strong>
                            <br><small><?php echo esc_html($customer->email); ?></small>
                        </td>
                        <td><?php echo intval($customer->order_count); ?></td>
                        <td><strong><?php echo \Cartt\Services\CurrencyService::formatPrice($customer->total_spent); ?></strong></td>
                        <td><?php echo \Cartt\Services\CurrencyService::formatPrice($avgOrder); ?></td>
                        <td><?php echo $customer->last_order ? date('M j, Y', strtotime($customer->last_order)) : 'N/A'; ?></td>
                        <td>
                            <span style="background: <?php echo $segments[$customerSegment]['color']; ?>; color: #fff; padding: 2px 8px; border-radius: 3px; font-size: 11px;">
                                <?php echo $segments[$customerSegment]['label']; ?>
                            </span>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
        
        <?php elseif ($subtab === 'sources'): ?>
        <!-- Traffic Sources & Attribution -->
        <div class="cartt-card">
            <h2>Traffic Sources & Revenue Attribution - <?php echo $periodLabel; ?></h2>
            
            <?php
            $sources = $wpdb->get_results(
                "SELECT 
                    COALESCE(utm_source, 'Direct / None') as source,
                    COALESCE(utm_medium, '-') as medium,
                    COALESCE(utm_campaign, '-') as campaign,
                    COUNT(DISTINCT session_id) as sessions,
                    COUNT(DISTINCT CASE WHEN event_type = 'purchase' THEN order_id END) as conversions,
                    COALESCE(SUM(CASE WHEN event_type = 'purchase' THEN 
                        (SELECT total FROM {$wpdb->prefix}cartt_orders WHERE id = order_id)
                    END), 0) as revenue
                FROM {$wpdb->prefix}cartt_analytics_events
                WHERE 1=1 {$dateFilter}
                GROUP BY utm_source, utm_medium, utm_campaign
                ORDER BY revenue DESC"
            );
            ?>
            
            <table class="wp-list-table widefat fixed striped">
                <thead>
                    <tr>
                        <th>Source</th>
                        <th>Medium</th>
                        <th>Campaign</th>
                        <th>Sessions</th>
                        <th>Conversions</th>
                        <th>Conv. Rate</th>
                        <th>Revenue</th>
                    </tr>
                </thead>
                <tbody>
                    <?php if (empty($sources)): ?>
                    <tr>
                        <td colspan="7" style="text-align: center; padding: 40px;">
                            No attribution data available. Make sure UTM parameters are being tracked.
                        </td>
                    </tr>
                    <?php else: ?>
                    <?php foreach ($sources as $source): 
                        $convRate = $source->sessions > 0 ? round(($source->conversions / $source->sessions) * 100, 2) : 0;
                    ?>
                    <tr>
                        <td><strong><?php echo esc_html($source->source); ?></strong></td>
                        <td><?php echo esc_html($source->medium); ?></td>
                        <td><?php echo esc_html($source->campaign); ?></td>
                        <td><?php echo number_format($source->sessions); ?></td>
                        <td><?php echo number_format($source->conversions); ?></td>
                        <td><?php echo $convRate; ?>%</td>
                        <td><strong><?php echo \Cartt\Services\CurrencyService::formatPrice($source->revenue); ?></strong></td>
                    </tr>
                    <?php endforeach; ?>
                    <?php endif; ?>
                </tbody>
            </table>
        </div>
        <?php endif; ?>
    </div>
</div>
