<?php
/**
 * AI Recommendations Admin View
 * 
 * @package Cartt
 * @since 1.4.0
 */

defined('ABSPATH') || exit;

$settings = get_option('cartt_ai_recommendations', [
    'enabled' => true,
    'algorithms' => [
        'frequently_bought' => true,
        'similar_products' => true,
        'personalized' => true,
        'trending' => true,
        'bestsellers' => true,
        'recently_viewed' => true
    ],
    'display' => [
        'product_page' => true,
        'cart_page' => true,
        'checkout_page' => false,
        'order_confirmation' => true
    ],
    'limits' => [
        'frequently_bought' => 4,
        'similar_products' => 4,
        'personalized' => 6,
        'trending' => 8,
        'bestsellers' => 8,
        'recently_viewed' => 6
    ],
    'ai_provider' => 'none',
    'openai_api_key' => '',
    'trending_period' => 7,
    'bestseller_period' => 30
]);

// Get stats (with fallbacks for missing tables)
global $wpdb;
$patterns_table = $wpdb->prefix . 'cartt_purchase_patterns';
$views_table = $wpdb->prefix . 'cartt_product_views';

$total_patterns = 0;
$total_views = 0;
$unique_products_tracked = 0;

// Suppress errors for tables that might not exist yet
$wpdb->suppress_errors(true);
$total_patterns = (int) $wpdb->get_var("SELECT COUNT(*) FROM {$patterns_table}");
$total_views = (int) $wpdb->get_var("SELECT COUNT(*) FROM {$views_table}");
$unique_products_tracked = (int) $wpdb->get_var("SELECT COUNT(DISTINCT product_id) FROM {$views_table}");
$wpdb->suppress_errors(false);

$isEmbedded = isset($GLOBALS['cartt_embedded']) && $GLOBALS['cartt_embedded'];
$current_tab = isset($_GET['subtab']) ? sanitize_key($_GET['subtab']) : 'settings';
?>

<?php if (!$isEmbedded): ?>
<div class="wrap cartt-admin">
    <h1 class="wp-heading-inline">AI Recommendations</h1>
    <hr class="wp-header-end">
<?php endif; ?>

    <div class="cartt-stats-grid" style="display: grid; grid-template-columns: repeat(3, 1fr); gap: 15px; margin-bottom: 20px;">
        <div class="cartt-stat-card" style="background: #fff; padding: 20px; border-radius: 8px; box-shadow: 0 1px 3px rgba(0,0,0,0.1);">
            <div style="color: #666; font-size: 13px; margin-bottom: 5px;">Purchase Patterns</div>
            <div style="font-size: 28px; font-weight: 600; color: #1d2327;"><?php echo number_format($total_patterns); ?></div>
            <div style="color: #666; font-size: 12px; margin-top: 5px;">Product correlations learned</div>
        </div>
        <div class="cartt-stat-card" style="background: #fff; padding: 20px; border-radius: 8px; box-shadow: 0 1px 3px rgba(0,0,0,0.1);">
            <div style="color: #666; font-size: 13px; margin-bottom: 5px;">Product Views Tracked</div>
            <div style="font-size: 28px; font-weight: 600; color: #50575e;"><?php echo number_format($total_views); ?></div>
            <div style="color: #666; font-size: 12px; margin-top: 5px;">For personalization</div>
        </div>
        <div class="cartt-stat-card" style="background: #fff; padding: 20px; border-radius: 8px; box-shadow: 0 1px 3px rgba(0,0,0,0.1);">
            <div style="color: #666; font-size: 13px; margin-bottom: 5px;">Products Tracked</div>
            <div style="font-size: 28px; font-weight: 600; color: #00a32a;"><?php echo number_format($unique_products_tracked); ?></div>
            <div style="color: #666; font-size: 12px; margin-top: 5px;">With view history</div>
        </div>
    </div>

    <?php 
    $baseUrl = $isEmbedded ? '?page=cartt-ai-conversion&tab=ai-recommendations&subtab=' : '?page=cartt-recommendations&subtab=';
    ?>
    <nav class="nav-tab-wrapper">
        <a href="<?php echo $baseUrl; ?>settings" class="nav-tab <?php echo $current_tab === 'settings' ? 'nav-tab-active' : ''; ?>">Settings</a>
        <a href="<?php echo $baseUrl; ?>algorithms" class="nav-tab <?php echo $current_tab === 'algorithms' ? 'nav-tab-active' : ''; ?>">Algorithms</a>
        <a href="<?php echo $baseUrl; ?>display" class="nav-tab <?php echo $current_tab === 'display' ? 'nav-tab-active' : ''; ?>">Display Options</a>
        <a href="<?php echo $baseUrl; ?>preview" class="nav-tab <?php echo $current_tab === 'preview' ? 'nav-tab-active' : ''; ?>">Preview</a>
        <a href="<?php echo $baseUrl; ?>performance" class="nav-tab <?php echo $current_tab === 'performance' ? 'nav-tab-active' : ''; ?>">Performance</a>
    </nav>

    <div style="background: #fff; padding: 20px; border: 1px solid #c3c4c7; border-top: none;">
        <?php if ($current_tab === 'settings'): ?>
            <form method="post" action="" id="ai-settings-form" style="max-width: 700px;">
                <?php wp_nonce_field('cartt_ai_recommendations', 'cartt_nonce'); ?>
                
                <h3>General Settings</h3>
                
                <table class="form-table">
                    <tr>
                        <th scope="row">Enable Recommendations</th>
                        <td>
                            <label>
                                <input type="checkbox" name="enabled" value="1" <?php checked($settings['enabled'] ?? true); ?>>
                                Show product recommendations throughout the store
                            </label>
                        </td>
                    </tr>
                </table>

                <h3>AI Enhancement (Optional)</h3>
                <p style="color: #666; margin-bottom: 20px;">
                    Connect an AI provider for enhanced personalized recommendations. 
                    Without AI, recommendations use collaborative filtering and content-based algorithms.
                </p>
                
                <table class="form-table">
                    <tr>
                        <th scope="row">AI Provider</th>
                        <td>
                            <select name="ai_provider" id="ai-provider">
                                <option value="none" <?php selected($settings['ai_provider'] ?? '', 'none'); ?>>None (Built-in algorithms only)</option>
                                <option value="openai" <?php selected($settings['ai_provider'] ?? '', 'openai'); ?>>OpenAI GPT-4</option>
                            </select>
                        </td>
                    </tr>
                    <tr class="openai-settings" style="<?php echo ($settings['ai_provider'] ?? '') !== 'openai' ? 'display:none;' : ''; ?>">
                        <th scope="row">OpenAI API Key</th>
                        <td>
                            <input type="password" name="openai_api_key" value="<?php echo esc_attr($settings['openai_api_key'] ?? ''); ?>" class="regular-text">
                            <p class="description">Get your API key from <a href="https://platform.openai.com/api-keys" target="_blank">OpenAI Dashboard</a></p>
                        </td>
                    </tr>
                    <tr class="openai-settings" style="<?php echo ($settings['ai_provider'] ?? '') !== 'openai' ? 'display:none;' : ''; ?>">
                        <th scope="row">Test Connection</th>
                        <td>
                            <button type="button" class="button" id="test-openai">Test OpenAI Connection</button>
                            <span id="openai-test-result" style="margin-left: 10px;"></span>
                        </td>
                    </tr>
                </table>

                <h3>Data Collection</h3>
                
                <table class="form-table">
                    <tr>
                        <th scope="row">Trending Period</th>
                        <td>
                            <select name="trending_period">
                                <option value="1" <?php selected($settings['trending_period'] ?? 7, 1); ?>>Last 24 hours</option>
                                <option value="7" <?php selected($settings['trending_period'] ?? 7, 7); ?>>Last 7 days</option>
                                <option value="14" <?php selected($settings['trending_period'] ?? 7, 14); ?>>Last 14 days</option>
                                <option value="30" <?php selected($settings['trending_period'] ?? 7, 30); ?>>Last 30 days</option>
                            </select>
                            <p class="description">Time period for calculating trending products</p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">Bestseller Period</th>
                        <td>
                            <select name="bestseller_period">
                                <option value="7" <?php selected($settings['bestseller_period'] ?? 30, 7); ?>>Last 7 days</option>
                                <option value="30" <?php selected($settings['bestseller_period'] ?? 30, 30); ?>>Last 30 days</option>
                                <option value="90" <?php selected($settings['bestseller_period'] ?? 30, 90); ?>>Last 90 days</option>
                                <option value="365" <?php selected($settings['bestseller_period'] ?? 30, 365); ?>>Last year</option>
                                <option value="0" <?php selected($settings['bestseller_period'] ?? 30, 0); ?>>All time</option>
                            </select>
                            <p class="description">Time period for calculating bestsellers</p>
                        </td>
                    </tr>
                </table>

                <h3>Data Management</h3>
                
                <table class="form-table">
                    <tr>
                        <th scope="row">Clear Data</th>
                        <td>
                            <button type="button" class="button" id="clear-patterns">Clear Purchase Patterns</button>
                            <button type="button" class="button" id="clear-views">Clear View History</button>
                            <p class="description" style="margin-top: 10px;">Clearing data will reset recommendation accuracy. Patterns rebuild automatically from new orders.</p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">Rebuild Patterns</th>
                        <td>
                            <button type="button" class="button button-primary" id="rebuild-patterns">Rebuild from Order History</button>
                            <p class="description">Regenerate purchase patterns from all historical orders. This may take a few minutes.</p>
                        </td>
                    </tr>
                </table>

                <p class="submit">
                    <button type="submit" class="button button-primary">Save Settings</button>
                </p>
            </form>

        <?php elseif ($current_tab === 'algorithms'): ?>
            <form method="post" action="" id="algorithms-form" style="max-width: 800px;">
                <?php wp_nonce_field('cartt_ai_recommendations', 'cartt_nonce'); ?>
                
                <p style="color: #666; margin-bottom: 25px;">
                    Enable or disable specific recommendation algorithms and set how many products to show for each type.
                </p>

                <table class="wp-list-table widefat fixed">
                    <thead>
                        <tr>
                            <th style="width: 50px;">Enable</th>
                            <th style="width: 200px;">Algorithm</th>
                            <th>Description</th>
                            <th style="width: 100px;">Products</th>
                        </tr>
                    </thead>
                    <tbody>
                        <tr>
                            <td><input type="checkbox" name="algorithms[frequently_bought]" value="1" <?php checked($settings['algorithms']['frequently_bought'] ?? true); ?>></td>
                            <td><strong>Frequently Bought Together</strong></td>
                            <td>Products commonly purchased together based on order history. Best for cross-selling.</td>
                            <td><input type="number" name="limits[frequently_bought]" value="<?php echo esc_attr($settings['limits']['frequently_bought'] ?? 4); ?>" min="1" max="12" class="small-text"></td>
                        </tr>
                        <tr>
                            <td><input type="checkbox" name="algorithms[similar_products]" value="1" <?php checked($settings['algorithms']['similar_products'] ?? true); ?>></td>
                            <td><strong>Similar Products</strong></td>
                            <td>Products in the same category with similar price range. Good fallback when purchase data is limited.</td>
                            <td><input type="number" name="limits[similar_products]" value="<?php echo esc_attr($settings['limits']['similar_products'] ?? 4); ?>" min="1" max="12" class="small-text"></td>
                        </tr>
                        <tr>
                            <td><input type="checkbox" name="algorithms[personalized]" value="1" <?php checked($settings['algorithms']['personalized'] ?? true); ?>></td>
                            <td><strong>Personalized</strong></td>
                            <td>Based on customer's purchase and browsing history. Requires user to be logged in or have session history.</td>
                            <td><input type="number" name="limits[personalized]" value="<?php echo esc_attr($settings['limits']['personalized'] ?? 6); ?>" min="1" max="12" class="small-text"></td>
                        </tr>
                        <tr>
                            <td><input type="checkbox" name="algorithms[trending]" value="1" <?php checked($settings['algorithms']['trending'] ?? true); ?>></td>
                            <td><strong>Trending Now</strong></td>
                            <td>Most viewed products in the configured time period. Shows what's popular right now.</td>
                            <td><input type="number" name="limits[trending]" value="<?php echo esc_attr($settings['limits']['trending'] ?? 8); ?>" min="1" max="12" class="small-text"></td>
                        </tr>
                        <tr>
                            <td><input type="checkbox" name="algorithms[bestsellers]" value="1" <?php checked($settings['algorithms']['bestsellers'] ?? true); ?>></td>
                            <td><strong>Bestsellers</strong></td>
                            <td>Top selling products by order volume. Proven performers.</td>
                            <td><input type="number" name="limits[bestsellers]" value="<?php echo esc_attr($settings['limits']['bestsellers'] ?? 8); ?>" min="1" max="12" class="small-text"></td>
                        </tr>
                        <tr>
                            <td><input type="checkbox" name="algorithms[recently_viewed]" value="1" <?php checked($settings['algorithms']['recently_viewed'] ?? true); ?>></td>
                            <td><strong>Recently Viewed</strong></td>
                            <td>Products the customer has recently looked at. Helps with comparison shopping.</td>
                            <td><input type="number" name="limits[recently_viewed]" value="<?php echo esc_attr($settings['limits']['recently_viewed'] ?? 6); ?>" min="1" max="12" class="small-text"></td>
                        </tr>
                    </tbody>
                </table>

                <p class="submit" style="margin-top: 20px;">
                    <button type="submit" class="button button-primary">Save Algorithm Settings</button>
                </p>
            </form>

        <?php elseif ($current_tab === 'display'): ?>
            <form method="post" action="" id="display-form" style="max-width: 700px;">
                <?php wp_nonce_field('cartt_ai_recommendations', 'cartt_nonce'); ?>
                
                <h3>Automatic Placement</h3>
                <p style="color: #666; margin-bottom: 20px;">Choose where recommendations automatically appear.</p>
                
                <table class="form-table">
                    <tr>
                        <th scope="row">Product Page</th>
                        <td>
                            <label style="display: block; margin-bottom: 8px;">
                                <input type="checkbox" name="display[product_page]" value="1" <?php checked($settings['display']['product_page'] ?? true); ?>>
                                Show "Frequently Bought Together" and "Similar Products"
                            </label>
                            <select name="product_page_position">
                                <option value="below_description" <?php selected($settings['product_page_position'] ?? '', 'below_description'); ?>>Below product description</option>
                                <option value="below_add_to_cart" <?php selected($settings['product_page_position'] ?? '', 'below_add_to_cart'); ?>>Below Add to Cart button</option>
                                <option value="sidebar" <?php selected($settings['product_page_position'] ?? '', 'sidebar'); ?>>In sidebar</option>
                            </select>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">Cart Page</th>
                        <td>
                            <label>
                                <input type="checkbox" name="display[cart_page]" value="1" <?php checked($settings['display']['cart_page'] ?? true); ?>>
                                Show "You May Also Like" based on cart contents
                            </label>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">Checkout Page</th>
                        <td>
                            <label>
                                <input type="checkbox" name="display[checkout_page]" value="1" <?php checked($settings['display']['checkout_page'] ?? false); ?>>
                                Show last-minute recommendations (use sparingly to avoid distraction)
                            </label>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">Order Confirmation</th>
                        <td>
                            <label>
                                <input type="checkbox" name="display[order_confirmation]" value="1" <?php checked($settings['display']['order_confirmation'] ?? true); ?>>
                                Show personalized recommendations for future purchases
                            </label>
                        </td>
                    </tr>
                </table>

                <h3>Widget Titles</h3>
                
                <table class="form-table">
                    <tr>
                        <th scope="row">Frequently Bought</th>
                        <td><input type="text" name="titles[frequently_bought]" value="<?php echo esc_attr($settings['titles']['frequently_bought'] ?? 'Frequently Bought Together'); ?>" class="regular-text"></td>
                    </tr>
                    <tr>
                        <th scope="row">Similar Products</th>
                        <td><input type="text" name="titles[similar_products]" value="<?php echo esc_attr($settings['titles']['similar_products'] ?? 'Similar Products'); ?>" class="regular-text"></td>
                    </tr>
                    <tr>
                        <th scope="row">Personalized</th>
                        <td><input type="text" name="titles[personalized]" value="<?php echo esc_attr($settings['titles']['personalized'] ?? 'Recommended For You'); ?>" class="regular-text"></td>
                    </tr>
                    <tr>
                        <th scope="row">Trending</th>
                        <td><input type="text" name="titles[trending]" value="<?php echo esc_attr($settings['titles']['trending'] ?? 'Trending Now'); ?>" class="regular-text"></td>
                    </tr>
                    <tr>
                        <th scope="row">Bestsellers</th>
                        <td><input type="text" name="titles[bestsellers]" value="<?php echo esc_attr($settings['titles']['bestsellers'] ?? 'Bestsellers'); ?>" class="regular-text"></td>
                    </tr>
                    <tr>
                        <th scope="row">Recently Viewed</th>
                        <td><input type="text" name="titles[recently_viewed]" value="<?php echo esc_attr($settings['titles']['recently_viewed'] ?? 'Recently Viewed'); ?>" class="regular-text"></td>
                    </tr>
                </table>

                <h3>Shortcodes</h3>
                <p style="color: #666; margin-bottom: 15px;">Use these shortcodes to place recommendations anywhere:</p>
                
                <div style="background: #f5f5f5; padding: 15px; border-radius: 5px; font-family: monospace; font-size: 13px;">
                    <p style="margin: 0 0 10px;"><code>[cartt_recommendations type="frequently_bought" product_id="123"]</code></p>
                    <p style="margin: 0 0 10px;"><code>[cartt_recommendations type="similar" product_id="123"]</code></p>
                    <p style="margin: 0 0 10px;"><code>[cartt_recommendations type="personalized" limit="6"]</code></p>
                    <p style="margin: 0 0 10px;"><code>[cartt_recommendations type="trending" limit="8"]</code></p>
                    <p style="margin: 0 0 10px;"><code>[cartt_recommendations type="bestsellers" limit="8"]</code></p>
                    <p style="margin: 0;"><code>[cartt_recommendations type="recently_viewed" limit="6"]</code></p>
                </div>

                <p class="submit" style="margin-top: 20px;">
                    <button type="submit" class="button button-primary">Save Display Settings</button>
                </p>
            </form>

        <?php elseif ($current_tab === 'preview'): ?>
            <div style="max-width: 900px;">
                <h3>Preview Recommendations</h3>
                <p style="color: #666; margin-bottom: 20px;">Select a product to preview what recommendations would be shown.</p>
                
                <div style="margin-bottom: 30px;">
                    <label for="preview-product" style="display: block; margin-bottom: 8px; font-weight: 600;">Select Product:</label>
                    <select id="preview-product" style="width: 300px;">
                        <option value="">Choose a product...</option>
                        <?php
                        $products = $wpdb->get_results("SELECT id, name FROM {$wpdb->prefix}cartt_products WHERE status = 'published' ORDER BY name LIMIT 100");
                        foreach ($products as $product):
                        ?>
                            <option value="<?php echo esc_attr($product->id); ?>"><?php echo esc_html($product->name); ?></option>
                        <?php endforeach; ?>
                    </select>
                    <button type="button" class="button" id="load-preview" style="margin-left: 10px;">Load Preview</button>
                </div>

                <div id="preview-results" style="display: none;">
                    <div id="preview-frequently-bought" class="preview-section" style="margin-bottom: 30px;">
                        <h4 style="border-bottom: 1px solid #ddd; padding-bottom: 10px;">Frequently Bought Together</h4>
                        <div class="preview-products" style="display: grid; grid-template-columns: repeat(4, 1fr); gap: 15px;"></div>
                    </div>

                    <div id="preview-similar" class="preview-section" style="margin-bottom: 30px;">
                        <h4 style="border-bottom: 1px solid #ddd; padding-bottom: 10px;">Similar Products</h4>
                        <div class="preview-products" style="display: grid; grid-template-columns: repeat(4, 1fr); gap: 15px;"></div>
                    </div>

                    <div id="preview-trending" class="preview-section" style="margin-bottom: 30px;">
                        <h4 style="border-bottom: 1px solid #ddd; padding-bottom: 10px;">Trending Now</h4>
                        <div class="preview-products" style="display: grid; grid-template-columns: repeat(4, 1fr); gap: 15px;"></div>
                    </div>
                </div>

                <div id="preview-loading" style="display: none; text-align: center; padding: 40px;">
                    <span class="spinner is-active" style="float: none;"></span>
                    <p>Loading recommendations...</p>
                </div>

                <div id="preview-empty" style="display: none; text-align: center; padding: 40px; color: #666;">
                    <p>Select a product above to preview recommendations.</p>
                </div>
            </div>

        <?php elseif ($current_tab === 'performance'): ?>
            <div style="max-width: 900px;">
                <h3>Recommendation Performance</h3>
                <p style="color: #666; margin-bottom: 20px;">Track how well your recommendations are converting.</p>
                
                <?php
                // Get real recommendation tracking data from database
                global $wpdb;
                $table_exists = $wpdb->get_var("SHOW TABLES LIKE '{$wpdb->prefix}cartt_recommendation_tracking'") !== null;
                $has_data = false;
                $stats = [];
                
                if ($table_exists) {
                    $stats_result = $wpdb->get_results(
                        "SELECT 
                            algorithm_type,
                            SUM(impressions) as impressions,
                            SUM(clicks) as clicks,
                            SUM(conversions) as conversions
                        FROM {$wpdb->prefix}cartt_recommendation_tracking
                        GROUP BY algorithm_type"
                    );
                    if (!empty($stats_result)) {
                        $has_data = true;
                        foreach ($stats_result as $row) {
                            $stats[$row->algorithm_type] = [
                                'impressions' => (int)$row->impressions,
                                'clicks' => (int)$row->clicks,
                                'conversions' => (int)$row->conversions
                            ];
                        }
                    }
                }
                
                $labels = [
                    'frequently_bought' => 'Frequently Bought Together',
                    'similar_products' => 'Similar Products',
                    'personalized' => 'Personalized',
                    'trending' => 'Trending Now',
                    'bestsellers' => 'Bestsellers',
                    'recently_viewed' => 'Recently Viewed'
                ];
                ?>
                
                <?php if (!$has_data): ?>
                <div style="text-align: center; padding: 60px 20px; background: #f6f7f7; border-radius: 8px;">
                    <span class="dashicons dashicons-chart-bar" style="font-size: 48px; color: #ccc; margin-bottom: 15px;"></span>
                    <h4 style="margin: 0 0 10px;">No Performance Data Yet</h4>
                    <p style="color: #666; margin: 0;">Once recommendations are displayed and customers interact with them, performance metrics will appear here.</p>
                </div>
                <?php else: ?>
                <table class="wp-list-table widefat fixed striped">
                    <thead>
                        <tr>
                            <th>Algorithm</th>
                            <th style="width: 120px;">Impressions</th>
                            <th style="width: 100px;">Clicks</th>
                            <th style="width: 100px;">CTR</th>
                            <th style="width: 100px;">Conversions</th>
                            <th style="width: 100px;">Conv. Rate</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php 
                        foreach ($labels as $type => $label): 
                            $data = $stats[$type] ?? ['impressions' => 0, 'clicks' => 0, 'conversions' => 0];
                            $ctr = $data['impressions'] > 0 ? ($data['clicks'] / $data['impressions']) * 100 : 0;
                            $conv_rate = $data['clicks'] > 0 ? ($data['conversions'] / $data['clicks']) * 100 : 0;
                        ?>
                            <tr>
                                <td><strong><?php echo esc_html($label); ?></strong></td>
                                <td><?php echo number_format($data['impressions']); ?></td>
                                <td><?php echo number_format($data['clicks']); ?></td>
                                <td><?php echo number_format($ctr, 2); ?>%</td>
                                <td><?php echo number_format($data['conversions']); ?></td>
                                <td style="color: <?php echo $conv_rate >= 15 ? '#00a32a' : ($conv_rate >= 10 ? '#dba617' : '#666'); ?>; font-weight: 600;">
                                    <?php echo number_format($conv_rate, 2); ?>%
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>

                <div style="margin-top: 30px; padding: 20px; background: #f6f7f7; border-radius: 5px;">
                    <h4 style="margin-top: 0;">Tips</h4>
                    <ul style="margin-bottom: 0;">
                        <li>Higher CTR indicates customers find your recommendations relevant.</li>
                        <li>Consider A/B testing different placements and titles to improve performance.</li>
                        <li>Personalized recommendations typically convert better with more customer data.</li>
                    </ul>
                </div>
                <?php endif; ?>
            </div>
        <?php endif; ?>
    </div>
<?php if (!$isEmbedded): ?>
</div>
<?php endif; ?>

<script>
jQuery(document).ready(function($) {
    // Toggle OpenAI settings
    $('#ai-provider').on('change', function() {
        if ($(this).val() === 'openai') {
            $('.openai-settings').show();
        } else {
            $('.openai-settings').hide();
        }
    });

    // Test OpenAI connection
    $('#test-openai').on('click', function() {
        const $btn = $(this);
        const $result = $('#openai-test-result');
        const apiKey = $('input[name="openai_api_key"]').val();

        if (!apiKey) {
            $result.html('<span style="color: #dc3545;">Please enter an API key first</span>');
            return;
        }

        $btn.prop('disabled', true);
        $result.html('<span class="spinner is-active" style="float: none;"></span>');

        $.post(ajaxurl, {
            action: 'cartt_test_openai',
            nonce: '<?php echo wp_create_nonce('cartt_admin_nonce'); ?>',
            api_key: apiKey
        }, function(response) {
            if (response.success) {
                $result.html('<span style="color: #00a32a;">Connection successful!</span>');
            } else {
                $result.html('<span style="color: #dc3545;">' + (response.data?.message || 'Connection failed') + '</span>');
            }
            $btn.prop('disabled', false);
        });
    });

    // Clear patterns
    $('#clear-patterns').on('click', function() {
        if (!confirm('Are you sure? This will reset "Frequently Bought Together" recommendations.')) return;
        
        $.post(ajaxurl, {
            action: 'cartt_clear_patterns',
            nonce: '<?php echo wp_create_nonce('cartt_admin_nonce'); ?>'
        }, function(response) {
            alert(response.success ? 'Patterns cleared!' : 'Failed to clear patterns');
            location.reload();
        });
    });

    // Clear views
    $('#clear-views').on('click', function() {
        if (!confirm('Are you sure? This will reset personalized and trending recommendations.')) return;
        
        $.post(ajaxurl, {
            action: 'cartt_clear_views',
            nonce: '<?php echo wp_create_nonce('cartt_admin_nonce'); ?>'
        }, function(response) {
            alert(response.success ? 'View history cleared!' : 'Failed to clear views');
            location.reload();
        });
    });

    // Rebuild patterns
    $('#rebuild-patterns').on('click', function() {
        if (!confirm('This will rebuild all purchase patterns from order history. Continue?')) return;
        
        const $btn = $(this);
        $btn.prop('disabled', true).text('Rebuilding...');

        $.post(ajaxurl, {
            action: 'cartt_rebuild_patterns',
            nonce: '<?php echo wp_create_nonce('cartt_admin_nonce'); ?>'
        }, function(response) {
            alert(response.success ? 'Patterns rebuilt: ' + response.data.count + ' correlations created' : 'Failed to rebuild');
            location.reload();
        });
    });

    // Preview recommendations
    $('#load-preview').on('click', function() {
        const productId = $('#preview-product').val();
        if (!productId) {
            alert('Please select a product');
            return;
        }

        $('#preview-empty').hide();
        $('#preview-results').hide();
        $('#preview-loading').show();

        $.post(ajaxurl, {
            action: 'cartt_preview_recommendations',
            nonce: '<?php echo wp_create_nonce('cartt_admin_nonce'); ?>',
            product_id: productId
        }, function(response) {
            $('#preview-loading').hide();
            
            if (response.success) {
                renderPreview(response.data);
                $('#preview-results').show();
            } else {
                $('#preview-empty').html('<p style="color: #dc3545;">Failed to load recommendations</p>').show();
            }
        });
    });

    function renderPreview(data) {
        ['frequently_bought', 'similar', 'trending'].forEach(function(type) {
            const $section = $('#preview-' + type);
            const $products = $section.find('.preview-products');
            $products.empty();

            if (data[type] && data[type].length > 0) {
                data[type].forEach(function(product) {
                    $products.append(`
                        <div style="background: #f9f9f9; padding: 15px; border-radius: 5px; text-align: center;">
                            <div style="height: 80px; background: #ddd; margin-bottom: 10px; border-radius: 3px;"></div>
                            <strong style="font-size: 13px;">${product.name}</strong>
                            <div style="color: #00a32a; font-weight: 600; margin-top: 5px;">$${product.price}</div>
                        </div>
                    `);
                });
                $section.show();
            } else {
                $section.hide();
            }
        });
    }

    // Save forms
    $('#ai-settings-form, #algorithms-form, #display-form').on('submit', function(e) {
        e.preventDefault();
        const $form = $(this);
        const $btn = $form.find('button[type="submit"]');

        $btn.prop('disabled', true).text('Saving...');

        $.post(ajaxurl, {
            action: 'cartt_save_ai_recommendations',
            nonce: '<?php echo wp_create_nonce('cartt_admin_nonce'); ?>',
            data: $form.serialize()
        }, function(response) {
            if (response.success) {
                alert('Settings saved!');
            } else {
                alert(response.data?.message || 'Failed to save settings');
            }
            $btn.prop('disabled', false).text($btn.text().replace('Saving...', 'Save'));
            location.reload();
        });
    });
});
</script>
