# Cartt v1.4.1 Development Continuation Guide

## Current State: v1.4.1

Version 1.4.1 includes Enterprise & AI features with consolidated admin menus and add-ons system.

---

## DESIGN GUIDELINES

### Color Palette (Greyscale Only)
| Usage | Color | Hex |
|-------|-------|-----|
| Primary/Buttons | Black | #000000, #1d2327 |
| Text | Dark grey | #1d2327 |
| Secondary text | Grey | #555, #666 |
| Borders | Light grey | #e0e0e0 |
| Backgrounds | White/Light grey | #fff, #f5f5f5, #fafafa |
| Success indicators | Green (checkmarks only) | #4CAF50 |

### Prohibited
- NO gradients anywhere
- NO colored buttons (all buttons must be black/white)
- NO emojis in code or UI - use text labels instead
- NO purple, blue, or other accent colors

### Border Radius
- Standard: `8px` (use everywhere for consistency)
- Small elements: `4px` (badges, small buttons)
- Inputs: `6px`

### UI Components
```css
/* Primary Button */
.button-primary {
    background: #1d2327;
    border-color: #1d2327;
    color: #fff;
    border-radius: 6px;
}

/* Card */
.cartt-card {
    background: #fff;
    border: 1px solid #e0e0e0;
    border-radius: 8px;
}

/* Pro Badge */
.cartt-pro-badge {
    background: #1d2327;
    color: #fff;
    font-size: 9px;
    padding: 2px 6px;
    border-radius: 4px;
}
```

---

## COMPLETED WORK

### 7 New Services (All in /includes/Services/)

| Service | File | Size | Status |
|---------|------|------|--------|
| AI Recommendations | AIRecommendationService.php | 18.7KB | ✅ Complete |
| Social Proof | SocialProofService.php | 17.6KB | ✅ Complete |
| Advanced Analytics | AdvancedAnalyticsService.php | 20.6KB | ✅ Complete |
| Multi-Vendor | VendorService.php | 24.6KB | ✅ Complete |
| Vendor Commissions | VendorCommissionService.php | 20.1KB | ✅ Complete |
| B2B/Wholesale | WholesaleService.php | 27.2KB | ✅ Complete |
| Dropshipping | DropshippingService.php | 32.2KB | ✅ Complete |

### Service Details

#### AIRecommendationService
- Methods: get_frequently_bought_together(), get_similar_products(), get_personalized_recommendations(), get_trending_products(), get_bestsellers(), get_recently_viewed()
- OpenAI integration via get_ai_recommendations() - uses GPT-4o-mini
- Tracking: record_purchase_pattern(), record_product_view()
- AJAX actions: cartt_get_recommendations, cartt_get_trending, cartt_get_bestsellers, cartt_get_recently_viewed, cartt_get_ai_recommendations
- Settings: cartt_openai_api_key, cartt_enable_ai_recommendations

#### SocialProofService
- Methods: get_recent_purchases(), get_live_visitor_count(), get_stock_urgency(), get_cart_activity(), get_sales_count(), get_trust_badges()
- Tracking: record_visitor(), record_cart_activity(), increment_sales_count()
- AJAX actions: cartt_get_social_proof_data, cartt_record_visitor
- Settings: cartt_social_proof_*, cartt_trust_badges_*

#### AdvancedAnalyticsService
- Methods: get_conversion_funnel(), get_cohort_analysis(), get_customer_clv(), get_rfm_segments(), get_product_performance(), get_revenue_attribution(), get_realtime_stats(), get_dashboard_summary()
- Tracking: track_event(), start_session()
- Event types: visitor, product_view, add_to_cart, checkout_start, purchase

#### VendorService
- Methods: register_vendor(), get_vendor(), update_vendor(), get_vendor_dashboard_stats(), get_vendor_products(), get_vendor_orders()
- Admin: approve_vendor(), reject_vendor(), suspend_vendor()
- Verification: verify_vendor()
- Store: render_storefront()
- AJAX: cartt_register_vendor, cartt_update_vendor_settings

#### VendorCommissionService
- Methods: calculate_commission(), get_vendor_balance(), request_payout(), process_payout(), reject_payout(), handle_refund()
- Reports: get_vendor_earnings_report()
- Rate hierarchy: product → category → vendor → platform default
- Settings: cartt_platform_commission_rate (default 15%), cartt_payout_hold_days (default 14), cartt_minimum_payout (default 50)

#### WholesaleService
- Methods: apply_wholesale_pricing(), get_customer_tier(), get_quantity_price(), get_customer_specific_price()
- Applications: submit_application(), approve_application(), reject_application()
- Quotes: create_quote(), update_quote_status(), convert_quote_to_order()
- Tiers: Bronze (5+ orders, 5%), Silver (15+ orders, 10%), Gold (30+ orders, 15%), Platinum (50+ orders, 20%)

#### DropshippingService
- Methods: add_supplier(), import_aliexpress_product(), sync_printful_products(), auto_fulfill_order(), sync_inventory(), sync_prices()
- Supplier types: generic, aliexpress, printful, printify
- Settings: cartt_aliexpress_api_key, cartt_printful_api_key, cartt_printify_api_key

---

## PENDING WORK

### 1. Database Tables (Priority: HIGH)

Add to Core/Activator.php create_tables() method:

```php
// AI Recommendations
CREATE TABLE {$prefix}cartt_purchase_patterns (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    product_id BIGINT UNSIGNED NOT NULL,
    related_product_id BIGINT UNSIGNED NOT NULL,
    purchase_count INT DEFAULT 1,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    updated_at DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    UNIQUE KEY product_pair (product_id, related_product_id),
    INDEX idx_product (product_id),
    INDEX idx_count (purchase_count)
);

CREATE TABLE {$prefix}cartt_product_views (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    product_id BIGINT UNSIGNED NOT NULL,
    customer_id BIGINT UNSIGNED DEFAULT NULL,
    session_id VARCHAR(64) NOT NULL,
    viewed_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    INDEX idx_product (product_id),
    INDEX idx_customer (customer_id),
    INDEX idx_session (session_id),
    INDEX idx_viewed (viewed_at)
);

// Analytics
CREATE TABLE {$prefix}cartt_analytics_events (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    event_type VARCHAR(50) NOT NULL,
    session_id VARCHAR(64) NOT NULL,
    customer_id BIGINT UNSIGNED DEFAULT NULL,
    product_id BIGINT UNSIGNED DEFAULT NULL,
    order_id BIGINT UNSIGNED DEFAULT NULL,
    event_data LONGTEXT,
    page_url VARCHAR(500),
    referrer VARCHAR(500),
    utm_source VARCHAR(100),
    utm_medium VARCHAR(100),
    utm_campaign VARCHAR(100),
    user_agent VARCHAR(500),
    ip_address VARCHAR(45),
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    INDEX idx_event_type (event_type),
    INDEX idx_session (session_id),
    INDEX idx_customer (customer_id),
    INDEX idx_created (created_at)
);

// Vendors
CREATE TABLE {$prefix}cartt_vendors (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    user_id BIGINT UNSIGNED NOT NULL UNIQUE,
    store_name VARCHAR(200) NOT NULL,
    store_slug VARCHAR(200) NOT NULL UNIQUE,
    email VARCHAR(200) NOT NULL,
    phone VARCHAR(50),
    description TEXT,
    logo_url VARCHAR(500),
    banner_url VARCHAR(500),
    address VARCHAR(500),
    city VARCHAR(100),
    state VARCHAR(100),
    country VARCHAR(2),
    postal_code VARCHAR(20),
    commission_rate DECIMAL(5,2) DEFAULT NULL,
    status ENUM('pending','active','suspended','rejected') DEFAULT 'pending',
    verified TINYINT(1) DEFAULT 0,
    verified_at DATETIME DEFAULT NULL,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    updated_at DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    INDEX idx_status (status),
    INDEX idx_slug (store_slug)
);

CREATE TABLE {$prefix}cartt_vendor_settings (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    vendor_id BIGINT UNSIGNED NOT NULL,
    setting_key VARCHAR(100) NOT NULL,
    setting_value LONGTEXT,
    UNIQUE KEY vendor_setting (vendor_id, setting_key)
);

CREATE TABLE {$prefix}cartt_vendor_commissions (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    order_id BIGINT UNSIGNED NOT NULL,
    order_item_id BIGINT UNSIGNED NOT NULL,
    vendor_id BIGINT UNSIGNED NOT NULL,
    product_id BIGINT UNSIGNED NOT NULL,
    order_subtotal DECIMAL(10,2) NOT NULL,
    commission_rate DECIMAL(5,2) NOT NULL,
    platform_fee DECIMAL(10,2) NOT NULL,
    vendor_amount DECIMAL(10,2) NOT NULL,
    status ENUM('pending','available','paid','refunded') DEFAULT 'pending',
    payout_id BIGINT UNSIGNED DEFAULT NULL,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    INDEX idx_order (order_id),
    INDEX idx_vendor (vendor_id),
    INDEX idx_status (status)
);

CREATE TABLE {$prefix}cartt_vendor_payouts (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    vendor_id BIGINT UNSIGNED NOT NULL,
    amount DECIMAL(10,2) NOT NULL,
    payment_method VARCHAR(50) NOT NULL,
    payment_email VARCHAR(200),
    payment_details TEXT,
    status ENUM('pending','processing','completed','rejected') DEFAULT 'pending',
    transaction_id VARCHAR(200),
    requested_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    processed_at DATETIME DEFAULT NULL,
    processed_by BIGINT UNSIGNED DEFAULT NULL,
    rejection_reason TEXT,
    INDEX idx_vendor (vendor_id),
    INDEX idx_status (status)
);

// Wholesale
CREATE TABLE {$prefix}cartt_wholesale_customers (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    customer_id BIGINT UNSIGNED NOT NULL UNIQUE,
    company_name VARCHAR(200) NOT NULL,
    tax_id VARCHAR(100),
    tax_exempt TINYINT(1) DEFAULT 0,
    discount_percent DECIMAL(5,2) DEFAULT 0,
    min_order_amount DECIMAL(10,2) DEFAULT 0,
    payment_terms VARCHAR(50) DEFAULT 'prepaid',
    credit_limit DECIMAL(10,2) DEFAULT 0,
    tier VARCHAR(20) DEFAULT 'bronze',
    status ENUM('pending','active','suspended') DEFAULT 'pending',
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    updated_at DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    INDEX idx_customer (customer_id),
    INDEX idx_status (status)
);

CREATE TABLE {$prefix}cartt_wholesale_applications (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    user_id BIGINT UNSIGNED DEFAULT NULL,
    company_name VARCHAR(200) NOT NULL,
    contact_name VARCHAR(200) NOT NULL,
    email VARCHAR(200) NOT NULL,
    phone VARCHAR(50),
    address VARCHAR(500),
    city VARCHAR(100),
    state VARCHAR(100),
    country VARCHAR(2),
    postal_code VARCHAR(20),
    tax_id VARCHAR(100),
    business_type VARCHAR(100),
    expected_volume VARCHAR(100),
    website VARCHAR(500),
    notes TEXT,
    status ENUM('pending','approved','rejected') DEFAULT 'pending',
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    reviewed_at DATETIME DEFAULT NULL,
    reviewed_by BIGINT UNSIGNED DEFAULT NULL,
    rejection_reason TEXT,
    INDEX idx_status (status),
    INDEX idx_email (email)
);

CREATE TABLE {$prefix}cartt_quantity_pricing (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    product_id BIGINT UNSIGNED NOT NULL,
    min_qty INT NOT NULL,
    max_qty INT DEFAULT NULL,
    price DECIMAL(10,2) DEFAULT NULL,
    discount_percent DECIMAL(5,2) DEFAULT NULL,
    INDEX idx_product (product_id)
);

CREATE TABLE {$prefix}cartt_customer_pricing (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    customer_id BIGINT UNSIGNED NOT NULL,
    product_id BIGINT UNSIGNED NOT NULL,
    price DECIMAL(10,2) NOT NULL,
    UNIQUE KEY customer_product (customer_id, product_id)
);

CREATE TABLE {$prefix}cartt_quotes (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    quote_number VARCHAR(50) NOT NULL UNIQUE,
    customer_id BIGINT UNSIGNED DEFAULT NULL,
    customer_name VARCHAR(200) NOT NULL,
    customer_email VARCHAR(200) NOT NULL,
    customer_phone VARCHAR(50),
    company_name VARCHAR(200),
    items LONGTEXT NOT NULL,
    notes TEXT,
    subtotal DECIMAL(10,2) NOT NULL,
    discount_amount DECIMAL(10,2) DEFAULT 0,
    total DECIMAL(10,2) NOT NULL,
    status ENUM('pending','sent','accepted','rejected','expired','converted') DEFAULT 'pending',
    admin_notes TEXT,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    updated_at DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    expires_at DATETIME DEFAULT NULL,
    converted_order_id BIGINT UNSIGNED DEFAULT NULL,
    INDEX idx_status (status),
    INDEX idx_customer (customer_id)
);

// Dropshipping
CREATE TABLE {$prefix}cartt_suppliers (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    name VARCHAR(200) NOT NULL,
    type ENUM('generic','aliexpress','printful','printify') DEFAULT 'generic',
    api_key VARCHAR(500),
    api_secret VARCHAR(500),
    store_id VARCHAR(100),
    settings LONGTEXT,
    status ENUM('active','inactive') DEFAULT 'active',
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    updated_at DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    INDEX idx_type (type),
    INDEX idx_status (status)
);

CREATE TABLE {$prefix}cartt_supplier_products (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    product_id BIGINT UNSIGNED NOT NULL,
    supplier_id BIGINT UNSIGNED NOT NULL,
    supplier_sku VARCHAR(200) NOT NULL,
    supplier_url VARCHAR(1000),
    supplier_price DECIMAL(10,2) NOT NULL,
    our_price DECIMAL(10,2) NOT NULL,
    profit_margin DECIMAL(10,2) AS (our_price - supplier_price) STORED,
    auto_fulfill TINYINT(1) DEFAULT 1,
    last_synced DATETIME DEFAULT NULL,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    UNIQUE KEY product_supplier (product_id, supplier_id),
    INDEX idx_supplier (supplier_id)
);
```

### 2. Service Registration (Priority: HIGH)

In Core/Plugin.php, add to init_services() method:

```php
// v1.4.0 Services
$this->services['ai_recommendations'] = new \Jewe\Cartt\Services\AIRecommendationService();
$this->services['social_proof'] = new \Jewe\Cartt\Services\SocialProofService();
$this->services['analytics'] = new \Jewe\Cartt\Services\AdvancedAnalyticsService();
$this->services['vendor'] = new \Jewe\Cartt\Services\VendorService();
$this->services['vendor_commission'] = new \Jewe\Cartt\Services\VendorCommissionService();
$this->services['wholesale'] = new \Jewe\Cartt\Services\WholesaleService();
$this->services['dropshipping'] = new \Jewe\Cartt\Services\DropshippingService();
```

### 3. Admin Menu Items (Priority: MEDIUM)

In Admin/AdminMenu.php add:

```php
// Under Cartt main menu
add_submenu_page('cartt', 'Vendors', 'Vendors', 'manage_options', 'cartt-vendors', [$this, 'render_vendors_page']);
add_submenu_page('cartt', 'Commissions', 'Commissions', 'manage_options', 'cartt-commissions', [$this, 'render_commissions_page']);
add_submenu_page('cartt', 'Wholesale', 'Wholesale', 'manage_options', 'cartt-wholesale', [$this, 'render_wholesale_page']);
add_submenu_page('cartt', 'Analytics', 'Analytics', 'manage_options', 'cartt-analytics', [$this, 'render_analytics_page']);
add_submenu_page('cartt', 'Dropshipping', 'Dropshipping', 'manage_options', 'cartt-dropshipping', [$this, 'render_dropshipping_page']);
```

### 4. Admin Views (Priority: MEDIUM)

Create in admin/views/:
- vendors.php - Vendor list with approve/reject/suspend actions
- vendor-applications.php - Pending applications
- commissions.php - Commission overview, payout requests
- wholesale.php - Wholesale customers, applications, quotes
- analytics.php - Dashboard with charts for funnel, cohort, RFM, CLV
- dropshipping.php - Suppliers, product mappings, sync controls

### 5. Headless API/Webhooks (Priority: LOW)

Create new service: HeadlessApiService.php
- GraphQL endpoint at /cartt/graphql
- Webhook registration and delivery
- Events: order.created, order.updated, product.created, product.updated, customer.created, inventory.low

### 6. Frontend Assets (Priority: LOW)

CSS needed:
- assets/css/social-proof.css - Popup styles, animations
- assets/css/recommendations.css - Widget layouts
- assets/css/vendor-store.css - Storefront styling

JS needed:
- assets/js/social-proof.js - Popup display, AJAX data fetch
- assets/js/recommendations.js - Widget interactions

### 7. Shortcodes (Priority: LOW)

In Frontend/Shortcodes.php add:
- [cartt_vendor_store slug="store-name"]
- [cartt_vendor_dashboard]
- [cartt_vendor_registration]
- [cartt_bulk_order_form]
- [cartt_quote_request]
- [cartt_recommendations type="similar" product_id="123"]
- [cartt_wholesale_application]

---

## TESTING CHECKLIST

### AI Recommendations
- [ ] Purchase pattern recording on order complete
- [ ] Product view tracking
- [ ] Frequently bought together returns correct products
- [ ] Similar products respects category/price
- [ ] OpenAI integration (if API key set)
- [ ] AJAX endpoints work

### Social Proof
- [ ] Live visitor count increments
- [ ] Recent purchase popup shows correct data
- [ ] Stock urgency displays at threshold
- [ ] Cart activity tracking
- [ ] Trust badges render

### Analytics
- [ ] Events track correctly
- [ ] Funnel shows accurate conversion rates
- [ ] Cohort analysis returns data
- [ ] RFM segments customers correctly
- [ ] CLV calculation accurate

### Vendors
- [ ] Registration creates pending vendor
- [ ] Approval creates WordPress user role
- [ ] Vendor sees only their products/orders
- [ ] Commission calculated on order complete
- [ ] Payout request workflow works
- [ ] Storefront renders

### Wholesale
- [ ] Application submission works
- [ ] Tier pricing applies correctly
- [ ] Quantity pricing works
- [ ] Customer-specific pricing overrides
- [ ] Quote system complete
- [ ] Tax exemption applies

### Dropshipping
- [ ] Supplier CRUD works
- [ ] Product mapping saves
- [ ] AliExpress import (if API available)
- [ ] Auto-fulfill triggers
- [ ] Inventory sync works

---

## ARCHITECTURE NOTES

### Service Pattern
All services follow this pattern:
- Constructor registers hooks (add_action, add_filter)
- Public methods for business logic
- Private methods for internal operations
- AJAX handlers for frontend interactions
- Settings via get_option('cartt_setting_name')

### Database Pattern
- All tables prefixed with {$wpdb->prefix}cartt_
- BIGINT UNSIGNED for IDs
- DECIMAL(10,2) for money
- DATETIME for timestamps
- LONGTEXT for JSON data

### Hook Naming
- Actions: cartt_{service}_{action} (e.g., cartt_vendor_approved)
- Filters: cartt_{service}_{filter} (e.g., cartt_commission_rate)
- AJAX: cartt_{action} (e.g., cartt_get_recommendations)

---

## QUICK RESUME COMMANDS

```bash
# Check service files
ls -la /home/claude/cartt/includes/Services/

# View specific service
cat /home/claude/cartt/includes/Services/AIRecommendationService.php

# Check current version
grep "Version:" /home/claude/cartt/cartt.php

# View Activator for table creation
cat /home/claude/cartt/includes/Core/Activator.php

# View Plugin.php for service registration
cat /home/claude/cartt/includes/Core/Plugin.php
```

---

## VERSION INFO

- Current version in cartt.php: 1.4.1
- CARTT_VERSION constant: 1.4.1
- Status: Beta (services complete, add-ons system ready)

### Changelog v1.4.1
- Consolidated admin menus (28 → 11 items)
- Added Add-ons page with licensing system
- Fixed CurrencyService::format → formatPrice
- Fixed singleton instantiation in Plugin.php
- Greyscale-only UI design implementation
- Updated design documentation

Last updated: January 2, 2026
